package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSON;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.neo4j.Neo4j1Config;
import com.digiwin.athena.kmservice.neo4j.Neo4j2Config;
import com.digiwin.athena.knowledgegraph.data.Neo4jConstants;
import com.digiwin.athena.knowledgegraph.domain.triples.*;
import com.digiwin.athena.knowledgegraph.po.TriplesRequest;
import com.digiwin.athena.repository.neo4j.TriplesRepository;
import com.digiwin.athena.knowledgegraph.service.ITriplesService;
import com.digiwin.athena.knowledgegraph.vo.TriplesResponse;
import com.digiwin.athena.knowledgegraph.vo.TriplesVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.ogm.session.Session;
import org.neo4j.ogm.session.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
public class TriplesService implements ITriplesService {

    @Autowired(required = false)
    @Qualifier(Neo4j1Config.SESSION_FACTORY)
    SessionFactory sessionFactory1;

    @Autowired(required = false)
    @Qualifier(Neo4j2Config.SESSION_FACTORY)
    SessionFactory sessionFactory2;
    @Autowired
    TriplesRepository triplesRepository;


    @Override
    public Object postTriplesSave(TriplesRequest triplesRequest) throws DWException {
        log.info("TriplesService.sava start,request:{}", triplesRequest);
        TriplesVO scene = triplesRequest.getScene();
        if (!ObjectUtils.isEmpty(scene)) {
            List<String> list =
                    Collections.singletonList("merge (f:" + scene.getCategory() + "{code:'" + scene.getCode() +
                            "'})" + "set f.name=" + toJsonStr(scene.getName()) + ";");
            sessionExecute(sessionFactory1, list);
            sessionExecute(sessionFactory2, list);
        }
        List<List<TriplesVO>> triplesVOList = triplesRequest.getTriplesVOList();
        for (List<TriplesVO> triplesVO : triplesVOList) {
            List<String> nodeSqlList = new ArrayList<>();
            List<String> relationSqlList = new ArrayList<>();
            List<String> versionSqlList = new ArrayList<>();
            for (int i = 0; i < triplesVO.size(); i++) {
                TriplesVO vo = triplesVO.get(i);
                StringBuilder nodeSql = new StringBuilder();
                StringBuilder relationSql = new StringBuilder();
                StringBuilder versionSql = new StringBuilder();
                //upsert 节点语句 ex:merge(f:scene{code:'supplier'})set f.name="供应商选择机制";
                nodeSql.append("merge(f:")
                        .append(vo.getCategory())
                        .append("{code:'")
                        .append(vo.getCode())
                        .append("'})")
                        .append("set f.name=")
                        .append(toJsonStr(vo.getName()))
                        .append(";");
                nodeSqlList.add(nodeSql.toString());
                //upsert 关系语句 是最后一个节点则不关联下一个  ex:match(f:scene{name:'供应商选择机制',code:'supplier'}),
                // (ff:factor{name:'异常识别因子',code:'abnormal_factor'})merge (f)-[:factor]->(ff);
                if (i != (triplesVO.size() - 1)) {
                    TriplesVO voo = triplesVO.get(i + 1);
                    relationSql.append("match(f:")
                            .append(vo.getCategory())
                            .append("{name:'")
                            .append(vo.getName())
                            .append("',code:'")
                            .append(vo.getCode())
                            .append("'}),(ff:")
                            .append(voo.getCategory())
                            .append("{name:'")
                            .append(voo.getName()).append("',code:'")
                            .append(voo.getCode())
                            .append("'}) merge (f)-[:")
                            .append(voo.getCategory())
                            .append("]->(ff);");
                    relationSqlList.add(relationSql.toString());
                }
                //upsert 版本节点语句   为空则不关联版本节点 ex:match(f:factor{name:'异常识别因子',code:'abnormal_factor'})merge (f)
                // -[:version]->(p:property{version:'1.0'}) set p.value=["on_time_rate","order_num"];
                if (StringUtils.isNotEmpty(vo.getVersion())) {
                    versionSql.append("match(f:")
                            .append(vo.getCategory())
                            .append("{name:'")
                            .append(vo.getName())
                            .append("',code:'")
                            .append(vo.getCode())
                            .append("'})")
                            .append("merge (f)-[:Version]->(p:Property{version:'")
                            .append(vo.getVersion())
                            .append("'}) set p.value=")
                            .append(toJsonStr(vo.getValue()))
                            .append(";");
                    versionSqlList.add(versionSql.toString());
                }
            }
            sessionExecute(sessionFactory1, nodeSqlList, relationSqlList, versionSqlList);
            sessionExecute(sessionFactory2, nodeSqlList, relationSqlList, versionSqlList);
            //绑定父节点
            bindParentNode(scene, triplesVO);
        }
        return "success";
    }

    private String toJsonStr(Object o) {
        if (ObjectUtils.isEmpty(o)) {
            return "";
        } else {
            return JSON.toJSONString(o);
        }
    }

    @Override
    public Object postTriplesDel(TriplesRequest triplesRequest) throws DWException {
        List<String> nodeSqlList = new ArrayList<>();
        //顶级节点存在先删除顶级节点
        TriplesVO scene = triplesRequest.getScene();
        if (!ObjectUtils.isEmpty(scene)) {
            nodeSqlList.add("match(f:" + scene.getCategory() + "{code:'" + scene.getCode() + "'}) detach delete f");
        }
        //删除 版本属性节点  然后如果此节点上没有属性 则删除此节点及后续节点+关系
        List<List<TriplesVO>> triplesVOList = triplesRequest.getTriplesVOList();
        for (List<TriplesVO> triplesVOS : triplesVOList) {
            for (TriplesVO triplesVO : triplesVOS) {
                if (StringUtils.isNotEmpty(triplesVO.getVersion())) {
                    nodeSqlList.add("match(f:" + triplesVO.getCategory() + "{code:'" + triplesVO.getCode() + "'})" +
                            "-[:Version]->(p:Property{version:'" + triplesVO.getVersion() + "'}) detach delete p");
                }
            }
        }
        List<String> searchRelationDelSql = new ArrayList<>();
        //补充删除 1.没有父关系 限定为Scene->Factor->Condition->Property  没有版本关系
        searchRelationDelSql.add("match (n:Factor) where not (n)-[:Factor]-(:Scene) detach delete n");
        searchRelationDelSql.add("match (n:Factor) where not (n)-[:Version]-(:Property) detach delete n");
        searchRelationDelSql.add("match (n:Condition) where not (n)-[:Condition]-(:Factor) detach delete n");
        searchRelationDelSql.add("match (n:Condition) where not (n)-[:Version]-(:Property) detach delete n");
        searchRelationDelSql.add("match (n:Conclusion) where not (n)-[:Conclusion]-(:Condition) detach delete n");
        searchRelationDelSql.add("match (n:Conclusion) where not (n)-[:Version]-(:Property) detach delete n");
        //2.所有Property标签下没有关系的节点
        searchRelationDelSql.add("match (n:Property) where size((n)--())=0 delete (n)");

        //执行删除逻辑
        sessionExecute(sessionFactory1, nodeSqlList, searchRelationDelSql);
        sessionExecute(sessionFactory2, nodeSqlList, searchRelationDelSql);
        return "success";
    }

    @Override
    public Object postTriplesQuery(TriplesRequest triplesRequest) throws Exception {
        double version = Double.parseDouble(triplesRequest.getVersion());
        boolean keepAllNodes = triplesRequest.getKeepAllNodes();
        List<Scene> scenes = triplesRepository.findAllByCode(triplesRequest.getScene().getCode());
        TriplesResponse triplesRes = new TriplesResponse();
        List<List<TriplesVO>> list = new ArrayList<>();
        if (CollectionUtils.isEmpty(scenes)) {
            return triplesRes;
        }
        triplesRes.setScenes(scenes);
        for (Scene scene : scenes) {
            triplesRes.setScene(TriplesVO.builder().id(scene.getId()).name(scene.getName()).code(scene.getCode()).build());
            List<Factor> factors = scene.getFactor();
            if (CollectionUtils.isEmpty(factors)) {
                continue;
            }
            for (Factor factor : factors) {
                //factor版本择优
                List<Property> fProperties = sortProperty(keepAllNodes, version, factor.getProperty());
                factor.setProperty(fProperties);
                List<Condition> conditions = factor.getCondition();
                if (CollectionUtils.isEmpty(conditions) || (!keepAllNodes && CollectionUtils.isEmpty(fProperties))) {
                    continue;
                }
                for (Condition condition : conditions) {
                    //condition版本择优
                    List<Property> conditionProperties = sortProperty(keepAllNodes, version, condition.getProperty());
                    condition.setProperty(conditionProperties);
                    List<Conclusion> conclusions = condition.getConclusion();
                    if (CollectionUtils.isEmpty(conclusions) || (!keepAllNodes && CollectionUtils.isEmpty(conditionProperties))) {
                        continue;
                    }
                    for (Conclusion conclusion : conclusions) {
                        //conclusion版本择优
                        List<Property> conclusionProperties = sortProperty(keepAllNodes, version,
                                conclusion.getProperty());
                        conclusion.setProperty(conclusionProperties);
                        if ((!keepAllNodes && CollectionUtils.isEmpty(conclusionProperties))) {
                            continue;
                        }
                        List<TriplesVO> tripList = new ArrayList<>();
                        tripList.add(TriplesVO.builder()
                                .category(Neo4jConstants.AUTO_PROGRESSIVE_FACTOR)
                                .id(factor.getId())
                                .name(factor.getName())
                                .code(factor.getCode())
                                .version(factor.getProperty().get(0).getVersion())
                                .value(factor.getProperty().get(0).getValue()).build());
                        tripList.add(TriplesVO.builder()
                                .category(Neo4jConstants.AUTO_PROGRESSIVE_CONDITION)
                                .id(condition.getId())
                                .name(condition.getName())
                                .code(condition.getCode())
                                .version(condition.getProperty().get(0).getVersion())
                                .value(condition.getProperty().get(0).getValue()).build());
                        tripList.add(TriplesVO.builder()
                                .category(Neo4jConstants.AUTO_PROGRESSIVE_CONCLUSION)
                                .id(conclusion.getId())
                                .name(conclusion.getName())
                                .code(conclusion.getCode())
                                .version(conclusion.getProperty().get(0).getVersion())
                                .value(conclusion.getProperty().get(0).getValue()).build());
                        list.add(tripList);
                    }
                }
            }
        }
        triplesRes.setTriplesVOList(list);
        return triplesRes;
    }

    private List<Property> sortProperty(boolean keepAllNodes, double version, List<Property> propertyList) {
        if (CollectionUtils.isEmpty(propertyList)) {
            if (keepAllNodes) {
                Property property = new Property();
                return Collections.singletonList(property);
            } else {
                return new ArrayList<>();
            }
        }
        return propertyList.stream().filter(v -> (Double.parseDouble(v.getVersion()) <= version)).sorted(Comparator.comparing(Property::getVersion).reversed()).collect(Collectors.toList());
    }

    private void bindParentNode(TriplesVO scene, List<TriplesVO> triplesVO) throws DWException {
        if (ObjectUtils.isEmpty(scene)) {
            return;
        }
        TriplesVO triplesVO1 = triplesVO.get(0);
        if (StringUtils.isNotEmpty(triplesVO1.getParent()) && triplesVO1.getParent().equals(scene.getCode())) {
            List<String> sql =
                    Collections.singletonList("match (f:" + scene.getCategory() + "{code:'" + scene.getCode() +
                            "'}),(ff:" + triplesVO1.getCategory() + "{code:'" + triplesVO1.getCode() + "'}) merge (f)" +
                            "-[:" + triplesVO1.getCategory() + "]->(ff);");
            sessionExecute(sessionFactory1, sql);
            sessionExecute(sessionFactory2, sql);
        }
    }

    @SafeVarargs
    private static void sessionExecute(SessionFactory sessionFactory,
                                       List<String>... sqlList) throws DWException {
        if (null != sessionFactory) {
            Session session = sessionFactory.openSession();
            for (List<String> string : sqlList) {
                for (String sql : string) {
                    if (StringUtils.isNotEmpty(sql)) {
                        try {
                            session.query(sql, new HashMap<>());
                        } catch (Exception e) {
                            if (false) {
                                log.info("ignoreException:{}", e.toString());
                            } else {
                                throw new DWException(e.toString(), e);
                            }
                        }
                    }
                }
            }
            session.clear();

        }
    }

}
