package com.digiwin.athena.knowledgegraph.svcomposition.graph;

import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * @ClassName MatrixGraph
 * @Description TODO
 * @Author zhuangli
 * @Date 2020/7/14 15:33
 * @Version 1.0
 **/
public class MatrixGraph<T> implements Graph{

    private final int DISCONNECTED = 0;
    private final int CONNECTED = 1;

    protected ArrayList<T> vertexList;

    protected int[][] matrix;

    private int order;

    public MatrixGraph(int size){
        size = size<10?10:size;
        this.vertexList = new ArrayList<T>(size);
        this.matrix = new int[size][size];

        for(int i=0;i<size;i++){
            for(int j=0;j<size;j++){
                this.matrix[i][j] = DISCONNECTED;
            }
        }
    }

    public MatrixGraph(T[] vertices, Edge[] edges){
        this(vertices.length);
        for (T vertex : vertices) {
            insertVertex(vertex);
        }
        if (edges != null) {
            for (Edge edge : edges) {
                insertEdge(edge);
            }
        }
    }

    public int[][] getMatrix() {
        return matrix;
    }

    public ArrayList<T> getVertexList() {
        return vertexList;
    }

    public int size() {
        return order;
    }

    @Override
    public int inDegree(int ver) {
        if (ver < 0 || ver > order) {
            throw new IndexOutOfBoundsException(I18nUtils.getValue("knowledgegraph.indexOutOfBounds",ver));
        }
        int inDegree = 0;
        for (int hor = 0; hor < order; hor++) {
            if (matrix[hor][ver] != DISCONNECTED) {
                inDegree++;
            }
        }
        return inDegree;
    }

    @Override
    public int outDegree(int hor) {
        if (hor < 0 || hor > order) {
            throw new IndexOutOfBoundsException(I18nUtils.getValue("knowledgegraph.indexOutOfBounds",hor));
        }
        int outDegree = 0;
        for (int ver = 0; ver < order; ver++) {
            if (matrix[hor][ver] != DISCONNECTED) {
                outDegree++;
            }
        }
        return outDegree;
    }

    @Override
    public List<Integer> inNodeIndexes(int currentNodeSeq) {
        List<Integer> inIndexes = new ArrayList<>();
        for (int i = 0; i < order; i++) {
            if (matrix[i][currentNodeSeq] > 0) {
                inIndexes.add(i);
            }
        }
        return inIndexes;
    }

    @Override
    public List<Integer> outNodeIndexes(int currentNodeSeq) {
        List<Integer> outIndexes = new ArrayList<>();
        for (int i = 0; i < order; i++) {
            if (matrix[currentNodeSeq][i] > 0) {
                outIndexes.add(i);
            }
        }
        return outIndexes;
    }

    public void trimEdge() {
        for (int i = 0; i < order; i++) {
            for (int j = 0; j < order; j++) {
                if (j == i || matrix[i][j] <= 0) {
                    continue;
                } else {
                    boolean existOtherPath = findOtherPath(i, j);
                    if (existOtherPath) {
                        removeEdge(i, j);
                    }
                }
            }
        }
    }

    private boolean findOtherPath(int start, int dest) {
        for (int i = 0; i < order; i++) {
            if (i == start || i == dest || matrix[i][dest] <= 0) {
                continue;
            } else {
                return traceBack(start, i);
            }
        }
        return false;
    }

    private boolean traceBack(int start, int mid) {
        for (int i = 0; i < order; i++) {
            if (i == mid || matrix[i][mid] <= 0) {
                continue;
            } else if (i == start) {
                return true;
            } else if(traceBack(start, i)){
                return true;
            }
        }
        return false;
    }

    public int vertexCount(){return this.vertexList.size();}

    public T get(int i){return this.vertexList.get(i);}

    public int getWeight(int i,int j){return this.matrix[i][j];}

    public int insertVertex(T x){
        //顺序表追加元素，自动扩充
        this.vertexList.add(x);
        //若二维数组不足,则扩充
        if(this.vertexCount()>this.matrix.length){
            //定义了局部变量i,j;
            int temp[][] = matrix,i,j;
            //二维数组扩充2倍
            this.matrix = new int[temp.length*2][temp.length*2];
            for(i=0;i<temp.length;i++){
                for (j = 0; j < temp.length; j++) {
                    this.matrix[i][j] = temp[i][j];
                }
                for (j = temp.length; j < temp.length * 2; j++) {
                    this.matrix[i][j] = DISCONNECTED;
                }
            }
            for (i = temp.length; i < temp.length * 2; i++) {
                for (j = 0; j < temp.length * 2; j++) {
                    this.matrix[i][j] = DISCONNECTED;
                }
            }
        }
        order++;
        //返回插入顶点的序号
        return this.vertexList.size()-1;
    }

    public void  insertEdge(int i,int j){
        int n = this.vertexCount();
        if (i >= 0 && i < n && j >= 0 && j < n && this.matrix[i][j] == DISCONNECTED && i != j) {
            this.matrix[i][j] = CONNECTED;
        }
    }

    public void  insertEdge(int i,int j,int weight){
        int n = this.vertexCount();
        if (i >= 0 && i < n && j >= 0 && j < n && this.matrix[i][j] == DISCONNECTED && i != j) {
            this.matrix[i][j] = weight;
        }
    }

    @Override
    public void insertEdge(Edge edge){
        this.insertEdge(edge.start, edge.dest, edge.weight);
    }

    @Override
    public void removeEdge(int i, int j){
        if (i >= 0 && i < vertexCount() && j >= 0 && j < vertexCount() && i != j) {
            this.matrix[i][j] = DISCONNECTED;
        }
    }

    public void removeVertex(int i){
        int n = this.vertexCount();
        if (i < 0 || i > n) {
            return;
        }
        this.vertexList.remove(i);
        for (int j = 0; j < i; j++) {
            for (int k = i + 1; k < n; k++) {
                //元素向左移一行
                this.matrix[j][k-1] = this.matrix[j][k];
            }
        }
        for (int j = i + 1; j < n; j++) {
            for (int k = 0; k < i; k++) {
                //元素向上移一行
                this.matrix[j-1][k] = this.matrix[j][k];
            }
        }

        for (int j = i + 1; j < n; j++) {
            for (int k = i + 1; k < n; k++) {
                this.matrix[j-1][k-1] = this.matrix[j][k];
            }
        }
        order--;
    }

    @Override
    public String toString(){
        String str = "顶点集合:"+this.vertexList.toString()+"\n邻接矩阵：\n";
        int n = this.vertexCount();
        for(int i=0;i<n;i++){
            for (int j = 0; j < n; j++) {
                str += this.matrix[i][j];
            }
            str +="\n";
        }
        return str;
    }

    public static void main(String[] args){
        String[] verices = {"A","B","C","D","E"};
        Edge edges[] = {new Edge(0,1,5),new Edge(0,3,2),new Edge(1,0,5),
                new Edge(1,2,7),new Edge(1,3,6),new Edge(2,1,7),
                new Edge(2,3,8),new Edge(2,4,3),new Edge(3,0,2),
                new Edge(3,1,6),new Edge(3,2,8),new Edge(3,4,9),
                new Edge(4,2,3),new Edge(4,3,9)};
        MatrixGraph<String> graph = new MatrixGraph<String>(verices,edges);
        System.out.println("带权无向图"+graph.toString());
        System.out.println("插入顶点F,插入边(A,F,9),删除顶点C,删除边(D,E)");
        int i = graph.insertVertex("F");
        graph.insertEdge(0,i,9);
        graph.insertEdge(i,0,9);
        graph.removeVertex(2);
        graph.removeEdge(2, 3);
        graph.removeEdge(3, 2);
        System.out.println(graph.toString());
    }
}
