package com.digiwin.athena.knowledgegraph.app;

import com.digiwin.app.container.exceptions.DWArgumentException;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.knowledgegraph.domain.application.AppTenantVersion;
import com.digiwin.athena.knowledgegraph.repo.AppTenantVersionRepository;
import com.digiwin.athena.knowledgegraph.service.IAppTenantVersionService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

/**
 * @program: athena_backend
 * @description: 租户应用版本服务
 * @author: lingxiang
 * @create: 2022-04-20 16:00
 **/
@Slf4j
@Service
public class AppTenantVersionService implements IAppTenantVersionService {

    @Autowired
    private AppTenantVersionRepository appTenantVersionRepository;
    @Autowired
    @Qualifier("knowledgegraphTenant")
    private MongoTemplate mongoTemplate;//租户级公用库

    @Override
    public Object put(AppTenantVersion appTenantVersion) throws DWBusinessException {
        if (StringUtils.isEmpty(appTenantVersion.getAppCode())) {
            throw new DWArgumentException("appTenantVersion", I18nUtils.getValue("knowledgegraph.appTenantVersion.appCodeEmpty"));
        }
        if (StringUtils.isEmpty(appTenantVersion.getAppVersion())) {
            throw new DWArgumentException("appTenantVersion", I18nUtils.getValue("knowledgegraph.appTenantVersion.appVersionEmpty"));
        }
        if (StringUtils.isEmpty(appTenantVersion.getDevVersion())) {
            throw new DWArgumentException("appTenantVersion", I18nUtils.getValue("knowledgegraph.appTenantVersion.devVersionEmpty"));
        }

        //入参有传tenantId 则以入参为准
        if (StringUtils.isEmpty(appTenantVersion.getTenantId())) {
            //没有传tenantId 以当前header token为准
            appTenantVersion.setTenantId(AthenaUtils.getTenantId());
        }

        try {
            AppTenantVersion re = appTenantVersionRepository.findOneByTenantIdAndAppCode(appTenantVersion.getTenantId(), appTenantVersion.getAppCode());
            if (re == null) {//不存在 则新增
                appTenantVersionRepository.insert(appTenantVersion);
            } else {//已存在 则以当前入参版本更新
                re.setAppVersion(appTenantVersion.getAppVersion());
                re.setDevVersion(appTenantVersion.getDevVersion());
                appTenantVersionRepository.save(re);
            }
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }
        return null;
    }

    @Override
    public Object post(AppTenantVersion appTenantVersion) throws DWBusinessException {
        if (StringUtils.isEmpty(appTenantVersion.getAppCode())) {
            throw new DWArgumentException("appTenantVersion", I18nUtils.getValue("knowledgegraph.appTenantVersion.appCodeEmpty"));
        }
        if (StringUtils.isEmpty(appTenantVersion.getAppVersion()) && StringUtils.isEmpty(appTenantVersion.getDevVersion())) {
            throw new DWArgumentException("appTenantVersion", I18nUtils.getValue("knowledgegraph.appTenantVersion.appVersionAndDevVersionEmpty"));
        }

        //入参有传tenantId 则以入参为准
        if (StringUtils.isEmpty(appTenantVersion.getTenantId())) {
            //没有传tenantId 以当前header token为准
            appTenantVersion.setTenantId(AthenaUtils.getTenantId());
        }

        //更新条件
        Criteria criteria = new Criteria();
        criteria.and("tenantId").is(appTenantVersion.getTenantId()).and("appCode").is(appTenantVersion.getAppCode());

        //更新字段
        Update update = new Update();
        if (!StringUtils.isEmpty(appTenantVersion.getAppVersion())) {
            update.set("appVersion", appTenantVersion.getAppVersion());//更新应用版本
        }
        if (!StringUtils.isEmpty(appTenantVersion.getDevVersion())) {
            update.set("devVersion", appTenantVersion.getDevVersion());//更新开发版本
        }

        try {
            mongoTemplate.upsert(new Query(criteria), update, AppTenantVersion.class);//不存在则新增
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }
        return null;
    }

    @Override
    public Object delete(String appCode) throws DWBusinessException {
        if (StringUtils.isEmpty(appCode)) {
            throw new DWArgumentException("appTenantVersion", I18nUtils.getValue("knowledgegraph.appTenantVersion.appCodeEmpty"));
        }

        try {
            //删除条件
            Criteria criteria = new Criteria();
            criteria.and("tenantId").is(AthenaUtils.getTenantId()).and("appCode").is(appCode);

            mongoTemplate.remove(new Query(criteria), AppTenantVersion.class);
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     *  根据当前登录租户 appCode appVersioin 查询对应的 开发版本号version
     */
    @Override
    public AppTenantVersion postQueryVersion(String appCode) throws DWBusinessException {
        AppTenantVersion re;
        try {
            re = appTenantVersionRepository.findOneByTenantIdAndAppCode(AthenaUtils.getTenantId(), appCode);
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }
        return re;
    }
}