package com.digiwin.athena.knowledgegraph.cache;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.dao.DWServiceResultBuilder;
import com.digiwin.athena.domain.log.OperationRecord;
import com.digiwin.athena.domain.log.OperationRecordType;
import com.digiwin.athena.knowledgegraph.service.ICacheService;
import com.digiwin.athena.knowledgegraph.service.TagSystemService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import com.digiwin.athena.knowledgegraph.utils.OperationRecordUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.*;

@Service
public class CacheService implements ICacheService {
    private static final Logger log = LoggerFactory.getLogger(CacheService.class);
    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;
    @Autowired
    @Qualifier("dataFootprintTemplate")
    RestTemplate restTemplate;
    @Value("${dataFootprintUrl}")
    private String dataFootprintUrl;
    @Autowired
    OperationRecordUtil recordUtil;
    @Autowired
    private TagSystemService tagSystemService;

    public static Map<String, String> compatibleKey = new HashMap<>();

    static {
        compatibleKey.put("kg", "knowledgegraph");
        compatibleKey.put("tag", "tagsystem");
        compatibleKey.put("presetLibrary", "presetlibrary");
    }

    @Override
    public Object deleteKeys(List<String> keys) throws Exception {
        final Long delete = this.redisTemplate.delete(keys);
        recordUtil.recordInfo(OperationRecord.builder()
                .type(OperationRecordType.CLEAN_REDIS.toString())
                .supplementaryContent(keys).build());
        return DWServiceResultBuilder.build(I18nUtils.getValue("knowledgegraph.deleteNums") + delete);
    }

    @Override
    public Object deleteWithPattern(String keyPattern) throws Exception {
        log.info("deleteWithPattern keyPattern,{}", keyPattern);
        keyPattern = compatiblePattern(keyPattern);
        int count = 2000;
        Long delete = blurScanDelete(keyPattern, count);
        log.info("deleteWithPattern delete size,{}", delete);
        deleteDataFootprintCache();
        // 初始化tag数据，存入redis缓存
        if (keyPattern.startsWith("tag")) {
            tagSystemService.initTagData();
        }
        return DWServiceResultBuilder.build(I18nUtils.getValue("knowledgegraph.deleteNums") + delete);
    }

    @Override
    public Object deleteWithPattern(List<String> keyPattern) throws Exception {
        int count = 2000;
        Long delete = 0L;
        for (String pattern : keyPattern) {
             delete += blurScanDelete(pattern, count);
            // 初始化tag数据，存入redis缓存
            if (pattern.startsWith("tag")) {
                tagSystemService.initTagData();
            }
        }
        deleteDataFootprintCache();
        return DWServiceResultBuilder.build(I18nUtils.getValue("knowledgegraph.deleteNums") + delete);
    }

    @Override
    public Object getData(String key) throws Exception {
        int count = 2000;
        return DWServiceResultBuilder.build(blurScan(key, count));
    }

    @Override
    public Object getSetData(String key, String value) throws Exception {
        redisTemplate.opsForValue().set(key, value);
        return DWServiceResultBuilder.build(I18nUtils.getValue("knowledgegraph.setDatasuccess"));
    }


    private void deleteDataFootprintCache() throws DWBusinessException {
        try {
            Map<String, Object> requestMap = new HashMap<>();
            // SD-胡威要求_改为:
            requestMap.put("pattern", "datafootprint:v2");
            HttpHeaders headers = new HttpHeaders();
            headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
            headers.add("token", AthenaUtils.getHeaderToken());
            MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
            headers.setContentType(type);
            HttpEntity<Map> requestEntity = new HttpEntity<>(requestMap, headers);
            ResponseEntity<Object> result = restTemplate.exchange(dataFootprintUrl + "/restful/service/DataFootprint" +
                    "/manual/deleteCache", HttpMethod.POST, requestEntity, Object.class);
            if (!(result.getStatusCodeValue() == 200)) {
                log.error("deleteDataFootprintCache failed");
            }
        } catch (Exception e) {
            log.error("deleteDataFootprintCache failed,{}", e.toString());
        }
    }

    /**
     * scan 方式模糊匹配 禁用keys
     *
     * @param matchKey  匹配的key
     * @param count     扫描数量
     * @return 扫描到的keys
     */
    public Set<String> blurScan(String matchKey, int count) {
        return redisTemplate.execute((RedisCallback<Set<String>>) connection -> {
            Set<String> keys = new HashSet<>();
            try(Cursor<byte[]> cursor = connection.scan(new ScanOptions.ScanOptionsBuilder().count(count).match(matchKey).build())) {
                while (cursor.hasNext()) {
                    keys.add(new String(cursor.next()));
                }
            } catch (Exception e) {
                log.error("blurScan failed,{}", e.toString());
            }
            return keys;
        });
    }

    /**
     * scan方式模糊匹配清除Redis缓存 禁用keys
     *
     * @param matchKey  匹配的key
     * @param count     扫描数量
     * @return 删除key的数量
     */
    public Long blurScanDelete(String matchKey, int count) {
        return redisTemplate.execute((RedisCallback<Long>) connection -> {
            long keyCount = 0;
            try(Cursor<byte[]> cursor = connection.scan(new ScanOptions.ScanOptionsBuilder().count(count).match(matchKey).build())) {
                long cursorId = cursor.getCursorId();
                Set<String> keys = new HashSet<>();
                while (cursor.hasNext()) {
                    keys.add(new String(cursor.next()));
                    if (cursorId != cursor.getCursorId()) {
                        keyCount += keys.size();
                        // 删除redis缓存并记录mongo
                        this.deleteByKeys(keys);
                        // 重置keys集合
                        keys = new HashSet<>();
                    }
                    cursorId = cursor.getCursorId();
                }
                // 处理最后一批数据或者唯一一批数据
                if (cursor.getCursorId() == 0 || CollectionUtils.isNotEmpty(keys)) {
                    keyCount += keys.size();
                    // 删除redis缓存并记录mongo
                    this.deleteByKeys(keys);
                }
            } catch (Exception e) {
                log.error("blurScanDelete failed,{}", e.toString());
            }
            return keyCount;
        });
    }

    /**
     * 根据keys删除缓存
     * @param keys  keys集合
     */
    private void deleteByKeys(Set<String> keys) {
        this.redisTemplate.delete(keys);
        recordUtil.recordInfo(OperationRecord.builder()
                .type(OperationRecordType.CLEAN_REDIS.toString())
                .supplementaryContent(keys).build());
    }

    private String compatiblePattern(String keyPattern) {
        //历史key兼容去除前缀
        keyPattern = keyPattern.replace("km:", "");
        for (Map.Entry<String, String> entry : compatibleKey.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            //有旧key就换为新的  有新key就不替换
            if (keyPattern.contains(key) && !keyPattern.contains(value)) {
                keyPattern = keyPattern.replace(key, value);
            }
        }
        return keyPattern;
    }
}
