package com.digiwin.athena.knowledgegraph.component;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.domain.component.ComponentTypeEnum;
import com.digiwin.athena.domain.component.bo.*;
import com.digiwin.athena.knowledgegraph.domain.ComponentStructuredList;
import com.digiwin.athena.knowledgegraph.domain.TenantEntity;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.knowledgegraph.domain.mechanism.MechanismTenantParadigm;
import com.digiwin.athena.knowledgegraph.service.DataMapService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.service.impl.AppService;
import com.digiwin.athena.knowledgegraph.service.impl.IamService;
import com.digiwin.athena.knowledgegraph.service.impl.TenantService;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.knowledgegraph.support.SignOffAppHolder;
import com.google.common.base.Joiner;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.ogm.model.Result;
import org.neo4j.ogm.session.Session;
import org.neo4j.ogm.session.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class ComponentInitManager {

    @Autowired
    SessionFactory neo4jSessionFactory;

    @Autowired
    private DataMapService dataMapService;

    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplateSystem;

    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplateUser;

    @Autowired
    TenantService tenantService;
    @Autowired
    IamService iamService;
    @Autowired
    ComponentManager componentManager;
    @Autowired
    AppService appService;
    @Autowired
    KgInnerService kgInnerService;
    @Autowired
    DataPickService dataPickService;
    @Autowired
    @Qualifier("baseExecutor")
    ThreadPoolTaskExecutor executor;

    public List<ComponentBO> getAllComponentListByApplication(String appNameSpace, String appCode, String version, String tenantId) throws DWBusinessException {
        List<ComponentBO> result = new ArrayList<>();
        //查找MonitorRuleComponent
        List<ComponentBO> monitorRuleComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.MonitorRule);
        //查找MechanismVariableComponent
        List<ComponentBO> mechanismVariableComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.MechanismVariable);
        //查找ProjectComponent
        List<ComponentBO> projectComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.Project);
        //查找TaskComponent
        List<ComponentBO> taskComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.Task);
        //查找BaseDataEntryComponent
        List<ComponentBO> baseDataEntryComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.BaseDataEntry);
        //查找ReportComponent
        List<ComponentBO> reportComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.Report);
        //TODO ActionComponent暂时不拉取，等杰程的接口

        //查找ServiceCompositionComponent
        List<ComponentBO> serviceCompositionComponent = this.getServiceCompositionComponent(appNameSpace, appCode, version, null);

        result.addAll(monitorRuleComponent);
        result.addAll(mechanismVariableComponent);
        result.addAll(projectComponent);
        result.addAll(taskComponent);
        result.addAll(baseDataEntryComponent);
        result.addAll(reportComponent);
        result.addAll(serviceCompositionComponent);
        return result;
    }

    /**
     * 查询应用的组件列表
     * @param appCodes 应用编码
     * @param version 版本
     * @param tenantId 租户id
     * @param typeEnum 组件类型
     * @return 组件列表
     * @throws DWBusinessException 抛出异常
     */
    public List<ComponentBO> getComponentList(List<String> appCodes, String version, String tenantId, ComponentTypeEnum typeEnum) throws DWBusinessException {
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        String type = null;
        String subType = null;
        switch (typeEnum){
            case MonitorRule:
                type = "monitorRule";
                break;
            case MechanismVariable:
                type = "variable";
                break;
            case Project:
                type = "task";
                break;
            case Task:
                type = "activity";
                subType = "activity";
                break;
            case BaseDataEntry:
                type = "activity";
                subType = "base_entry";
                break;
            case Report:
                type = "activity";
                subType = "statement";
                break;
            case ServiceComposition:
                type = "serviceComposition";
                break;
            default:
                throw new DWBusinessException("Unexpected value: " + typeEnum);
        }
        List<ApplicationRelation> monitorRules = dataPickService.applicationRelations(appCodes, type, subType, tenantId, version);
        for (ApplicationRelation applicationRelation : monitorRules) {
            if(existsCodes.contains(applicationRelation.getCode())){
                continue;
            }
            existsCodes.add(applicationRelation.getCode());
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(applicationRelation.getCode());
            componentBO.setType(typeEnum);
            componentBO.setName(applicationRelation.getName());
            componentBO.setVersion(applicationRelation.getVersion());
            result.add(componentBO);
        }
        return result;
    }

    /**
     * 获取通用组件列表
     * @param version 版本
     * @param tenantId 租户id
     * @param typeEnum 组件类型
     * @return 组件列表
     * @throws DWBusinessException 抛出异常
     */
    public List<ComponentBO> getCommonComponentList(String version, String tenantId, ComponentTypeEnum typeEnum) throws DWBusinessException {
        List<String> commonAppCodes = dataPickService.commonAppCodes(version);
        return getComponentList(commonAppCodes, version, tenantId, typeEnum);
    }

    public void initializeApplicationComponentList(String appCode, String paradigm, String version) throws DWBusinessException {
        List<ComponentBO> result = this.getAllComponentListByApplication(appCode, appCode, version, null);

        Query query = new Query();
        Criteria criteria = Criteria.where("type").is("BaseFunction").and("appCode").is(appCode)
                .and("paradigm").is(paradigm).and("version").is(version).and("structureVersion").is("1.0");
        ComponentStructuredList componentStructuredList = dataPickService.findOneByCondition(criteria, ComponentStructuredList.class, "componentStructuredList");
        criteria.and("tenantId").is(componentStructuredList.getTenantId());
        if (componentStructuredList.getSourceLevel() != null) {
            criteria.and("sourceLevel").is(componentStructuredList.getSourceLevel());
        }
        query.addCriteria(criteria);
        Update update = new Update();
        update.set("components", result);
        this.mongoTemplateSystem.upsert(query, update, "componentStructuredList");

    }

    /**
     * 根据应用code获取机制变量
     * @param appNameSpace 应用空间名称
     * @param appCode 应用code
     * @param version 版本号
     * @param tenantId 租户id
     * @return 返回机制变量
     */
    public List<ComponentBO> getMechanismVariableComponent(String appNameSpace, String appCode, String version, String tenantId){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        Session session = this.neo4jSessionFactory.openSession();
        Result query = null;
        if(StringUtils.isEmpty(tenantId)) {
            query = session.query("match(n:GetMechanismVariableAction)-[:Creates]->(m:DataField) where n.version = '" + version + "' and (n.nameSpace='" + appNameSpace + "' or n.athena_namespace = '" + appCode + "') return distinct m.name as code", new HashMap<>());
        }else{
          //  query = session.query("match (te:TenantEntity{tenantId:'" + tenantId + "'})-[:ACTION]->(n:GetMechanismVariableAction)-[:Creates]->(m:DataField) where n.version = te.version and (n.nameSpace='" + appNameSpace + "' or n.athena_namespace = '" + appCode + "') return distinct m.name as code", new HashMap<>());
            query = session.query("match (n:GetMechanismVariableAction)-[:Creates]->(m:DataField) where n.version = '"+version+"' and (n.tenantId is null or  n.tenantId in ['SYSTEM','"+tenantId+"']) and (n.nameSpace='" + appNameSpace + "' or n.athena_namespace = '" + appCode + "') return distinct m.name as code", new HashMap<>());

        }
        Iterator<Map<String, Object>> iterator = query.iterator();
        while(iterator.hasNext()){
            Map<String, Object> next = iterator.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.MechanismVariable);
            componentBO.setVersion("1.0");
            result.add(componentBO);
        }

        return result;
    }

    public List<ComponentBO> getCommonMechanismVariableComponent(String version){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        Session session = this.neo4jSessionFactory.openSession();
       // Result query = session.query("match (te:TenantEntity{ifCommon:true,version:'" + version + "'})-[:ACTION]->(n:GetMechanismVariableAction)-[:Creates]->(m:DataField) where n.version = te.version return distinct m.name as code", new HashMap<>());
        Result query = session.query("match (n:GetMechanismVariableAction{commonApp:true})-[:Creates]->(m:DataField) where n.version ='" + version + "'  return distinct m.name as code", new HashMap<>());

        Iterator<Map<String, Object>> iterator = query.iterator();
        while(iterator.hasNext()){
            Map<String, Object> next = iterator.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.MechanismVariable);
            componentBO.setVersion("1.0");
            result.add(componentBO);
        }

        return result;
    }

    public static void main(String[] args) {
        String version = "1.0";
        String tenantId ="tenantId";
        String appNameSpace = "appNameSpace1";
        String appCode="appCode1";
        String str ="(monitorRule2:MonitorRule) where monitorRule2.version = '"+version+"'   and (monitorRule2.tenantId is null or monitorRule2.tenantId in ['SYSTEM','"+tenantId+"'] ) and (monitorRule2.nameSpace='" + appNameSpace + "' or monitorRule2.relation_namespace='" + appNameSpace + "' or monitorRule2.athena_namespace = '" + appCode + "') return distinct monitorRule2.code as code, monitorRule2.name as name, monitorRule2.version as version";
        System.out.println(str);
    }

    /**
     * 根据应用code获取侦测数据
     * @param appNameSpace 应用空间名称
     * @param appCode 应用code
     * @param version 租户版本号
     * @param tenantId 租户id
     * @return 返回侦测数据
     */
    public List<ComponentBO> getMonitorRuleComponent(String appNameSpace, String appCode, String version, String tenantId){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        Session session = this.neo4jSessionFactory.openSession();
        Result query = null;
        if(StringUtils.isEmpty(tenantId)) {
            query = session.query("match(monitorRule2:MonitorRule) where monitorRule2.version = '" + version + "' and (monitorRule2.nameSpace='" + appNameSpace + "' or monitorRule2.relation_namespace='" + appNameSpace + "' or monitorRule2.athena_namespace = '" + appCode + "') return distinct monitorRule2.code as code, monitorRule2.name as name, monitorRule2.version as version", new HashMap<>());
        }else{
            //query = session.query("match (te:TenantEntity{tenantId:'" + tenantId + "'})-[:USE]->(monitorRule2:MonitorRule) where monitorRule2.version = te.version and (monitorRule2.nameSpace='" + appNameSpace + "' or monitorRule2.relation_namespace='" + appNameSpace + "' or monitorRule2.athena_namespace = '" + appCode + "') return distinct monitorRule2.code as code, monitorRule2.name as name, monitorRule2.version as version", new HashMap<>());
            query = session.query("match(monitorRule2:MonitorRule) where monitorRule2.version = '"+version+"'   and (monitorRule2.tenantId is null or monitorRule2.tenantId in ['SYSTEM','"+tenantId+"'] ) and (monitorRule2.nameSpace='" + appNameSpace + "' or monitorRule2.relation_namespace='" + appNameSpace + "' or monitorRule2.athena_namespace = '" + appCode + "') return distinct monitorRule2.code as code, monitorRule2.name as name, monitorRule2.version as version", new HashMap<>());

        }
        Iterator<Map<String, Object>> iterator = query.iterator();
        while(iterator.hasNext()){
            Map<String, Object> next = iterator.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));

            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.MonitorRule);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));
            result.add(componentBO);
        }

        return result;
    }

    public List<ComponentBO> getCommonMonitorRuleComponent(String version){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        Session session = this.neo4jSessionFactory.openSession();
    //    Result query = session.query("match (te:TenantEntity{ifCommon:true,version:'" + version + "'})-[:USE]->(monitorRule2:MonitorRule) where monitorRule2.version = te.version return distinct monitorRule2.code as code, monitorRule2.name as name, monitorRule2.version as version", new HashMap<>());
        Result query = session.query("match(monitorRule2:MonitorRule{commonApp:true}) where monitorRule2.version = '"+version+"'  return distinct monitorRule2.code as code, monitorRule2.name as name, monitorRule2.version as version", new HashMap<>());
        Iterator<Map<String, Object>> iterator = query.iterator();
        while(iterator.hasNext()){
            Map<String, Object> next = iterator.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));

            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.MonitorRule);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));
            result.add(componentBO);
        }

        return result;
    }

    /**
     * 根据应用code获取项目数据
     * @param appNameSpace 应用空间名称
     * @param appCode 应用code
     * @param version 租户版本号
     * @param tenantId 租户id
     * @return 项目数据
     */
    public List<ComponentBO> getProjectComponent(String appNameSpace, String appCode, String version, String tenantId){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        Session session = this.neo4jSessionFactory.openSession();
        Result query1 = null;
        if(StringUtils.isEmpty(tenantId)) {
            query1 = session.query("match(task2:Task) where task2.version = '" + version + "' and (task2.nameSpace = '" + appNameSpace + "' or task2.athena_namespace = '" + appCode + "' or task2.relation_namespace = '" + appNameSpace + "') return distinct task2.code as code, task2.name as name, task2.`lang.name.en_US` as name_us, task2.`lang.name.zh_CN` as name_cn, task2.`lang.name.zh_TW` as name_tw, task2.version as version", new HashMap<>());
        }else{
          //  query1 = session.query("match (te:TenantEntity{tenantId:'" + tenantId + "'})-[:TASK]-> (task2:Task) where task2.version = te.version and (task2.nameSpace = '" + appNameSpace + "' or task2.athena_namespace = '" + appCode + "' or task2.relation_namespace = '" + appNameSpace + "') return distinct task2.code as code, task2.name as name, task2.`lang.name.en_US` as name_us, task2.`lang.name.zh_CN` as name_cn, task2.`lang.name.zh_TW` as name_tw, task2.version as version", new HashMap<>());
            query1 = session.query("match(task2:Task) where task2.version ='" + version + "' and  (task2.tenantId is null or task2.tenantId in ['SYSTEM','"+tenantId+"']) and (task2.nameSpace = '" + appNameSpace + "' or task2.athena_namespace = '" + appCode + "' or task2.relation_namespace = '" + appNameSpace + "') return distinct task2.code as code, task2.name as name, task2.`lang.name.en_US` as name_us, task2.`lang.name.zh_CN` as name_cn, task2.`lang.name.zh_TW` as name_tw, task2.version as version", new HashMap<>());

        }
        Iterator<Map<String, Object>> iterator1 = query1.iterator();
        while(iterator1.hasNext()){
            Map<String, Object> next = iterator1.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.Project);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));

            Map<String, Map<String, String>> lang = new HashMap<>();
            Map<String, String> nameLang = new HashMap<>();
            if(!ObjectUtils.isEmpty(next.get("name_cn"))){
                nameLang.put("zh_CN",  String.valueOf(next.get("name_cn")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_tw"))){
                nameLang.put("zh_TW",  String.valueOf(next.get("name_tw")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_us"))){
                nameLang.put("en_US",  String.valueOf(next.get("name_us")));
            }
            if(!nameLang.isEmpty()){
                lang.put("name", nameLang);
                componentBO.setLang(lang);
            }

            result.add(componentBO);
        }

        try {
            //任务引擎项目
            List<ApplicationRelation> taskList = this.appService.getApplicationRelationByTypeAndAppCodeAndVersion("task", appCode, version);
            if(CollectionUtils.isNotEmpty(taskList)){
                if (Boolean.TRUE.equals(SignOffAppHolder.getSignOffAppFlag())) {
                    // 签核套件定制逻辑，DataMap中只存了项目或者任务模板，没有保存实际运行实例的code
                    handleComponentForSignOffApp(result, taskList, existsCodes, ComponentTypeEnum.Project);
                } else {
                    List<String> collect = taskList.stream().map(ApplicationRelation::getCode).distinct().collect(Collectors.toList());
//                    Map<String,Object> dataMapQuery = new HashMap<>();
//                    dataMapQuery.put("tenantId", "SYSTEM");
//                    dataMapQuery.put("code", collect);
//                    dataMapQuery.put("version", version);
//
//                    Object projectObj = this.dataMapService.getBatchProjects(dataMapQuery);
                    // 根据code从project中获取项目数据
                    Object projectObj = this.dataMapService.queryBatchCodeAndName(collect, "project", version);
                    if(!ObjectUtils.isEmpty(projectObj)){
                        JSONObject jsonObject1 = JSON.parseObject(JSON.toJSONString(projectObj));
                        JSONArray projects = jsonObject1.getJSONArray("datas");
                        for(Object obj : projects){
                            JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(obj));
                            if(existsCodes.contains(String.valueOf(jsonObject.get("code")))){
                                continue;
                            }
                            existsCodes.add(String.valueOf(jsonObject.get("code")));
                            ComponentBO componentBO = new StructuredComponentBO();
                            componentBO.setCode(String.valueOf(jsonObject.get("code")));
                            componentBO.setType(ComponentTypeEnum.Project);
                            componentBO.setName(String.valueOf(jsonObject.get("name")));
                            componentBO.setVersion(String.valueOf(jsonObject.get("version")));

                            JSONObject lang = jsonObject.getJSONObject("lang");
                            if(!ObjectUtils.isEmpty(lang) && lang.containsKey("name")){
                                Map<String, Map<String, String>> lang2 = new HashMap<>();
                                JSONObject name = lang.getJSONObject("name");
                                lang2.put("name", (Map)name);
                                componentBO.setLang(lang2);
                            }

                            result.add(componentBO);
                        }
                    }
                }

                //此处为applicationRelation维护，但可能改了nameSpace导致没查出来的项目，一般为原先流程引擎项目
                List<String> leftCodes = taskList.stream().filter(e -> !existsCodes.contains(e.getCode())).map(ApplicationRelation::getCode).distinct().collect(Collectors.toList());
                if(CollectionUtils.isNotEmpty(leftCodes)){
                    String leftCodesStr = "['" + StringUtils.join(leftCodes, "','") + "']";
                    if(StringUtils.isEmpty(tenantId)) {
                        query1 = session.query("match(task2:Task) where task2.version = '" + version + "' and task2.code in " + leftCodesStr +" return distinct task2.code as code, task2.name as name, task2.`lang.name.en_US` as name_us, task2.`lang.name.zh_CN` as name_cn, task2.`lang.name.zh_TW` as name_tw, task2.version as version", new HashMap<>());
                    }else{
                      //  query1 = session.query("match (te:TenantEntity{tenantId:'" + tenantId + "'})-[:TASK]-> (task2:Task) where task2.version = te.version and task2.code in " + leftCodesStr +" return distinct task2.code as code, task2.name as name, task2.`lang.name.en_US` as name_us, task2.`lang.name.zh_CN` as name_cn, task2.`lang.name.zh_TW` as name_tw, task2.version as version", new HashMap<>());
                        query1 = session.query("match (task2:Task) where task2.version ='" + version + "' and (task2.tenantId is null or task2.tenantId in ['SYSTEM','"+tenantId+"']) and task2.code in " + leftCodesStr +" return distinct task2.code as code, task2.name as name, task2.`lang.name.en_US` as name_us, task2.`lang.name.zh_CN` as name_cn, task2.`lang.name.zh_TW` as name_tw, task2.version as version", new HashMap<>());

                    }

                    Iterator<Map<String, Object>> iterator2 = query1.iterator();
                    while(iterator2.hasNext()){
                        Map<String, Object> next = iterator2.next();
                        if(existsCodes.contains(String.valueOf(next.get("code")))){
                            continue;
                        }
                        existsCodes.add(String.valueOf(next.get("code")));
                        ComponentBO componentBO = new StructuredComponentBO();
                        componentBO.setCode(String.valueOf(next.get("code")));
                        componentBO.setType(ComponentTypeEnum.Project);
                        componentBO.setName(String.valueOf(next.get("name")));
                        componentBO.setVersion(String.valueOf(next.get("version")));

                        Map<String, Map<String, String>> lang = new HashMap<>();
                        Map<String, String> nameLang = new HashMap<>();
                        if(!ObjectUtils.isEmpty(next.get("name_cn"))){
                            nameLang.put("zh_CN",  String.valueOf(next.get("name_cn")));
                        }
                        if(!ObjectUtils.isEmpty(next.get("name_tw"))){
                            nameLang.put("zh_TW",  String.valueOf(next.get("name_tw")));
                        }
                        if(!ObjectUtils.isEmpty(next.get("name_us"))){
                            nameLang.put("en_US",  String.valueOf(next.get("name_us")));
                        }
                        if(!nameLang.isEmpty()){
                            lang.put("name", nameLang);
                            componentBO.setLang(lang);
                        }

                        result.add(componentBO);
                    }
                }
            }


        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

//        Map<String,Object> dataMapQuery = new HashMap<>();
//        dataMapQuery.put("tenantId", "SYSTEM");
//        dataMapQuery.put("application", Arrays.asList(appNameSpace, appCode));
//        dataMapQuery.put("version", version);
//        Object projectObj = this.dataMapService.getBatchProjects(dataMapQuery);
//        if(!ObjectUtils.isEmpty(projectObj)){
//            JSONObject jsonObject1 = JSON.parseObject(JSON.toJSONString(projectObj));
//            JSONArray projects = jsonObject1.getJSONArray("datas");
//            for(Object obj : projects){
//                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(obj));
//                if(existsCodes.contains(String.valueOf(jsonObject.get("code")))){
//                    continue;
//                }
//                existsCodes.add(String.valueOf(jsonObject.get("code")));
//                ComponentBO componentBO = new StructuredComponentBO();
//                componentBO.setCode(String.valueOf(jsonObject.get("code")));
//                componentBO.setType(ComponentTypeEnum.Project);
//                componentBO.setName(String.valueOf(jsonObject.get("name")));
//                componentBO.setVersion(String.valueOf(jsonObject.get("version")));
//
//                result.add(componentBO);
//            }
//        }
//
//        //任务引擎项目
//        List<ApplicationRelation> relations = mongoTemplateSystem.find(Query.query(Criteria.where("type").is("task")
//                .and("appCode").is(appCode)
//                .and("version").is(version)), ApplicationRelation.class);
//        if(CollectionUtils.isNotEmpty(relations)){
//            for(ApplicationRelation applicationRelation : relations){
//                if(existsCodes.contains(String.valueOf(applicationRelation.getCode()))){
//                    continue;
//                }
//                existsCodes.add(applicationRelation.getCode());
//                ComponentBO componentBO = new StructuredComponentBO();
//                componentBO.setCode(applicationRelation.getCode());
//                componentBO.setType(ComponentTypeEnum.Project);
////              componentBO.setName(String.valueOf(jsonObject.get("name")));
//                componentBO.setVersion(version);
//
//                result.add(componentBO);
//            }
//        }

        return result;
    }

    /**
     * 如果是签核套件定制逻辑
     * DataMap中只存了项目或者任务模板，没有保存实际运行实例的code
     *
     * @param result 结果
     * @param list 应用关联信息列表
     * @param existsCodes result中已经有的code
     */
    private void handleComponentForSignOffApp(List<ComponentBO> result, List<ApplicationRelation> list, Set<String> existsCodes, ComponentTypeEnum type) {
        list.forEach(project -> {
            if (!existsCodes.contains(project.getCode())) {
                ComponentBO componentBO = new StructuredComponentBO();
                componentBO.setCode(project.getCode());
                componentBO.setType(type);
                componentBO.setName(project.getName());
                componentBO.setVersion(project.getVersion());
                result.add(componentBO);
            }
        });
    }

    public List<ComponentBO> getTaskComponent(String appNameSpace, String appCode, String version, String tenantId){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        Session session = this.neo4jSessionFactory.openSession();
        Result query1 = null;
        if(StringUtils.isEmpty(tenantId)) {
            query1 = session.query("match(n:Activity) where  n.version = '" + version + "' and n.pattern <> 'STATEMENT' and n.pattern <> 'DATA_ENTRY' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
        }else{
            query1 = session.query("match(n:Activity) where  n.version ='" + version + "' and (n.tenantId is null or n.tenantId in ['SYSTEM','"+tenantId+"']) and n.pattern <> 'STATEMENT' and n.pattern <> 'DATA_ENTRY' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());

        }
        Iterator<Map<String, Object>> iterator1 = query1.iterator();
        while(iterator1.hasNext()){
            Map<String, Object> next = iterator1.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.Task);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));

            Map<String, Map<String, String>> lang = new HashMap<>();
            Map<String, String> nameLang = new HashMap<>();
            if(!ObjectUtils.isEmpty(next.get("name_cn"))){
                nameLang.put("zh_CN",  String.valueOf(next.get("name_cn")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_tw"))){
                nameLang.put("zh_TW",  String.valueOf(next.get("name_tw")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_us"))){
                nameLang.put("en_US",  String.valueOf(next.get("name_us")));
            }
            if(!nameLang.isEmpty()){
                lang.put("name", nameLang);
                componentBO.setLang(lang);
            }
            result.add(componentBO);
        }

        try {
            //查询任务引擎的任务
            List<ApplicationRelation> activityList = this.appService.getApplicationRelationByTypeAndAppCodeAndVersion("activity", appCode, version);
            if(CollectionUtils.isNotEmpty(activityList)){
                List<String> collect = activityList.stream().map(ApplicationRelation::getCode).distinct().collect(Collectors.toList());
//                Map<String,Object> dataMapQuery = new HashMap<>();
//                dataMapQuery.put("tenantId", "SYSTEM");
//                dataMapQuery.put("code", collect);
//                dataMapQuery.put("version", version);
//
//                Object taskObj = this.dataMapService.getBatchTasks(dataMapQuery);
                Object taskObj = this.dataMapService.queryBatchCodeAndName(collect, "task", version);
                if(!ObjectUtils.isEmpty(taskObj)){
                    JSONObject jsonObject1 = JSON.parseObject(JSON.toJSONString(taskObj));
                    JSONArray tasks = jsonObject1.getJSONArray("datas");
                    for(Object obj : tasks){
                        JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(obj));
                        if(existsCodes.contains(String.valueOf(jsonObject.get("code")))){
                            continue;
                        }
                        existsCodes.add(String.valueOf(jsonObject.get("code")));
                        ComponentBO componentBO = new StructuredComponentBO();
                        componentBO.setCode(String.valueOf(jsonObject.get("code")));
                        componentBO.setType(ComponentTypeEnum.Task);
                        componentBO.setName(String.valueOf(jsonObject.get("name")));
                        componentBO.setVersion(String.valueOf(jsonObject.get("version")));

                        JSONObject lang = jsonObject.getJSONObject("lang");
                        if(!ObjectUtils.isEmpty(lang) && lang.containsKey("name")){
                            Map<String, Map<String, String>> lang2 = new HashMap<>();
                            JSONObject name = lang.getJSONObject("name");
                            lang2.put("name", (Map)name);
                            componentBO.setLang(lang2);
                        }
                        result.add(componentBO);
                    }
                }
            }


        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }


//        //任务引擎任务
//        Map<String,Object> dataMapQuery = new HashMap<>();
//        dataMapQuery.put("tenantId", "SYSTEM");
//        dataMapQuery.put("application", Arrays.asList(appNameSpace, appCode));
//        dataMapQuery.put("version", version);
//        Object taskObj = this.dataMapService.getBatchTasks(dataMapQuery);
//        if(!ObjectUtils.isEmpty(taskObj)){
//            JSONObject jsonObject1 = JSON.parseObject(JSON.toJSONString(taskObj));
//            JSONArray projects = jsonObject1.getJSONArray("datas");
//            for(Object obj : projects){
//                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(obj));
//                if(existsCodes.contains(String.valueOf(jsonObject.get("code")))){
//                    continue;
//                }
//                existsCodes.add(String.valueOf(jsonObject.get("code")));
//                ComponentBO componentBO = new StructuredComponentBO();
//                componentBO.setCode(String.valueOf(jsonObject.get("code")));
//                componentBO.setType(ComponentTypeEnum.Task);
//                componentBO.setName(String.valueOf(jsonObject.get("name")));
//                componentBO.setVersion(String.valueOf(jsonObject.get("version")));
//
//                result.add(componentBO);
//            }
//        }

        return result;
    }

    public List<ComponentBO> getBaseDataEntryComponent(String appNameSpace, String appCode, String version, String tenantId){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        String baseDataCodeStr = null;
        try {
            List<ApplicationRelation> activityList = this.appService.getApplicationRelationByTypeAndAppCodeAndVersion("activity", appCode, version);
            List<String> baseDataCodeList = activityList.stream().map(ApplicationRelation::getCode).distinct().collect(Collectors.toList());
            if(CollectionUtils.isNotEmpty(baseDataCodeList)){
                baseDataCodeStr = "['" + StringUtils.join(baseDataCodeList, "','") + "']";
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        Session session = this.neo4jSessionFactory.openSession();
        Result query1 = null;
        if(StringUtils.isEmpty(baseDataCodeStr)){
            if(StringUtils.isEmpty(tenantId)) {
                query1 = session.query("match(n:Activity) where  n.version = '" + version + "' and n.pattern = 'DATA_ENTRY' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
            }else{
              //  query1 = session.query("match (te:TenantEntity{tenantId:'" + tenantId + "'})-[:ACTIVITY]-> (n:Activity) where  n.version = te.version and n.pattern = 'DATA_ENTRY' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
                query1 = session.query("match(n:Activity) where  n.version ='" + version + "' and (n.tenantId is null or n.tenantId in ['SYSTEM','"+tenantId+"']) and n.pattern = 'DATA_ENTRY' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());

            }
        }else{
            if(StringUtils.isEmpty(tenantId)) {
                query1 = session.query("match(n:Activity) where  n.version = '" + version + "' and n.pattern = 'DATA_ENTRY' and n.code in " + baseDataCodeStr + "  return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
            }else{
             //   query1 = session.query("match (te:TenantEntity{tenantId:'" + tenantId + "'})-[:ACTIVITY]-> (n:Activity) where  n.version = te.version and n.pattern = 'DATA_ENTRY' and n.code in " + baseDataCodeStr + "  return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
                query1 = session.query("match(n:Activity) where  n.version ='" + version + "' and (n.tenantId is null or n.tenantId in ['SYSTEM','"+tenantId+"']) and n.pattern = 'DATA_ENTRY' and n.code in " + baseDataCodeStr + "  return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());

            }
        }

        Iterator<Map<String, Object>> iterator1 = query1.iterator();
        while(iterator1.hasNext()){
            Map<String, Object> next = iterator1.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.BaseDataEntry);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));

            Map<String, Map<String, String>> lang = new HashMap<>();
            Map<String, String> nameLang = new HashMap<>();
            if(!ObjectUtils.isEmpty(next.get("name_cn"))){
                nameLang.put("zh_CN",  String.valueOf(next.get("name_cn")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_tw"))){
                nameLang.put("zh_TW",  String.valueOf(next.get("name_tw")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_us"))){
                nameLang.put("en_US",  String.valueOf(next.get("name_us")));
            }
            if(!nameLang.isEmpty()){
                lang.put("name", nameLang);
                componentBO.setLang(lang);
            }

            result.add(componentBO);
        }
        return result;
    }

    public List<ComponentBO> getCommonBaseDataEntryComponent(String version){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();

        Session session = this.neo4jSessionFactory.openSession();
      //  Result query1 = session.query("match (te:TenantEntity{ifCommon:true,version:'" + version + "'})-[:ACTIVITY]-> (n:Activity) where  n.version = te.version and n.pattern = 'DATA_ENTRY'  return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
        Result query1 = session.query("match (n:Activity{commonApp:true})  where  n.version ='" + version + "'  and n.pattern = 'DATA_ENTRY'  return distinct n.code as code, n.name as name, n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());

        Iterator<Map<String, Object>> iterator1 = query1.iterator();
        while(iterator1.hasNext()){
            Map<String, Object> next = iterator1.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.BaseDataEntry);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));

            Map<String, Map<String, String>> lang = new HashMap<>();
            Map<String, String> nameLang = new HashMap<>();
            if(!ObjectUtils.isEmpty(next.get("name_cn"))){
                nameLang.put("zh_CN",  String.valueOf(next.get("name_cn")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_tw"))){
                nameLang.put("zh_TW",  String.valueOf(next.get("name_tw")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_us"))){
                nameLang.put("en_US",  String.valueOf(next.get("name_us")));
            }
            if(!nameLang.isEmpty()){
                lang.put("name", nameLang);
                componentBO.setLang(lang);
            }

            result.add(componentBO);
        }
        return result;
    }


    public List<ComponentBO> getReportComponent(String appNameSpace, String appCode, String version, String tenantId){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();

        String reportCodeStr = null;
        try {
            List<ApplicationRelation> activityList = this.appService.getApplicationRelationByTypeAndAppCodeAndVersion("activity", appCode, version);
            List<String> reportCodeList = activityList.stream().map(ApplicationRelation::getCode).distinct().collect(Collectors.toList());
            if(CollectionUtils.isNotEmpty(reportCodeList)){
                reportCodeStr = "['" + StringUtils.join(reportCodeList, "','") + "']";
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        Session session = this.neo4jSessionFactory.openSession();
        Result query1 = null;
        if(StringUtils.isEmpty(reportCodeStr)){
            if(StringUtils.isEmpty(tenantId)) {
                query1 = session.query("match(n:Activity) where  n.version = '" + version + "' and n.pattern = 'STATEMENT' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.code as code, n.name as name,  n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
            }else{
                //query1 = session.query("match(te:TenantEntity{tenantId:'" + tenantId + "'})-[:ACTIVITY]-> (n:Activity) where  n.version = te.version and n.pattern = 'STATEMENT' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.code as code, n.name as name,  n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
                query1 = session.query("match(n:Activity)  where  n.version ='" + version + "' and (n.tenantId is null or n.tenantId in ['SYSTEM','"+tenantId+"']) and n.pattern = 'STATEMENT' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.code as code, n.name as name,  n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());

            }
        }else{
            if(StringUtils.isEmpty(tenantId)) {
                query1 = session.query("match(n:Activity) where  n.version = '" + version + "' and n.pattern = 'STATEMENT' and n.code in " + reportCodeStr + " return distinct n.code as code, n.name as name,  n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
            }else{
               // query1 = session.query("match(te:TenantEntity{tenantId:'" + tenantId + "'})-[:ACTIVITY]-> (n:Activity) where  n.version = te.version and n.pattern = 'STATEMENT' and n.code in " + reportCodeStr + " return distinct n.code as code, n.name as name,  n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());
                query1 = session.query("match(n:Activity)  where  n.version ='" + version + "' and (n.tenantId is null or n.tenantId in ['SYSTEM','"+tenantId+"']) and n.pattern = 'STATEMENT' and n.code in " + reportCodeStr + " return distinct n.code as code, n.name as name,  n.`lang.name.en_US` as name_us, n.`lang.name.zh_CN` as name_cn, n.`lang.name.zh_TW` as name_tw, n.version as version", new HashMap<>());

            }
        }

        Iterator<Map<String, Object>> iterator1 = query1.iterator();
        while(iterator1.hasNext()){
            Map<String, Object> next = iterator1.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.Report);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));

            Map<String, Map<String, String>> lang = new HashMap<>();
            Map<String, String> nameLang = new HashMap<>();
            if(!ObjectUtils.isEmpty(next.get("name_cn"))){
                nameLang.put("zh_CN",  String.valueOf(next.get("name_cn")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_tw"))){
                nameLang.put("zh_TW",  String.valueOf(next.get("name_tw")));
            }
            if(!ObjectUtils.isEmpty(next.get("name_us"))){
                nameLang.put("en_US",  String.valueOf(next.get("name_us")));
            }
            if(!nameLang.isEmpty()){
                lang.put("name", nameLang);
                componentBO.setLang(lang);
            }

            result.add(componentBO);
        }
        return result;
    }


    public List<ComponentBO> getServiceCompositionComponent(String appNameSpace, String appCode, String version, String tenantId){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        Session session = this.neo4jSessionFactory.openSession();
        Result query1 = null;
        if(StringUtils.isEmpty(tenantId)) {
            query1 = session.query("match(n:StartServiceComposerAction) where n.version = '" + version + "' and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.serviceComposerId as code, n.name as name, n.version as version", new HashMap<>());
        }else{
        //    query1 = session.query("match (te:TenantEntity{tenantId:'" + tenantId + "'})-[:ACTION]-> (n:StartServiceComposerAction) where n.version = te.version and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.serviceComposerId as code, n.name as name, n.version as version", new HashMap<>());
            query1 = session.query("match (n:StartServiceComposerAction) where  n.version ='" + version + "' and (n.tenantId is null or n.tenantId in ['SYSTEM','"+tenantId+"']) and (n.nameSpace = '" + appNameSpace + "' or n.athena_namespace = '" + appCode + "' or n.relation_namespace = '" + appNameSpace + "') return distinct n.serviceComposerId as code, n.name as name, n.version as version", new HashMap<>());

        }
        Iterator<Map<String, Object>> iterator1 = query1.iterator();
        while(iterator1.hasNext()){
            Map<String, Object> next = iterator1.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.ServiceComposition);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));
            result.add(componentBO);
        }

        return result;
    }

    public List<ComponentBO> getCommonServiceCompositionComponent(String version){
        List<ComponentBO> result = new ArrayList<>();
        Set<String> existsCodes = new HashSet<>();
        Session session = this.neo4jSessionFactory.openSession();
     //   Result query1 = session.query("match (te:TenantEntity{ifCommon:true,version:'" + version + "'})-[:ACTION]-> (n:StartServiceComposerAction) where n.version = te.version return distinct n.serviceComposerId as code, n.name as name, n.version as version", new HashMap<>());
        Result query1 = session.query("match (n:StartServiceComposerAction{commonApp:true}) where  n.version ='" + version + "' return distinct n.serviceComposerId as code, n.name as name, n.version as version", new HashMap<>());
        Iterator<Map<String, Object>> iterator1 = query1.iterator();
        while(iterator1.hasNext()){
            Map<String, Object> next = iterator1.next();
            if(existsCodes.contains(String.valueOf(next.get("code")))){
                continue;
            }
            existsCodes.add(String.valueOf(next.get("code")));
            ComponentBO componentBO = new StructuredComponentBO();
            componentBO.setCode(String.valueOf(next.get("code")));
            componentBO.setType(ComponentTypeEnum.ServiceComposition);
            componentBO.setName(String.valueOf(next.get("name")));
            componentBO.setVersion(String.valueOf(next.get("version")));
            result.add(componentBO);
        }

        return result;
    }


    /**
     * @param tenantId
     * @param appCode
     * @param component
     * @return
     *      * {
     *      *   "tenantId":"athenaTestW",
     *      *   "appCode":"PPC",
     *      *   "Project":['p1','p2'],
     *      *   "Task":['t1']
     *      * }
     */
    public Map<String, Object> queryComponent(String tenantId, String appCode, String eventType, List<String> component) throws DWBusinessException {

        Map<String, Object> result = new HashMap<>();
        result.put("tenantId", tenantId);
        result.put("appCode", appCode);
        result.put("eventType", eventType);

        if(CollectionUtils.isEmpty(component)){
            return result;
        }
        //检查是否没有正在生效的应用了
//        boolean allAppExpired = this.tenantService.checkAllAppExpired(tenantId,appCode,eventType);
        boolean allAppExpired = false;

        TenantEntity tenant = this.tenantService.getTenant(tenantId);
        String tenantVersion = null;
        if(tenant == null){ //找不到租户直接返回还是默认个1.0的版本进行后续？？
            tenantVersion = "1.0";
        }else{
            tenantVersion = tenant.getVersion();
        }
        Application application = appService.getApplicationByCode(appCode, tenantId);


        if(application == null){
            for(String str : component){
                result.put(str, new ArrayList<>());
            }
            log.info("queryComponentResult not found appEntity:{}", result);
            return result;
        }

        for(String str : component){
            if(StringUtils.equals(str, ComponentTypeEnum.MonitorRule.toString())){
                List<String> monitorRule = this.getMonitorRule(appCode, tenantVersion, tenantId, allAppExpired);
                result.put(str, monitorRule);
            }else if(StringUtils.equals(str, ComponentTypeEnum.MechanismVariable.toString())){
                List<String> mechanismVariable = this.getMechanismVariable(appCode, tenantVersion, tenantId, allAppExpired);
                result.put(str, mechanismVariable);
            }else if(StringUtils.equals(str, ComponentTypeEnum.Project.toString())){
                List<String> project = this.getProject(appCode, tenantVersion, tenantId, allAppExpired);
                result.put(str, project);
            }else if(StringUtils.equals(str, ComponentTypeEnum.Task.toString())){
                List<String> task = this.getTask(appCode, tenantVersion, tenantId, allAppExpired);
                result.put(str, task);
            }else if(StringUtils.equals(str, ComponentTypeEnum.BaseDataEntry.toString())){
                List<String> baseDataEntry = this.getBaseDataEntry(appCode, tenantVersion, tenantId, allAppExpired);
                result.put(str, baseDataEntry);
            }else if(StringUtils.equals(str, ComponentTypeEnum.Report.toString())){
                List<String> report = this.getReport(appCode, tenantVersion, tenantId, allAppExpired);
                result.put(str, report);
            }else if(StringUtils.equals(str, ComponentTypeEnum.ServiceComposition.toString())){
                List<String> serviceComposition = this.getServiceComposition(appCode, appCode, tenantVersion, tenantId, allAppExpired);
                result.put(str, serviceComposition);
            }else{
                result.put(str, new ArrayList<>());
            }
        }

        log.info("queryComponentResult:{}", result);
        return result;
    }

    private List<String> getServiceComposition(String appNameSpace, String appCode, String version, String tenantId, boolean allAppExpired) {
        List<ComponentBO> appComponent = this.getServiceCompositionComponent(appNameSpace, appCode, version, tenantId);
//        if(allAppExpired){
//            List<ComponentBO> commonComponent = this.getCommonServiceCompositionComponent(version);
//            if(CollectionUtils.isNotEmpty(commonComponent)){
//                appComponent.addAll(commonComponent);
//            }
//        }
        List<String> collect = appComponent.stream().map(ComponentBO::getCode).collect(Collectors.toList());
        return collect;
    }

    private List<String> getReport(String appCode, String version, String tenantId, boolean allAppExpired) throws DWBusinessException {
        List<ComponentBO> appComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.Report);
        //不存在公共报表
        /*if(allAppExpired){
            List<ComponentBO> commonComponent = this.getReportComponent("common", "common", version, tenantId);
            if(CollectionUtils.isNotEmpty(commonComponent)){
                appComponent.addAll(commonComponent);
            }
        }*/
        List<String> collect = appComponent.stream().map(ComponentBO::getCode).collect(Collectors.toList());
        return collect;
    }

    private List<String> getBaseDataEntry(String appCode, String version, String tenantId, boolean allAppExpired) throws DWBusinessException {
        List<ComponentBO> appComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.BaseDataEntry);
//        if(allAppExpired){
//            List<ComponentBO> commonComponent = getCommonComponentList(version, tenantId, ComponentTypeEnum.BaseDataEntry);
//            if(CollectionUtils.isNotEmpty(commonComponent)){
//                appComponent.addAll(commonComponent);
//            }
//        }
        List<String> collect = appComponent.stream().map(ComponentBO::getCode).collect(Collectors.toList());
        return collect;
    }

    private List<String> getTask(String appCode, String version, String tenantId, boolean allAppExpired) throws DWBusinessException {
        List<ComponentBO> appComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.Task);
        //不存在除基础资料以外的activity
        /*if(allAppExpired){
            List<ComponentBO> commonComponent = this.getTaskComponent("common", "common", version, tenantId);
            if(CollectionUtils.isNotEmpty(commonComponent)){
                appComponent.addAll(commonComponent);
            }
        }*/
        List<String> collect = appComponent.stream().map(ComponentBO::getCode).collect(Collectors.toList());
        return collect;
    }

    private List<String> getMonitorRule(String appCode, String version, String tenantId, boolean allAppExpired) throws DWBusinessException {
        List<ComponentBO> appComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.MonitorRule);
//        if(allAppExpired){
//            List<ComponentBO> commonComponent = getCommonComponentList(version, tenantId, ComponentTypeEnum.MonitorRule);
//            if(CollectionUtils.isNotEmpty(commonComponent)){
//                appComponent.addAll(commonComponent);
//            }
//        }
        List<String> collect = appComponent.stream().map(ComponentBO::getCode).collect(Collectors.toList());
        return collect;
    }


    private List<String> getMechanismVariable(String appCode, String version, String tenantId, boolean allAppExpired) throws DWBusinessException {
        List<ComponentBO> appComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.MechanismVariable);
//        if(allAppExpired){
//            List<ComponentBO> commonComponent = getCommonComponentList(version, tenantId, ComponentTypeEnum.MechanismVariable);
//            if(CollectionUtils.isNotEmpty(commonComponent)){
//                appComponent.addAll(commonComponent);
//            }
//        }
        List<String> collect = appComponent.stream().map(ComponentBO::getCode).collect(Collectors.toList());
        return collect;
    }

    private List<String> getProject(String appCode, String version, String tenantId, boolean allAppExpired) throws DWBusinessException {
        List<ComponentBO> appComponent = getComponentList(Collections.singletonList(appCode), version, tenantId, ComponentTypeEnum.Project);
        //不存在common project
        /*if(allAppExpired){
            List<ComponentBO> commonComponent = this.getProjectComponent("common", "common", version, tenantId);
            if(CollectionUtils.isNotEmpty(commonComponent)){
                appComponent.addAll(commonComponent);
            }
        }*/
        List<String> collect = appComponent.stream().map(ComponentBO::getCode).collect(Collectors.toList());
        return collect;
    }


    public void updateApplicationComponentList(String appCode, String version, String tenantId) throws DWBusinessException{
        Criteria appcriteria = new Criteria();
        appcriteria.and("code").is(appCode).and("version").is(version);
        Application application = dataPickService.findOneByCondition(appcriteria, tenantId, Application.class, "application");
        if (null != application && null != application.getCommonApp() && application.getCommonApp()) {
            return;
        }
        Boolean individualAll = Optional.ofNullable(application).map(Application::getIndividualAll).orElse(false);
        //公共应用无需初始化 开发平台调用我们跳过
        if (appCode.equalsIgnoreCase("common")) {
            return;
        }
        List<ComponentBO> result = this.getAllComponentListByApplication(appCode, appCode, version, tenantId);

        //查询下应用下有哪些范式
        List<ApplicationRelation> applicationRelationList = dataPickService.applicationRelations(appCode, "paradigm", null, tenantId, version);
        if(CollectionUtils.isNotEmpty(applicationRelationList)){ // 当找到范式
            Set<String> paradigmCodeList = applicationRelationList.stream().map(ApplicationRelation::getCode).collect(Collectors.toSet());
            // 先暂存下已存在的不同范式的组件信息
            Map<String, Set<String>> existsParadigmAppComponent = this.getExistsParadigmAppComponent(appCode, version, individualAll ? tenantId : "SYSTEM");

            //新增或更新应用组件清单
            for(String paradigmCode : paradigmCodeList){
                log.info("updateApplicationComponentList paradigmCode:{}", paradigmCode);
                Query query1 = new Query();
                Criteria criteria = Criteria.where(ComponentManager.APP_CODE).is(appCode)
                        .and(ComponentManager.PARADIGM).is(paradigmCode)
                        .and(ComponentManager.TYPE).is(ComponentStructuredListType.BaseFunction).and("version").is(version).and(ComponentManager.TENANT_ID).is(individualAll ? tenantId : "SYSTEM");
                query1.addCriteria(criteria);
                ComponentStructuredList componentStructuredList = mongoTemplateSystem.findOne(Query.query(criteria), ComponentStructuredList.class, "componentStructuredList");
                if(componentStructuredList == null){
                    Update update = new Update();
                    update.set("components", result);
                    update.set("structureVersion", "1.0");
                    update.set("application", appCode);
                    update.set("tenantId", individualAll ? tenantId : "SYSTEM");
                    this.mongoTemplateSystem.upsert(query1, update, "componentStructuredList");
                }else{
                    //对比下多出哪些组件，更新到应用清单
                    List<ComponentBO> extraComponet = this.componentManager.getExtraComponent(result, componentStructuredList.getComponents(),
                            existsParadigmAppComponent, paradigmCode);
                    if(CollectionUtils.isNotEmpty(extraComponet)) {
                        log.info("updateApplicationComponentList extraComponent size:{}", extraComponet.size());
                        Update update = new Update();
                        Update.AddToSetBuilder add = update.new AddToSetBuilder("components");
                        update = add.each(extraComponet);
                        this.mongoTemplateSystem.upsert(query1, update, "componentStructuredList");
                    }
                }
            }
            // 更新组户级组件清单
            updateTenantApplicationComponentList(appCode, version, tenantId, result, individualAll);
        }

    }

    private void updateTenantApplicationComponentList(String appCode, String version, String tenantId, List<ComponentBO> result, Boolean individualAll) throws DWBusinessException {
        Query tenantComponentQuery = new Query();
        Criteria criteria = Criteria.where(ComponentManager.APP_CODE).is(appCode);
        // 有租户id代表是个案级应用的更新，只查指定的租户数据
        if (StringUtils.isNotEmpty(tenantId)) {
            criteria.and(ComponentManager.TENANT_ID).is(tenantId);
        }
        tenantComponentQuery.addCriteria(criteria);
        // 组户级组件清单
        List<TenantAppComponentListBO> tenantAppComponentListBOS = this.mongoTemplateUser.find(tenantComponentQuery, TenantAppComponentListBO.class);

        if(CollectionUtils.isNotEmpty(tenantAppComponentListBOS)){
            for(TenantAppComponentListBO tenantAppComponentListBO : tenantAppComponentListBOS){
                String tenantVersion = null;
                try {
                    tenantVersion = kgInnerService.getTenantVersion(tenantAppComponentListBO.getTenantId());
                } catch (Exception e) {
                    log.error("租户查不到版本{}", tenantAppComponentListBO.getTenantId());
                }
                //不是入参版本的租户不参与数据刷新
                if (!version.equals(tenantVersion)) {
                    continue;
                }
                List<ComponentBO> tenantAppComponent = new ArrayList<>();
                log.info("update TenantAppComponentListBO tenantId:{}", tenantAppComponentListBO.getTenantId());
                Query tenantParadigmQuery = new Query();
                tenantParadigmQuery.addCriteria(Criteria.where("code").is(appCode).and("tenantId").is(tenantAppComponentListBO.getTenantId()));
                // 租户启用的范式和机制
                MechanismTenantParadigm mechanismTenantParadigm = this.mongoTemplateUser.findOne(tenantParadigmQuery, MechanismTenantParadigm.class, "mechanismTenantParadigm");
                if(mechanismTenantParadigm != null){
                    Map<String, List<String>> tenantParadigmMap = this.componentManager.getTenantParadigm(mechanismTenantParadigm, version, tenantId);
                    if(tenantParadigmMap != null){
                        Set<Map.Entry<String, List<String>>> entries = tenantParadigmMap.entrySet();
                        for(Map.Entry<String, List<String>> entry : entries){
                            String paradigm = entry.getKey();
                            TenantAppComponentListBO tenantAppComponentListBO1 = this.componentManager.GetTenantUsingApplicationComponentList(appCode, individualAll ? tenantAppComponentListBO.getTenantId() : "SYSTEM", paradigm, entry.getValue(),version);
                            if(CollectionUtils.isNotEmpty(tenantAppComponentListBO1.getComponents())){
                                tenantAppComponent.addAll(tenantAppComponentListBO1.getComponents());
                            }
                        }
                    }else{
                        tenantAppComponent.addAll(result);
                    }
                }else{
                    tenantAppComponent.addAll(result);
                }

                this.compareAndUpdate(tenantAppComponentListBO, tenantAppComponent);

            }
        }
    }



    /**
     * 根据应用code查询系统级组件清单表中的范式组件数据，并按照范式分组返回
     * @param appCode 应用code
     * @param version 版本号
     * @return 返回范式组件数据 {范式code,[组件类型#组件code]}
     */
    private Map<String, Set<String>> getExistsParadigmAppComponent(String appCode, String version, String tenantId) throws DWBusinessException {
        Map<String, Set<String>> result = new HashMap<>();
        Criteria criteria = Criteria.where(ComponentManager.APP_CODE).is(appCode)
                .and(ComponentManager.TYPE).is(ComponentStructuredListType.BaseFunction).and("version")
                .is(version).and(ComponentManager.TENANT_ID).is(tenantId);
        List<ComponentStructuredList> componentStructuredListBOS = mongoTemplateSystem.find(Query.query(criteria), ComponentStructuredList.class, "componentStructuredList");


        if(CollectionUtils.isEmpty(componentStructuredListBOS)){
            return result;
        }

        for(ComponentStructuredList componentStructuredListBO : componentStructuredListBOS){
            List<ComponentBO> mergedList = new ArrayList<ComponentBO>();
            this.componentManager.Merge(mergedList,componentStructuredListBO.getComponents(), new ArrayList());
            Set<String> collect = mergedList.stream().map(e -> e.getType().toString() + "#" + e.getCode()).collect(Collectors.toSet());
            result.put(componentStructuredListBO.getParadigm(), collect);
        }
        return result;
    }

    /**
     * 将待处理的组件数据添加与现有数据做对比，并新增或者更新
     * @param tenantAppComponentListBO 已存在租户应用组件清单数据
     * @param tenantAppComponent 待处理组件数据
     */
    private void compareAndUpdate(TenantAppComponentListBO tenantAppComponentListBO, List<ComponentBO> tenantAppComponent){
        if(CollectionUtils.isEmpty(tenantAppComponent)){
            return;
        }
        List<ComponentBO> components = tenantAppComponentListBO.getComponents();
        if(CollectionUtils.isEmpty(components)){
            tenantAppComponentListBO.setComponents(tenantAppComponent);
        }else{
            Set<String> existsSet = components.stream().map(componentBO -> componentBO.getType().toString() + "#" + componentBO.getCode()).collect(Collectors.toSet());
            Set<String> tenantAppComponentSet = tenantAppComponent.stream().map(componentBO -> componentBO.getType().toString() + "#" + componentBO.getCode()).collect(Collectors.toSet());
            List<ComponentBO> toAdd = new ArrayList<>();
            for(ComponentBO componentBO : tenantAppComponent){
                if(!existsSet.contains(componentBO.getType().toString() + "#" + componentBO.getCode())){
                    existsSet.add(componentBO.getType().toString() + "#" + componentBO.getCode());
                    toAdd.add(componentBO);
                }
            }
            components.addAll(toAdd);
            components.removeIf(v -> !tenantAppComponentSet.contains(v.getType().toString() + "#" + v.getCode()));
            tenantAppComponentListBO.setComponents(components);
        }
        this.mongoTemplateUser.save(tenantAppComponentListBO);
    }


    public void updateApplicationComponentList(String appCode, List<String> tenantIds, String version) throws DWBusinessException {
        CountDownLatch latch = new CountDownLatch(tenantIds.size());
        List<String> errorMessage = new ArrayList<>();
        for (String tenantId : tenantIds) {
            executor.submit(() -> {
                try {
                    log.info("updateApplicationComponentList tenantId:{}", tenantId);
                    this.updateApplicationComponentList(appCode, version, tenantId);
                } catch (DWBusinessException e) {
                    log.error("updateApplicationComponentList error", e);
                    errorMessage.add("个案应用更新组件清单失败，租户：" + tenantId + "，异常信息：" + e.getMessage());
                }finally {
                    latch.countDown();
                }
            });
        }
        try {
            latch.await();
        } catch (InterruptedException e) {
            // 恢复线程的中断状态
            Thread.currentThread().interrupt();
            // 处理中断情况
            log.error("更新个案组件线程中断");
            // 抛出一个更具体的异常
            throw new DWBusinessException("更新个案组件线程中断");
        }
        if (CollectionUtils.isNotEmpty(errorMessage)) {
            throw new DWBusinessException(Joiner.on("\\n").join(errorMessage));
        }
    }

}
