package com.digiwin.athena.knowledgegraph.service;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWArgumentException;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.AllowAnonymous;
import com.digiwin.app.service.DWService;
import com.digiwin.athena.annotation.ScheduledCall;
import com.digiwin.athena.knowledgegraph.domain.TenantEntity;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.knowledgegraph.domain.valueobject.VirtualTenantParams;
import com.digiwin.athena.knowledgegraph.dto.TenantAppNoticeDTO;
import com.digiwin.athena.knowledgegraph.po.InitMonitorRuleParam;
import com.digiwin.athena.knowledgegraph.po.TenantInitParam;
import com.digiwin.athena.kg.report.hz.model.sence.BigScreenViewDTO;

import java.util.List;
import java.util.Map;

/**
 * 租户服务
 * @javadoc
 */
public interface ITenantService extends DWService {
    /**
     * 获取所有租户
     * @return
     */
    @AllowAnonymous
    List<TenantEntity> getAllTenant();

    /**
     * 根据租户id获取租户
     * @param tenantId
     * @return
     */
    @AllowAnonymous
    @ScheduledCall
    TenantEntity getTenant(String tenantId);

    /**
     * 根据租户id获取租户版本
     * @param tenantId 租户id
     * @return
     */
    String getTenantVersion(String tenantId);

    /**
     * 清除租户数据
     * @param param 参数
     * @return
     * @throws DWBusinessException
     */
    Object postCleanTenantInfo(TenantInitParam param) throws DWBusinessException;

    /**
     * 根据租户id获取授权的appCode集合
     * @return
     * @throws DWBusinessException
     */
    List<String> getAppCodes() throws DWBusinessException;

    /**
     * 根据租户id获取授权的应用
     * @return
     * @throws DWBusinessException
     */
    Object getApps() throws DWBusinessException;

    /**
     * 获取所有项目任务
     * @return
     * @throws Exception
     */
    Map<String,Object> getAllTaskAndActivity() throws Exception;

    /**
     * 将虚拟租户链接到节点上
     * @param virtualTenantParams
     * @return
     * @throws DWArgumentException
     */
    @AllowAnonymous
    Object postConnectVirtualTenantToNodes(VirtualTenantParams virtualTenantParams) throws DWBusinessException;

    /**
     * 删除虚拟租户
     * @param virtualTenantParams
     * @return
     * @throws DWArgumentException
     */
    Object postDeleteVirtualTenantToNodes(VirtualTenantParams virtualTenantParams) throws DWBusinessException;

    /**
     * 初始化租户信息
     * @param param
     * @return
     * @throws DWBusinessException
     */
    @AllowAnonymous
    @ScheduledCall
    Object postInitTenantInfoV2(TenantInitParam param) throws DWBusinessException;

    /**
     * 针对存在的租户初始化侦测规则
     * @param param
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postInitMonitorRuleForExistsTenant(InitMonitorRuleParam param) throws Exception;

    /**
     * 租户过期处理
     * @param param
     * @return
     * @throws DWBusinessException
     */
    @AllowAnonymous
    Object postHandleTenantAppExpire(TenantInitParam param) throws DWBusinessException;

    /**
     * 租户续约
     * @param param
     * @return
     * @throws DWBusinessException
     */
    @AllowAnonymous
    Object postAppRenewNotice(TenantInitParam param) throws DWBusinessException;

    /**
     * 调用aim续约接口
     * @param tenantId 租户id
     * @param appCode 应用Code
     */
    @AllowAnonymous
    Object postAppAddPurchaseNotice(String tenantId ,String appCode) throws DWBusinessException;

    /**
     * 租户过期应用数据处理
     */
    @ScheduledCall
    @AllowAnonymous
    void postHandleTenantAppExpire();

    /**
     * 租户过期发送消息
     * @param param
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postTenantExpiredNotice(TenantAppNoticeDTO param) throws Exception;

    /**
     * 租户保留期消息提醒
     * @param param
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postTenantAppMaterialRemainNotice(TenantAppNoticeDTO param) throws Exception;

    /**
     * 租户续约发送消息
     * @param param 参数
     * @return
     * @throws Exception
     */
    @AllowAnonymous
    Object postTenantAppRenewNotice(TenantAppNoticeDTO param) throws Exception;

    /**
     * 租户应用保留期时数据处理
     */
    @AllowAnonymous
    @ScheduledCall
    void postHandleTenantAppExpireRetainDaysAfter();

    /**
     * 获取用户完整性检查
     * @return
     * @throws DWBusinessException
     */
    Object getUserIntegrityCheck() throws DWBusinessException ;

    /**
     * 获取租户用户完整性检查
     * @throws DWBusinessException
     */
    @AllowAnonymous
    @ScheduledCall
    void getTenantUsersIntegrityCheck() throws DWBusinessException;

    /**
     * mock，后期删除
     * @param param
     * @throws DWBusinessException
     */
    @AllowAnonymous
    void postMockNoticeSuperAdminUserIntegrityCheckResult(JSONObject param) throws DWBusinessException;

    /**
     * 语义获取租户购买的应用信息和租户版本信息
     * @return
     * @throws DWBusinessException
     */
    Object getAppsInfo() throws DWBusinessException;

    /**
     * 获取用户授权应用
     * @return
     * @throws DWBusinessException
     */
    Object getUserAppAuth() throws DWBusinessException;

    /**
     * 获取用户授权应用 ,传userId,解决集成token对应userId错误的问题
     * @param userId 用户id
     * @return
     * @throws DWBusinessException
     */
    Object getUserAppAuth(String userId) throws DWBusinessException;

    /**
     * 用户粒度查询授权应用
     * @return
     * @throws DWBusinessException
     */
    List<Application> getUserMetricApplication() throws DWBusinessException;

    /**
     * 用户粒度查询授权应用
     * @param type 类型，类型是1代表查敏捷数据应用
     * @return
     * @throws DWBusinessException
     */
    List<Application> getUserMetricApplication(String type) throws DWBusinessException;

    /**
     * 用户粒度查询授权应用
     * @param type 类型，类型是1代表查敏捷数据应用
     * @param userId 用户id
     * @return
     * @throws DWBusinessException
     */
    List<Application> getUserMetricApplication(String type,String userId) throws DWBusinessException;

    /**
     * 获取租户指标应用数据 - 敏数
     * @return
     * @throws DWBusinessException
     */
    List<Application> getTenantMetricApplication() throws DWBusinessException;

    /**
     * 根据租户查询场景
     * @return
     * @throws Exception
     */
    List<BigScreenViewDTO> getQuerySceneByTenant() throws Exception;

    /**
     * 获取系统级与租户级服务配置
     * @param tenantId      租户ID
     * @param serviceName   服务名称
     * @return  Map
     * @throws DWBusinessException
     */
    @AllowAnonymous
    Object getSystemTenantServiceConfig(String tenantId, String serviceName) throws DWBusinessException;

    /**
     * 获取用户权限下所有场景
     * @param codes  场景编码
     * @return  List
     * @throws DWBusinessException
     * @throws NoSuchFieldException
     * @throws IllegalAccessException
     */
    Object postQueryPermissionSceneByCode(List<String> codes) throws DWBusinessException, NoSuchFieldException, IllegalAccessException;


    Object postQueryPermissionSceneByCode(List<String> codes,String userId) throws DWBusinessException, NoSuchFieldException, IllegalAccessException;
    /**
     * 获取用户权限下所有场景
     * @param userId 用户id
     * @param appCodes 应用code
     * @param codes 场景编码
     * @return
     * @throws DWBusinessException
     */
    Object postQueryPermissionSceneByCode(String userId,List<String> appCodes,List<String> codes) throws DWBusinessException;

    /**
     * 获取应用对应的产品线
     * @param appCodes  应用编码
     * @return  List
     * @throws DWBusinessException
     */
    Object postQueryProductLineByApp(List<String> appCodes) throws DWBusinessException;
}