package com.digiwin.athena.knowledgegraph.service.impl;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.knowledgegraph.clients.CACUtils;
import com.digiwin.athena.knowledgegraph.dto.report.dataCollect.DataCollectConfig;
import com.digiwin.athena.knowledgegraph.dto.report.dataCollect.DataCollectQuestions;
import com.digiwin.athena.kg.report.hz.model.dataCollectGroup.DataCollectGroupConfig;
import com.digiwin.athena.kg.report.hz.model.dataCollectGroup.DelConfigDTO;
import com.digiwin.athena.knowledgegraph.service.IDataCollectGroupService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.DataSubscriptionUtils;
import io.seata.common.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
@MyExceptionHandler
public class DataCollectGroupService implements IDataCollectGroupService {
    @Autowired
    KgInnerService kgInnerService;

    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplate;

    @Autowired
    @Qualifier("knowledgegraphTenant")
    MongoTemplate mongoTemplateUser;

    @Autowired
    DataCollectService dataCollectService;

    @Autowired
    SceneService sceneService;

    @Autowired
    CACUtils cacUtils;

    public static final String FAIL_MSG = "fail";

    public static final String SUCCESS_MSG = "success";

    @Override
    public Object postAddDataCollectGroupConfig(List<DataCollectGroupConfig> configs) {
        if (CollectionUtils.isEmpty(configs)) {
            return FAIL_MSG;
        }

        BulkOperations bulkOperations = mongoTemplateUser.bulkOps(BulkOperations.BulkMode.UNORDERED, DataCollectGroupConfig.class);
        for (DataCollectGroupConfig config : configs) {
            if (StringUtils.isEmpty(config.getGroupId())) {
                config.setGroupId("group_" + UUID.randomUUID().toString().replace("-", ""));
            }
            Query query = Query.query(Criteria
                    .where("tenantId").is(config.getTenantId())
                    .and("groupId").is(config.getGroupId())
                    .and("userId").is(config.getUserId())
                    .and("type").is(config.getType()));

            Update update = new Update()
                    .set("orderId", config.getOrderId())
                    .set("groupValue", config.getGroupValue())
                    .set("asaCode", config.getAsaCode())
                    .set("type", config.getType())
                    .set("userName", config.getUserName());

            bulkOperations.upsert(query, update);
        }

        bulkOperations.execute();
        return SUCCESS_MSG;
    }

    @Override
    public Object postRemoveDataCollectGroupConfig(DelConfigDTO delConfig) throws Exception {
        if (CollectionUtils.isEmpty(delConfig.getGroupIds())) {
            return FAIL_MSG;
        }
        this.mongoTemplateUser.remove(Query.query(Criteria.where("groupId").in(delConfig.getGroupIds())), DataCollectGroupConfig.class);
        dataCollectService.postRemoveDataCollectConfig(delConfig.getAsaCode(), delConfig.getGroupIds());
        return DataSubscriptionUtils.SUCCESS_MSG;
    }

    @Override
    public Object postQueryDataCollectGroupConfig(DataCollectGroupConfig config) throws Exception {
        Query query = queryCriteria(config);
        Map<String, Object> result = new HashMap<>();
        List<DataCollectGroupConfig> groupList = this.mongoTemplateUser.find(query, DataCollectGroupConfig.class);
        if (CollectionUtils.isEmpty(groupList) && StringUtils.isEmpty(config.getGroupValue())) {
            DataCollectGroupConfig dataCollectGroupConfig = getDataCollectGroupConfig(config);
            groupList.add(dataCollectGroupConfig);
            postAddDataCollectGroupConfig(groupList);
            assignGroupIdToCollectQuestion(config, dataCollectGroupConfig);
        }

        Map<String, List<DataCollectQuestions>> collectData = getDataCollectQuestionsConfig(config);
        result.put("groupData", groupList);
        result.put("collectData", collectData);
        result.put("sysData",getSysGroupData());
        return sceneService.getTranslateContent(result);
    }

    private Object getSysGroupData() throws DWBusinessException {
        String userId = AthenaUtils.getUserId();
        if (org.apache.commons.lang3.StringUtils.isEmpty(userId)) {
            throw new DWBusinessException("userId is empty");
        }

        // 获取用户对应的应用列表
        List<String> uniqueAppCodes = cacUtils.getAppAuth(userId);

        if (CollectionUtils.isEmpty(uniqueAppCodes)) {
            return Collections.emptyList();
        }

        String tenantVersion = kgInnerService.currentTenantVersion();
        Query query = Query.query(Criteria.where("appCode").in(uniqueAppCodes).and("version").is(tenantVersion));
//        query.with(Sort.by(Sort.Direction.ASC, "orderId"));

        return mongoTemplate.find(query,Object.class,"agiledata_board_info");
    }

    private Query queryCriteria(DataCollectGroupConfig config) throws DWBusinessException {
        String userId = AthenaUtils.getUserId();
        String tenantId = AthenaUtils.getTenantId();

        Query query = Query.query(Criteria.where("tenantId").is(tenantId)
                .and("userId").is(userId));

        if (StringUtils.isNotEmpty(config.getGroupId())) {
            query.addCriteria(Criteria.where("groupId").is(config.getGroupId()));
        }

        if (StringUtils.isNotEmpty(config.getType())) {
            query.addCriteria(Criteria.where("type").is(config.getType()));
        }

        if (StringUtils.isNotEmpty(config.getGroupValue())) {
            query.addCriteria(Criteria.where("groupValue").is(config.getGroupValue()));
        }

        // 兼容敏数融合
        if (StringUtils.isNotEmpty(config.getAsaCode())) {
            query.addCriteria(Criteria.where("asaCode").is(config.getAsaCode()));
        } else {
            Criteria criteria = new Criteria().orOperator(
                    Criteria.where("asaCode").is(""),
                    Criteria.where("asaCode").is(null));
            query.addCriteria(criteria);
        }
        query.with(Sort.by(Sort.Direction.ASC, "orderId"));
        return query;
    }

    private DataCollectGroupConfig getDataCollectGroupConfig(DataCollectGroupConfig config) {
        DataCollectGroupConfig groupConfig = new DataCollectGroupConfig();
        groupConfig.setGroupId("group_" + UUID.randomUUID().toString().replace("-", ""));
        groupConfig.setOrderId(1);
        groupConfig.setTenantId(StringUtils.isEmpty(config.getTenantId()) ? DWServiceContext.getContext().getProfile().get("tenantId").toString() : config.getTenantId());
        groupConfig.setUserId(StringUtils.isEmpty(config.getUserId()) ? DWServiceContext.getContext().getProfile().get("userId").toString() : config.getUserId());
        groupConfig.setUserName(StringUtils.isEmpty(config.getUserName()) ? DWServiceContext.getContext().getProfile().get("userName").toString() : config.getUserName());
        groupConfig.setGroupValue("默认");
        groupConfig.setType("user");
        groupConfig.setAsaCode(config.getAsaCode());
        return groupConfig;
    }

    private void assignGroupIdToCollectQuestion(DataCollectGroupConfig config, DataCollectGroupConfig dataCollectGroupConfig) throws Exception {
        List<DataCollectConfig> dataCollectConfigs;
        if (StringUtils.isEmpty(config.getAsaCode())) {
            dataCollectConfigs = dataCollectService.getQueryDataCollectConfig();
        } else {
            dataCollectConfigs = dataCollectService.getQueryDataCollectConfig(config.getAsaCode());
        }

        if (CollectionUtils.isEmpty(dataCollectConfigs)) {
            return;
        }

        DataCollectConfig dataCollectConfig = dataCollectConfigs.get(0);
        List<DataCollectQuestions> dataCollectQuestions = dataCollectConfig.getDataCollectQuestions();
        if (!CollectionUtils.isEmpty(dataCollectQuestions)) {
            dataCollectQuestions.forEach(dataCollectQuestion -> {
                dataCollectQuestion.setGroupId(dataCollectGroupConfig.getGroupId());
                dataCollectQuestion.setGroupValue(dataCollectGroupConfig.getGroupValue());
            });
            dataCollectService.updateDataCollectConfig(dataCollectConfig.getBoardId(), dataCollectConfig, dataCollectQuestions);
        }
    }

    public Map<String, List<DataCollectQuestions>> getDataCollectQuestionsConfig(DataCollectGroupConfig config) throws Exception {
        List<DataCollectConfig> dataCollectConfigs = StringUtils.isNotEmpty(config.getAsaCode())
                ? dataCollectService.getQueryDataCollectConfig(config.getAsaCode())
                : dataCollectService.getQueryDataCollectConfig();

        if (CollectionUtils.isEmpty(dataCollectConfigs)) {
            return Collections.emptyMap();
        }

        List<DataCollectQuestions> dataCollectQuestions = dataCollectConfigs.get(0).getDataCollectQuestions();

        if (CollectionUtils.isEmpty(dataCollectQuestions)) {
            return Collections.emptyMap();
        }
        return dataCollectQuestions.stream().collect(Collectors.groupingBy(DataCollectQuestions::getGroupId));
    }
}