package com.digiwin.athena.knowledgegraph.service.impl;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.knowledgegraph.clients.EocService;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.knowledgegraph.dto.IMMessageDTO;
import com.digiwin.athena.knowledgegraph.dto.IMMessageTemplateDTO;
import com.digiwin.athena.knowledgegraph.dto.TenantAppExpireDTO;
import com.digiwin.athena.knowledgegraph.service.IIMService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.LanguageUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.text.SimpleDateFormat;
import java.util.*;

@Lang
@Slf4j
@Service
public class IMService implements IIMService {

    @Autowired
    @Qualifier("aimRestTemplate")
    RestTemplate restTemplate;

    @Value("${aimUrl}")
    private String aimUrl;

    @Autowired
    EocService eocService;

    @Autowired
    AppService appService;

    @Autowired
    IamService iamService;

    private Integer bufferDays = 30;

    private Integer expiredDays = 60;

    @Override
    public void noticeAppRenew(String tenantId, String appCode) throws Exception {
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("tenantId", tenantId);
        requestMap.put("appCode", appCode);
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        headers.add("token", iamService.getTenantToken(tenantId));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        HttpEntity<Map> requestEntity = new HttpEntity<>(requestMap, headers);
        ResponseEntity<Map> result = restTemplate.exchange(aimUrl + "/api/aim/v2/expire/renewal", HttpMethod.POST, requestEntity, Map.class);
        if (!(result.getStatusCodeValue() == 200)) {
            log.error("notice app [renew] failed for tenant:{} appcode:{}", tenantId, appCode);
        }
    }

    @Override
    public void noticeAppRestricted(String tenantId, String appCode) throws Exception {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        String tenantToken = iamService.getTenantToken(tenantId);
        headers.add("token", tenantToken);
        TenantAppExpireDTO tenantAppExpireDTO = buildAppMessage(tenantId, appCode, "buffer_period", tenantToken);
        HttpEntity<TenantAppExpireDTO> requestEntity = new HttpEntity<>(tenantAppExpireDTO, headers);
        ResponseEntity<Map> result = restTemplate.exchange(aimUrl + "/api/aim/v2/expire/message", HttpMethod.POST, requestEntity, Map.class);
        if (!(result.getStatusCodeValue() == 200)) {
            log.error("notice app [restricted] failed for tenant:{} appcode:{}", tenantId, appCode);
        }
    }

    /**
     * @return
     * @Author zhuangli
     * @Description 构建缓冲期和资料保留期通知消息对象
     * @Date 10:05 2023/9/28
     * @Param
     **/
    private TenantAppExpireDTO buildAppMessage(String tenantId, String appCode, String type, String token) throws DWBusinessException, NoSuchFieldException, IllegalAccessException {
        TenantAppExpireDTO tenantAppExpireDTO = new TenantAppExpireDTO();
        tenantAppExpireDTO.setTenantId(tenantId);
        Set<String> userIds = eocService.appHasUser(appCode, tenantId, token);
        if (CollectionUtils.isEmpty(userIds)) {
            log.warn("tenant-[{}] app-[{}] has no auth user", tenantId, appCode);
            return null;
        }
        tenantAppExpireDTO.setUserIdList(userIds);
        tenantAppExpireDTO.setMessage(buildAppRestrictedMessageDTO(tenantId, appCode, type));
        return tenantAppExpireDTO;
    }

    private Date calculateBufferStartDay() {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, - 1);
        calendar.set(Calendar.HOUR_OF_DAY, 0); // 将小时设为0
        calendar.set(Calendar.MINUTE, 0); // 将分钟设为0
        calendar.set(Calendar.SECOND, 0); // 将秒设为0
        calendar.set(Calendar.MILLISECOND, 0); // 将毫秒设为0
        return calendar.getTime();
    }

    private Date calculateExpireStartDay() {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR_OF_DAY, 0); // 将小时设为0
        calendar.set(Calendar.MINUTE, 0); // 将分钟设为0
        calendar.set(Calendar.SECOND, 0); // 将秒设为0
        calendar.set(Calendar.MILLISECOND, 0); // 将毫秒设为0
        return calendar.getTime();
    }

    private IMMessageDTO buildAppRestrictedMessageDTO(String tenantId, String appCode, String type) throws DWBusinessException, NoSuchFieldException, IllegalAccessException {
        IMMessageDTO imMessageDTO = new IMMessageDTO();
        imMessageDTO.setAppCode(appCode);
        imMessageDTO.setType(type);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        switch (type) {
            case "buffer_period":
                Date bufferStartDay = calculateBufferStartDay();
                imMessageDTO.setEndTime(sdf.format(calculateEndDay(bufferStartDay, bufferDays)));
                imMessageDTO.setTemplate(buildTemplate(appCode, type, bufferStartDay, tenantId));
                break;
            case "data_retention":
                Date expireStartDay = calculateExpireStartDay();
                imMessageDTO.setEndTime(sdf.format(calculateEndDay(expireStartDay, expiredDays)));
                imMessageDTO.setTemplate(buildTemplate(appCode, type, expireStartDay, tenantId));
                break;
            default:
                throw new DWBusinessException("unsupported message type");
        }
        return imMessageDTO;
    }

    private IMMessageTemplateDTO buildTemplate(String appCode, String type, Date startDay, String tenantId) throws DWBusinessException, NoSuchFieldException, IllegalAccessException {
        IMMessageTemplateDTO imMessageTemplateDTO = new IMMessageTemplateDTO();
        List<Map> variableList = new ArrayList<>();
        Map<String, String> var1 = new HashMap<>();
        Application application = appService.getApplicationByCode(appCode, tenantId);
        LanguageUtil.processLocaleLanguage(application, AthenaUtils.getCurrentLocale());
        var1.put("key", "app_name");
        var1.put("value", application.getName());
        variableList.add(var1);
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        Map<String, String> var2 = new HashMap<>();
        var2.put("key", "expire_date");
        var2.put("value", sdf.format(startDay));
        variableList.add(var2);
        Map<String, String> var3 = new HashMap<>();
        switch (type) {
            case "buffer_period":
                imMessageTemplateDTO.setCode("buffer_period_template");
                var3.put("key", "deadline_date");
                var3.put("value", sdf.format(calculateEndDay(startDay, bufferDays)));
                variableList.add(var3);
                imMessageTemplateDTO.setVariableList(variableList);
                break;
            case "data_retention":
                imMessageTemplateDTO.setCode("data_retention_period_template");
                var3.put("key", "deadline_date");
                var3.put("value", sdf.format(calculateEndDay(startDay, expiredDays)));
                variableList.add(var3);
                imMessageTemplateDTO.setVariableList(variableList);
                break;
            default:
                throw new DWBusinessException("unsupported message type");
        }
        imMessageTemplateDTO.setVariableList(variableList);
        return imMessageTemplateDTO;
    }

    private Date calculateEndDay(Date startDay, Integer bufferDays) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startDay);
        calendar.add(Calendar.DATE, bufferDays);
        return calendar.getTime();
    }

    @Override
    public void noticeAppExpired(String tenantId, String appCode) throws Exception {
        Map<String, Object> requestMap = new HashMap<>();
        requestMap.put("tenantId", tenantId);
        requestMap.put("appId", appCode);
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        String tenantToken = iamService.getTenantToken(tenantId);
        headers.add("token", tenantToken);
        //构建消息对象
        TenantAppExpireDTO tenantAppExpireDTO = buildAppMessage(tenantId, appCode, "data_retention", tenantToken);
        HttpEntity<TenantAppExpireDTO> requestEntity = new HttpEntity<>(tenantAppExpireDTO, headers);
        ResponseEntity<Map> result = restTemplate.exchange(aimUrl + "/api/aim/v2/expire/message", HttpMethod.POST, requestEntity, Map.class);
        if (!(result.getStatusCodeValue() == 200)) {
            log.error("notice app [expired] failed for tenant:{} appcode:{}", tenantId, appCode);
        }
    }

}
