package com.digiwin.athena.knowledgegraph.component;

import com.digiwin.app.container.exceptions.DWArgumentException;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.annotation.ScheduledCall;
import com.digiwin.athena.domain.component.ComponentTypeEnum;
import com.digiwin.athena.domain.component.bo.*;
import com.digiwin.athena.kmservice.neo4j.Neo4j1Config;
import com.digiwin.athena.kmservice.neo4j.Neo4j2Config;
import com.digiwin.athena.knowledgegraph.component.dao.ComponentStructuredListDAO;
import com.digiwin.athena.knowledgegraph.component.dao.TenantAppComponentListDAO;
import com.digiwin.athena.knowledgegraph.domain.ComponentStructuredList;
import com.digiwin.athena.knowledgegraph.domain.VariableDefinition;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.domain.definition.FieldDescription;
import com.digiwin.athena.knowledgegraph.domain.mechanism.Mechanism;
import com.digiwin.athena.knowledgegraph.domain.mechanism.MechanismTenantParadigm;
import com.digiwin.athena.knowledgegraph.domain.mechanism.Principle;
import com.digiwin.athena.knowledgegraph.service.IMechanismService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.service.impl.AppService;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.ogm.session.Session;
import org.neo4j.ogm.session.SessionFactory;
import org.neo4j.ogm.transaction.Transaction;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
public class ComponentManager {

    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate mongoTemplateSystem;

    @Autowired
    @Qualifier("knowledgegraphTenant")
    private MongoTemplate mongoTemplate;

    @Autowired
    SessionFactory neo4jSessionFactory;

    @Autowired(required = false)
    @Qualifier(Neo4j1Config.SESSION_FACTORY)
    SessionFactory sessionFactoryDomain1;

    @Autowired(required = false)
    @Qualifier(Neo4j2Config.SESSION_FACTORY)
    SessionFactory sessionFactoryDomain2;

    @Autowired
    @Lazy
    private ComponentStructuredListDAO componentStructuredListDAO;

    @Autowired
    private TenantAppComponentListDAO tenantAppComponentListDAO;

    @Autowired
    IMechanismService mechanismService;
    @Autowired
    AppService appService;

    @Autowired
    private List<ComponentApplyService> applyServices;

    @Autowired
    KgInnerService kgInnerService;

    @Autowired
    private DataPickService dataPickService;

    public static final String APP_CODE = "appCode";
    public static final String PARADIGM = "paradigm";
    public static final String TYPE = "type";
    public static final String VERSION = "version";
    public static final String TENANT_ID = "tenantId";
    public static final String MECHANISM_CODE = "mechanismCode";


    public void applyComponent(ComponentBO componentBO,ComponentApplyContext context) throws Exception {
        ComponentApplyService componentApplyService = null;
        for (ComponentApplyService applyService: applyServices){
            if (applyService.accept(componentBO)){
                componentApplyService = applyService;
                break;
            }
        }
        if(null!=componentApplyService){
            componentApplyService.apply(componentBO,context);
        }
    }

    public void unApplyComponent(ComponentBO componentBO,ComponentApplyContext context) throws Exception {
        ComponentApplyService componentApplyService = null;
        for (ComponentApplyService applyService: applyServices){
            if (applyService.accept(componentBO)) {
                componentApplyService = applyService;
                break;
            }
        }
        if (null != componentApplyService) {
            componentApplyService.unApply(componentBO, context);
        }
    }

    /**
     * 根据应用code,范式code从系统级组件清单中获取系统级范式组件
     *
     * @param appCode  应用code
     * @param paradigm 范式code
     * @param version  版本号
     * @return
     * @throws DWBusinessException
     */
    public ComponentStructuredList findBaseFunctionComponentStructuredList(String appCode, String paradigm, String version, String tenantId) throws DWBusinessException {
        return componentStructuredListDAO.findBaseFunctionComponentStructuredList(appCode, paradigm,version, tenantId);
    }

    /**
     * 根据应用code,范式code从组件清单中获取范式组件
     *
     * @param appCode   应用code
     * @param paradigms 范式code集合
     * @param version   版本号
     * @return
     * @throws DWBusinessException
     */
    public List<ComponentStructuredList> findBaseFunctionComponentStructuredList(String appCode, List<String> paradigms, String version, String tenantId) throws DWBusinessException {
        return componentStructuredListDAO.findBaseFunctionComponentStructuredList(appCode, paradigms,version, tenantId);
    }

    /**
     * 根据机制code从应用组件清单表中获取机制下组件数据
     *
     * @param mechanismCode 机制code
     * @return
     * @throws DWBusinessException
     */
    public ComponentStructuredList findMechanismComponentStructuredList(String mechanismCode ) throws DWBusinessException {
        String version = kgInnerService.getTenantVersion(AthenaUtils.getTenantId());
        return componentStructuredListDAO.findMechanismComponentStructuredList(mechanismCode, version, null);
    }

    /**
     * 根据机制code从应用组件清单表中获取系统级机制组件数据
     *
     * @param mechanismCode 机制code
     * @param version       租户版本
     * @return
     * @throws DWBusinessException
     */
    public ComponentStructuredList findMechanismComponentStructuredList(String mechanismCode, String version, String tenantId) throws DWBusinessException {
        return componentStructuredListDAO.findMechanismComponentStructuredList(mechanismCode,version, tenantId);
    }

    /**
     * 根据机制code从应用组件清单表中获取机制下组件数据
     *
     * @param mechanismCodes 机制code
     * @param version        租户版本
     * @return
     * @throws DWBusinessException
     */
    public List<ComponentStructuredList> findMechanismComponentStructuredList(List<String> mechanismCodes,String version, String tenantId) throws DWBusinessException {
        return componentStructuredListDAO.findMechanismComponentStructuredList(mechanismCodes, version, tenantId);
    }

    public TenantAppComponentListBO findTenantComponentList(String appCode, String tenantId) throws DWBusinessException {
        TenantAppComponentListBO list = tenantAppComponentListDAO.findTenantAppComponentList(appCode, tenantId);
        if (list == null) {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.noAvailableComponent4", tenantId, appCode));
        }
        return list;
    }

    /**
     * 根据应用code和租户id清除租户下的范式和机制数据
     * @param appCode 应用code
     * @param tenantId 租户id
     */
    private void clearMechanismTenantParadigm(String appCode, String tenantId){
        Query query = new Query();
        Criteria criteria = Criteria.where("code").is(appCode).and("tenantId").is(tenantId);
        query.addCriteria(criteria);
        this.mongoTemplate.remove(query, "mechanismTenantParadigm");
    }

    /**
     * 初始化租户机制数据表
     * @param appCode 应用code
     * @param tenantId 租户id
     * @param mechanismCodes 机制code集合
     * @param paradigms 范式code集合
     */
    private void initMechanismTenantParadigm(String appCode, String tenantId, List<String> mechanismCodes, List<String> paradigms){
//        if(CollectionUtils.isEmpty(mechanismCodes)){
//            return;
//        }

        Query query = new Query();
        Criteria criteria = Criteria.where("code").is(appCode).and("tenantId").is(tenantId);
        query.addCriteria(criteria);

        Update update = new Update();
        Update.AddToSetBuilder add = update.new AddToSetBuilder("value");
        update = add.each(mechanismCodes);

        //把租户所属范式也更新到paradigms字段
        Update.AddToSetBuilder addParadigms = update.new AddToSetBuilder("paradigms");
        update = addParadigms.each(paradigms);

        this.mongoTemplate.upsert(query, update, MechanismTenantParadigm.class);
    }

    /**
     * 初始化时调用，根据应用(议题)编号、范式编号、机制，初始化租户级应用组件清单
     * @param appCode 应用code
     * @param tenantId 租户id
     * @param paradigm 范式code
     * @param mechanismCodes 用户需要绑定的机制code集合
     * @throws DWBusinessException
     */
    public void InitializeTenantApplicationComponentList(String appCode, String tenantId, String paradigm, List<String> mechanismCodes) throws DWBusinessException {
        TenantAppComponentListBO list = tenantAppComponentListDAO.findTenantAppComponentList(appCode, tenantId);
        String version = kgInnerService.getTenantVersion(tenantId);
        if (list != null) {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.initializedComponent",tenantId,appCode));
        }
        this.checkDependMechanism(appCode, tenantId, mechanismCodes, false);

        // 从系统级应用组件中根据应用code范式code获取范式组件，以及根据机制code获取机制组件
        TenantAppComponentListBO tenantAppComponentList = GetTenantAllApplicationComponentList(appCode, tenantId, paradigm, mechanismCodes);

        // 根据应用code获取系统级所有机制组件
        List<String> allMechanismCodes = componentStructuredListDAO.findAllMechanismCodes(appCode,version, tenantId);
        // 获取App下所有的范式组件数据和机制组件数据
        TenantAppComponentListBO allAppComponentList = GetAllApplicationComponentList(appCode, tenantId, allMechanismCodes);

        log.info("开始依据租户组件清单更新组件:{}", tenantId);
        Application application = appService.getApplicationByCode(appCode, tenantId);
        if(application == null){
            log.error("appEntity节点不存在或者appEntity节点中nameSpace为空，请先创建后再初始化!");
            throw new DWArgumentException("appEntity",I18nUtils.getValue("knowledgegraph.appEntity.notExist"));
        }

//        Session session1 = sessionFactoryDomain1.openSession();
//        Transaction transaction1 = session1.beginTransaction();
//        Session session2 = sessionFactoryDomain2.openSession();
//        Transaction transaction2 = session2.beginTransaction();
//        List<Session> sessions = Arrays.asList(session1, session2);
        List<String> cypherList = new ArrayList<>();
        //禁用多余的组件
        allAppComponentList.getComponents().forEach((component) -> {
            if (!Contains(tenantAppComponentList.getComponents(), component)) {
                //基础资料、报表，禁用关系
                DisableComponent(component, tenantId, false, cypherList, appCode, mechanismCodes, null);
            }
        });
        //设置默认值
        tenantAppComponentList.getComponents().forEach((component) -> {
            try {
                SetDefaultComponent(component, tenantId, true, cypherList, mechanismCodes, null, null);
            } catch (Exception e) {
                log.error("error while setting variable {} for tenant {}", component.getCode(), tenantId);
            }
        });

        executeCypher(cypherList, this.sessionFactoryDomain1);
        executeCypher(cypherList, this.sessionFactoryDomain2);
//        transaction1.commit();
//        transaction2.commit();

        //最后才保存，中间失败下次还有机会重新初始化
        tenantAppComponentListDAO.SaveTenantAppComponentList(tenantAppComponentList);
        //初始化MechanismTenantParadigm表
        this.initMechanismTenantParadigm(appCode, tenantId, mechanismCodes, Arrays.asList(paradigm));
    }
    @ScheduledCall
    public void InitializeTenantApplicationComponentList(String appCode, String tenantId, List<String> paradigms, List<String> mechanismCodes) throws DWBusinessException {
        TenantAppComponentListBO list = tenantAppComponentListDAO.findTenantAppComponentList(appCode, tenantId);
        String version = kgInnerService.getTenantVersion(tenantId);
        if (list != null) {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.initializedComponent",tenantId,appCode));
        }
        this.checkDependMechanism(appCode, tenantId, mechanismCodes, false);

        //初始化时可以使用机制中的所有组件
        TenantAppComponentListBO tenantAppComponentList = GetTenantAllApplicationComponentList(appCode, tenantId, paradigms, mechanismCodes);

        List<String> allMechanismCodes = componentStructuredListDAO.findAllMechanismCodes(appCode,version, tenantId);
        TenantAppComponentListBO allAppComponentList = GetAllApplicationComponentList(appCode, tenantId, allMechanismCodes);

        log.info("开始依据租户组件清单更新组件:{}", tenantId);
        Application application = appService.getApplicationByCode(appCode, tenantId);
        if(application == null){
            log.error("appEntity节点不存在或者appEntity节点中nameSpace为空，请先创建后再初始化!");
            throw new DWArgumentException("appEntity",I18nUtils.getValue("knowledgegraph.appEntity.notExist"));
        }
        List<String> cypherList = new ArrayList<>();
        //禁用多余的组件
        allAppComponentList.getComponents().forEach((component) -> {
            if (!Contains(tenantAppComponentList.getComponents(), component)) {
                //基础资料、报表，禁用关系
                DisableComponent(component, tenantId, false, cypherList, appCode, mechanismCodes, null);
            }
        });
        //设置默认值
        tenantAppComponentList.getComponents().forEach((component) -> {
            try {
                SetDefaultComponent(component, tenantId, true, cypherList, mechanismCodes, null, null);
            } catch (Exception e) {
                log.error("error while setting variable {} for tenant {}", component.getCode(), tenantId);
            }
        });

        executeCypher(cypherList, this.sessionFactoryDomain1);
        executeCypher(cypherList, this.sessionFactoryDomain2);


        //最后才保存，中间失败下次还有机会重新初始化
        tenantAppComponentListDAO.SaveTenantAppComponentList(tenantAppComponentList);
        //初始化MechanismTenantParadigm表
        this.initMechanismTenantParadigm(appCode, tenantId, mechanismCodes, paradigms);
    }

    private void executeCypher(List<String> cypherList, SessionFactory sessionFactory){
        if(CollectionUtils.isEmpty(cypherList)) {
            return;
        }
        if (sessionFactory == null) {
            return;
        }
        Session session1 = sessionFactory.openSession();
        Transaction transaction1 = session1.beginTransaction();
        cypherList.forEach(cypher -> session1.query(cypher, new HashMap<>()));
        transaction1.commit();
        session1.clear();
    }

    /**
     * 更新租户级机制数据以及组件数据
     * @param appCode        应用code
     * @param tenantId       租户id
     * @param paradigm       范式
     * @param mechanismCodes 应用下有权限的机制code集合
     * @param mechanismCode  待更新的机制code
     * @throws DWBusinessException
     */
    //机制内容调整时调用，根据应用(议题)编号、范式编号、机制，删除的机制原理或原理明细，更新租户应用组件清单组件状态
    //specificMechanismCode 用于交付设计器更新机制时，能知道更新的是哪个机制
    public void UpdateTenantApplicationComponentListByMechanism(String appCode, String tenantId, String paradigm, List<String> mechanismCodes, String mechanismCode) throws DWBusinessException {
        TenantAppComponentListBO list = tenantAppComponentListDAO.findTenantAppComponentList(appCode, tenantId);
        if (list == null) {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.noAvailableComponent4", tenantId, appCode));
        }

        this.checkDependMechanism(appCode, tenantId, mechanismCodes, true);
        // 从系统级应用组件中根据应用code,范式code获取范式组件，以及根据机制code获取机制组件
        TenantAppComponentListBO allAppComponentList = GetTenantAllApplicationComponentList(appCode, tenantId, paradigm, mechanismCodes);
        //找到租户级下的机制组件
        String version = kgInnerService.getTenantVersion(tenantId);
        TenantAppComponentListBO tenantAppComponentList = GetTenantUsingApplicationComponentList(appCode, tenantId, paradigm, mechanismCodes,version);

        log.info("开始依据租户组件清单更新组件:{}", tenantId);
        Application application = appService.getApplicationByCode(appCode, tenantId);
        if(application == null){
            log.error("appEntity节点不存在或者appEntity节点中nameSpace为空，请先创建后再初始化!");
            throw new DWArgumentException("appEntity",I18nUtils.getValue("knowledgegraph.appEntity.notExist"));
        }
//        Session session1 = sessionFactoryDomain1.openSession();
//        Transaction transaction1 = session1.beginTransaction();
//        Session session2 = sessionFactoryDomain2.openSession();
//        Transaction transaction2 = session2.beginTransaction();
//        List<Session> sessions = Arrays.asList(session1, session2);
        List<String> cypherList = new ArrayList<>();

        //禁用多余的组件
        allAppComponentList.getComponents().forEach((component)->{
            if (!Contains(tenantAppComponentList.getComponents(), component)) {
                component.setDisabled(true);
                //基础资料、报表，禁用关系
                DisableComponent(component, tenantId, true, cypherList, appCode, mechanismCodes, mechanismCode);
            }
        });
        if(StringUtils.isEmpty(mechanismCode)) {
            //设置默认值
            tenantAppComponentList.getComponents().forEach((component) -> {
                try {
                    if (!component.isDisabled()) {  //当存在机制的时候，才会可能导致基础资料或者报表关系的变更
                        SetDefaultComponent(component, tenantId, false, cypherList, mechanismCodes, mechanismCode, null);
                    }
                } catch (Exception e) {
                    log.error("error while setting variable {} for tenant {}", component.getCode(), tenantId);
                }
            });
        }else{
            List<ComponentBO> componentBOS = GetTenantMechanismComponentList(mechanismCode, tenantId,version);
            Map allMechanismVariable = this.getAllMechanismVariable(componentBOS);
            componentBOS.forEach((component) -> {
                try {
                    if (!component.isDisabled()) {
                        SetDefaultComponent(component, tenantId, true, cypherList, mechanismCodes, mechanismCode, allMechanismVariable);
                    }
                } catch (Exception e) {
                    log.error("error while setting variable {} for tenant {}", component.getCode(), tenantId);
                }
            });
        }

        executeCypher(cypherList, this.sessionFactoryDomain1);
        executeCypher(cypherList, this.sessionFactoryDomain2);
//        transaction1.commit();
//        transaction2.commit();

        tenantAppComponentList.setUid(list.getUid());
        tenantAppComponentListDAO.SaveTenantAppComponentList(tenantAppComponentList);
        //初始化MechanismTenantParadigm表
        this.initMechanismTenantParadigm(appCode, tenantId, mechanismCodes, Arrays.asList(paradigm));
    }

    /**
     * 获取机制变量详情数据，包含租户设置的具体数值
     * @param componentBOS 组件集合
     * @return
     */
    private Map getAllMechanismVariable(List<ComponentBO> componentBOS) {
        List<String> collect = componentBOS.stream().filter(e -> ComponentTypeEnum.MechanismVariable.equals(e.getType())
                || ComponentTypeEnum.MechanismParam.equals(e.getType())).map(ComponentBO::getCode).distinct().collect(Collectors.toList());
        Map map = new HashMap();
        if(CollectionUtils.isNotEmpty(collect)){
            collect.forEach(key->{
                try {
                    Object variable = this.mechanismService.getVariable(key, AthenaUtils.getTenantId());
                    if(!ObjectUtils.isEmpty(variable)){
                        Map<String, Object> variableMap = (Map<String, Object>) variable;
                        map.putAll(variableMap);
                    }
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                }
            });
        }

        return map;
    }
    @ScheduledCall
    public void UpdateTenantApplicationComponentListByMechanism(String appCode, String tenantId, List<String> paradigms, List<String> mechanismCodes, String mechanismCode) throws DWBusinessException {
        TenantAppComponentListBO list = tenantAppComponentListDAO.findTenantAppComponentList(appCode, tenantId);
        if (list == null) {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.noAvailableComponent4",tenantId,appCode));
        }

        this.checkDependMechanism(appCode, tenantId, mechanismCodes, true);
        String version = kgInnerService.getTenantVersion(tenantId);
        TenantAppComponentListBO allAppComponentList = GetTenantAllApplicationComponentList(appCode, tenantId, paradigms, mechanismCodes);
        //找到租户有储存的机制组件
        TenantAppComponentListBO tenantAppComponentList = GetTenantUsingApplicationComponentList(appCode, tenantId, paradigms, mechanismCodes);

        log.info("开始依据租户组件清单更新组件:{}", tenantId);
        Application application = appService.getApplicationByCode(appCode, tenantId);
        if(application == null){
            log.error("appEntity节点不存在或者appEntity节点中nameSpace为空，请先创建后再初始化!");
            throw new DWArgumentException("appEntity",I18nUtils.getValue("knowledgegraph.appEntity.notExist"));
        }
//        Session session1 = sessionFactoryDomain1.openSession();
//        Transaction transaction1 = session1.beginTransaction();
//        Session session2 = sessionFactoryDomain2.openSession();
//        Transaction transaction2 = session2.beginTransaction();
//        List<Session> sessions = Arrays.asList(session1, session2);
        List<String> cypherList = new ArrayList<>();

        //禁用多余的组件
        allAppComponentList.getComponents().forEach((component)->{
            if (!Contains(tenantAppComponentList.getComponents(), component)) {
                component.setDisabled(true);
                //基础资料、报表，禁用关系
                DisableComponent(component, tenantId, true, cypherList, appCode, mechanismCodes, mechanismCode);
            }
        });
        if(StringUtils.isEmpty(mechanismCode)) {
            //设置默认值
            tenantAppComponentList.getComponents().forEach((component) -> {
                try {
                    if (!component.isDisabled()) {  //当存在机制的时候，才会可能导致基础资料或者报表关系的变更
                        SetDefaultComponent(component, tenantId, false, cypherList, mechanismCodes, mechanismCode, null);
                    }
                } catch (Exception e) {
                    log.error("error while setting variable {} for tenant {}", component.getCode(), tenantId);
                }
            });
        }else{
            List<ComponentBO> componentBOS = GetTenantMechanismComponentList(mechanismCode, tenantId, version);
            Map allMechanismVariable = this.getAllMechanismVariable(componentBOS);
            componentBOS.forEach((component) -> {
                try {
                    if (!component.isDisabled()) {
                        SetDefaultComponent(component, tenantId, true, cypherList, mechanismCodes, mechanismCode, allMechanismVariable);
                    }
                } catch (Exception e) {
                    log.error("error while setting variable {} for tenant {}", component.getCode(), tenantId);
                }
            });
        }

        executeCypher(cypherList, this.sessionFactoryDomain1);
        executeCypher(cypherList, this.sessionFactoryDomain2);
//        transaction1.commit();
//        transaction2.commit();

        tenantAppComponentList.setUid(list.getUid());
        tenantAppComponentListDAO.SaveTenantAppComponentList(tenantAppComponentList);
        //初始化MechanismTenantParadigm表
        this.initMechanismTenantParadigm(appCode, tenantId, mechanismCodes, paradigms);
    }

    /**
     * 根据应用code获取租户授权的应用组件清单
     * @param appCode 应用code
     * @param tenantId 租户id
     * @param components 应用下的组件清单
     * @return
     * @throws DWBusinessException
     */
    public List<ComponentBaseInfoBO> FilterByTenantApplicationComponentList(String appCode, String tenantId, List<ComponentBaseInfoBO> components) throws DWBusinessException {
        //return components;

        try {
            TenantAppComponentListBO appComponentListDO = tenantAppComponentListDAO.findTenantAppComponentList(appCode, tenantId);
            //未初始化前不做过滤
            if (appComponentListDO == null || appComponentListDO.getComponents().isEmpty()) {
                return components;
            }
            List<ComponentBaseInfoBO> filteredComponents = new ArrayList<ComponentBaseInfoBO>();
            List<ComponentBO> tenantComponents = appComponentListDO.getComponents();
            for (ComponentBaseInfoBO component : components) {
                boolean enabled = false;
                for (ComponentBO tenantComonent : tenantComponents) {
                    if (component.getCode().equals(tenantComonent.getCode()) && component.getType().equals(tenantComonent.getType())) {
                        if (!tenantComonent.isDisabled()) {
                            enabled = true;
                        }
                        break;
                    }
                }
                if (enabled) {
                    filteredComponents.add(component);
                }
            }
            return filteredComponents;
        } catch (Exception exception) {
            log.error("error while filtering components: " + exception.getLocalizedMessage());
            return components;
        }
    }

    /**
     * 从系统级应用组件中根据应用code,范式code获取范式组件，以及根据机制code获取机制组件
     *
     * @param appCode        应用code
     * @param tenantId       租户id
     * @param paradigm       范式
     * @param mechanismCodes 用户需要绑定的机制code集合
     * @return
     * @throws DWBusinessException
     */
    private TenantAppComponentListBO GetTenantAllApplicationComponentList(String appCode, String tenantId, String paradigm, List<String> mechanismCodes) throws DWBusinessException {
        List<ComponentBaseInfoBO> emptyList = new ArrayList<ComponentBaseInfoBO>();
        String version = kgInnerService.getTenantVersion(tenantId);
        // 根据应用code,范式code从系统级组件清单中获取范式组件
        ComponentStructuredList baseFunctionList = findBaseFunctionComponentStructuredList(appCode, paradigm, version, tenantId);
        List<ComponentBO> mergedList = new ArrayList<ComponentBO>();
        Merge(mergedList, baseFunctionList.getComponents(), emptyList);

        if (CollectionUtils.isNotEmpty(mechanismCodes)) {
            for (String mechanismCode : mechanismCodes) {
                // 将应用下系统级组件和用户需要绑定机制下的组件做合并
                Merge(mergedList, findMechanismComponentStructuredList(mechanismCode,version, tenantId).getComponents(), emptyList);
            }
        }
        TenantAppComponentListBO tenantAppComponentList = new TenantAppComponentListBO();

        tenantAppComponentList.setTenantId(tenantId);
        tenantAppComponentList.setAppCode(appCode);
        tenantAppComponentList.setComponents(mergedList);
        return tenantAppComponentList;
    }

    private TenantAppComponentListBO GetTenantAllApplicationComponentList(String appCode, String tenantId, List<String> paradigms, List<String> mechanismCodes) throws DWBusinessException {
        List<ComponentBaseInfoBO> emptyList = new ArrayList<ComponentBaseInfoBO>();
        String version = kgInnerService.getTenantVersion(tenantId);
        List<ComponentStructuredList> baseFunctionComponentStructuredList = findBaseFunctionComponentStructuredList(appCode, paradigms,version, tenantId);
        List<ComponentBO> mergedList = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(baseFunctionComponentStructuredList)) {
            for (ComponentStructuredList baseFunctionList : baseFunctionComponentStructuredList) {
                Merge(mergedList, baseFunctionList.getComponents(), emptyList);
            }
        }

        if(CollectionUtils.isNotEmpty(mechanismCodes)) {
            for (String mechanismCode : mechanismCodes) {
                Merge(mergedList, findMechanismComponentStructuredList(mechanismCode,version, tenantId).getComponents(), emptyList);
            }
        }
        TenantAppComponentListBO tenantAppComponentList = new TenantAppComponentListBO();

        tenantAppComponentList.setTenantId(tenantId);
        tenantAppComponentList.setAppCode(appCode);
        tenantAppComponentList.setComponents(mergedList);
        return tenantAppComponentList;
    }

    /**
     * 根据应用code获取系统级范式组件数据，以及传参中的机制组件数据
     * @param appCode 应用code
     * @param tenantId 租户id
     * @param mechanismCodes 应用code下系统级所有机制组件机制code集合
     * @return
     * @throws DWBusinessException
     */
    private TenantAppComponentListBO GetAllApplicationComponentList(String appCode, String tenantId, List<String> mechanismCodes) throws DWBusinessException {
        List<ComponentBaseInfoBO> emptyList = new ArrayList<>();
        String version = kgInnerService.getTenantVersion(tenantId);
        List<ComponentStructuredList> baseFunctionLists = componentStructuredListDAO.findBaseFunctionComponentStructuredList(appCode, version, tenantId);
        List<ComponentBO> mergedList = new ArrayList<ComponentBO>();
        baseFunctionLists.forEach((baseFunctionList)-> {
            Merge(mergedList, baseFunctionList.getComponents(), emptyList);
        });

        for (String mechanismCode : mechanismCodes) {
            Merge(mergedList, findMechanismComponentStructuredList(mechanismCode,version, tenantId).getComponents(), emptyList);
        }
        TenantAppComponentListBO tenantAppComponentList = new TenantAppComponentListBO();

        tenantAppComponentList.setTenantId(tenantId);
        tenantAppComponentList.setAppCode(appCode);
        tenantAppComponentList.setComponents(mergedList);
        return tenantAppComponentList;
    }

    /**
     * 获取租户级下的组件，例如项目、任务、侦测、机制变量、数据录入、报表、服务编排，机制原理
     * @param appCode 应用code
     * @param tenantId 租户id
     * @param paradigm 方式
     * @param mechanismCodes 机制code
     * @param version 租户版本
     * @return
     * @throws DWBusinessException
     */
    public TenantAppComponentListBO GetTenantUsingApplicationComponentList(String appCode, String tenantId, String paradigm, List<String> mechanismCodes, String version) throws DWBusinessException {

        List<ComponentBaseInfoBO> emptyList = new ArrayList<>();
        //根据应用code,范式code从系统级组件清单中获取范式组件
        ComponentStructuredList baseFunctionList = findBaseFunctionComponentStructuredList(appCode, paradigm, version, tenantId);
        List<ComponentBO> mergedList = new ArrayList<ComponentBO>();
        Merge(mergedList, baseFunctionList.getComponents(), emptyList);

        for (String mechanismCode : mechanismCodes) {
            //根据机制code从组件清单表获取机制组件数据，并将不在租户级机制中的机制原理排除
            List<ComponentBO> mechanismComponents = GetTenantMechanismComponentList(mechanismCode, tenantId,version);
            mechanismComponents.forEach((mechanismComponent)-> {
                if (!Contains(mergedList, mechanismComponent)) {
                    mergedList.add(mechanismComponent);
                }
            });
        }
        TenantAppComponentListBO tenantAppComponentList = new TenantAppComponentListBO();
        tenantAppComponentList.setTenantId(tenantId);
        tenantAppComponentList.setAppCode(appCode);
        tenantAppComponentList.setComponents(mergedList);
        return tenantAppComponentList;
    }

    public TenantAppComponentListBO GetTenantUsingApplicationComponentList(String appCode, String tenantId, List<String> paradigms, List<String> mechanismCodes) throws DWBusinessException {
        List<ComponentBaseInfoBO> emptyList = new ArrayList<>();
        String version = kgInnerService.getTenantVersion(tenantId);
        List<ComponentStructuredList> baseFunctionComponentStructuredList = findBaseFunctionComponentStructuredList(appCode, paradigms,version, tenantId);
        List<ComponentBO> mergedList = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(baseFunctionComponentStructuredList)) {
            for (ComponentStructuredList baseFunctionList : baseFunctionComponentStructuredList) {
                Merge(mergedList, baseFunctionList.getComponents(), emptyList);
            }
        }

        for (String mechanismCode : mechanismCodes) {
            List<ComponentBO> mechanismComponents = GetTenantMechanismComponentList(mechanismCode, tenantId, version);
            mechanismComponents.forEach((mechanismComponent)-> {
                if (!Contains(mergedList, mechanismComponent)) {
                    mergedList.add(mechanismComponent);
                }
            });
        }
        TenantAppComponentListBO tenantAppComponentList = new TenantAppComponentListBO();
        tenantAppComponentList.setTenantId(tenantId);
        tenantAppComponentList.setAppCode(appCode);
        tenantAppComponentList.setComponents(mergedList);
        return tenantAppComponentList;
    }

    /**
     * 根据机制code从组件清单表获取机制组件数据，并将不在租户级机制中的机制原理排除
     * @param mechanismCode 机制code
     * @param tenantId 租户code
     * @param version 版本
     * @return
     * @throws DWBusinessException
     */
    private List<ComponentBO> GetTenantMechanismComponentList(String mechanismCode, String tenantId, String version) throws DWBusinessException {
        List<ComponentBO> components = new ArrayList<>();
        ComponentStructuredList mechanismComponentList = findMechanismComponentStructuredList(mechanismCode, version, tenantId);
        Query query = new Query();
        Criteria criteria = Criteria.where("code").is(mechanismCode).and("tenantId").is(tenantId);//根据主机制code查询
        query.addCriteria(criteria);

        List<ComponentBaseInfoBO> componentsToRemove = new ArrayList<ComponentBaseInfoBO>();
        //获取租户级机制数据
        Mechanism mechanism =  this.mongoTemplate.findOne(query, Mechanism.class);
        if (mechanism != null) {
            //先找到哪些原理和原理明细在租户的机制里不存在
            mechanismComponentList.getComponents().forEach((principleComponent)->{
                Principle principle = null;
                if(CollectionUtils.isNotEmpty(mechanism.getPrinciples())){
                    for (Principle item : mechanism.getPrinciples()) {
                        if (principleComponent.getCode().equals(item.getCode())) {
                            principle = item;
                        }
                    }
                }
                if (principle == null) {
                    ComponentBaseInfoBO componentToRemove = new ComponentBaseInfoBO();
                    componentToRemove.setCode(principleComponent.getCode());
                    componentToRemove.setType(ComponentTypeEnum.MechanismPrinciple);
                    componentsToRemove.add(componentToRemove);
                } else {
                    List<String> principleDetailCodes = new ArrayList<String>();
                    principle.getDetails().forEach((principleDetail)->{
                        principleDetailCodes.add(principleDetail.getCode());
                    });
                    principleComponent.getSubComponents().forEach((detailComponent)-> {
                        if (detailComponent.getType().equals(ComponentTypeEnum.MechanismPrincipleDetail) && !principleDetailCodes.contains(detailComponent.getCode())) {
                            ComponentBaseInfoBO componentToRemove = new ComponentBaseInfoBO();
                            componentToRemove.setCode(detailComponent.getCode());
                            componentToRemove.setType(ComponentTypeEnum.MechanismPrincipleDetail);
                            componentsToRemove.add(componentToRemove);
                        }
                    });
                }
            });
            Merge(components, mechanismComponentList.getComponents(), componentsToRemove);
        } else {
            Merge(components, mechanismComponentList.getComponents(), new ArrayList<>());
        }
        return components;
    }

    /**
     * 将组件清单中的组件平铺合并
     *
     * @param original 原有的组件
     * @param toAdd    清单组件中待添加的组件
     * @param skipList 需过滤的组件
     */
    public void Merge(List<ComponentBO> original, List<StructuredComponentBO> toAdd, List<ComponentBaseInfoBO> skipList) {
        if (CollectionUtils.isNotEmpty(toAdd)) {
            for (StructuredComponentBO componentToAdd : toAdd) {
                boolean toSkip = false;
                for (ComponentBaseInfoBO item : skipList) {
                    if (componentToAdd.getCode().equals(item.getCode()) && componentToAdd.getType().equals(item.getType())) {
                        toSkip = true;
                    }
                }
                if (toSkip) {
                    continue;
                }
                List<StructuredComponentBO> subComponents = componentToAdd.getSubComponents();
                if(!Contains(original, componentToAdd)){
                    componentToAdd.setSubComponents(null);
                    original.add(componentToAdd);
                }
                if(CollectionUtils.isNotEmpty(subComponents)){
                    Merge(original, subComponents, skipList);
                }
//            ComponentBO component = new ComponentBO();
//            component.setCode(componentToAdd.getCode());
//            component.setName(componentToAdd.getName());
//            component.setType(componentToAdd.getType());
//            component.setVersion(componentToAdd.getVersion());
//            component.setConfigs(componentToAdd.getConfigs());
//            if (!Contains(original, component)) {
//                original.add(component);
//            }
//            if (componentToAdd.getSubComponents().size() > 0) {
//                Merge(original, componentToAdd.getSubComponents(), skipList);
//            }
            }
        }
    }

    private static Boolean Contains(List<ComponentBO> components, ComponentBaseInfoBO component) {
        for (ComponentBO item : components) {
            if (component.getCode().equals(item.getCode()) && component.getType().equals(item.getType())) {
                return true;
            }
        }
        return false;
    }

    /**
     * 将指定组件失效
     * @param componentBO 组件
     * @param tenantId 租户id
     * @param update 是否更新
     * @param cypherList 更新语句
     * @param appNameSpace 应用名称
     * @param mechanismCodes 机制code集合
     * @param mechanismCode 机制code
     */
    //将来这里可以重构，现在写的比较粗糙
    private void DisableComponent(ComponentBO componentBO, String tenantId, boolean update, List<String> cypherList, String appNameSpace,  List<String> mechanismCodes, String mechanismCode) {
        if (componentBO.getType().equals(ComponentTypeEnum.MechanismVariable)) {
            //初始化时机制变量一定还没有值，只有更新时才需要删除
            if (update) {
                this.TryDeleteVariable(componentBO.getCode(), tenantId);
            }
        }
        else if (componentBO.getType().equals(ComponentTypeEnum.BaseDataEntry) || componentBO.getType().equals(ComponentTypeEnum.Report)) {
        //    String cypher = "match (te:TenantEntity{tenantId:'" + tenantId + "'})-[r:ACTIVITY]->(activity:Activity {code:'" + componentBO.getCode() + "'}) delete r";
       //     cypherList.add(cypher);
//            sessions.forEach(session -> session.query(cypher, new HashMap<>()));
//            session.query(cypher, new HashMap<>());
         //   log.info("delete program: {}", cypher);
        } else {
            try {
                if(StringUtils.isNotEmpty(mechanismCode)) {  //当mechanismCode不为空时，才是机制生效调用
                    ComponentApplyContext componentApplyContext = new ComponentApplyContext();
                    componentApplyContext.setMechanismCode(mechanismCode);
                    this.unApplyComponent(componentBO, componentApplyContext);
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
    }

    private void SetDefaultComponent(ComponentBO componentBO, String tenantId, boolean update, List<String> cypherList,List<String> mechanismCodes, String mechanismCode, Map allMechaniseVariables) throws Exception {
        if (componentBO.getType().equals(ComponentTypeEnum.MechanismVariable)) {
            if(componentBO instanceof MechanismVariableComponentBO){
                MechanismVariableComponentBO mechanismVariableComponent = (MechanismVariableComponentBO) componentBO;
                if(!ObjectUtils.isEmpty(mechanismVariableComponent.getValue())){
                    mechanismService.postVariable(mechanismVariableComponent.getCode(), tenantId, mechanismVariableComponent.getValue());
                    log.info("set variable {} value {} for tenant {}", mechanismVariableComponent.getCode(), mechanismVariableComponent.getValue(), tenantId);
                }
            }/*else{
                if (componentBO.getConfigs()!= null && componentBO.getConfigs().containsKey("value")) {
                    mechanismService.postVariable(componentBO.getCode(), tenantId, componentBO.getConfigs().get("value"));
                    log.info("set variable {} value {} for tenant {}", componentBO.getCode(), componentBO.getConfigs().get("value"), tenantId);
                }
            }*/
        } else if (componentBO.getType().equals(ComponentTypeEnum.BaseDataEntry) || componentBO.getType().equals(ComponentTypeEnum.Report)) {
            if(update) { //当更新时才需要建立关系，新建时没有删除过关系，关系在租户系统初始化的时候已经创建好
             //   String cypher = "match (te:TenantEntity{tenantId:'" + tenantId + "'}) match (activity:Activity {code:'" + componentBO.getCode() + "'}) merge (te)-[:ACTIVITY]->(activity)";
          //      cypherList.add(cypher);
//                sessions.forEach(session -> session.query(cypher, new HashMap<>()));
//                session.query(cypher, new HashMap<>());
            //    log.info("merge program: {}", cypher);
            }
        }
        else{
            if(StringUtils.isNotEmpty(mechanismCode)) {  //当mechanismCode不为空时，才是机制生效调用
                ComponentApplyContext componentApplyContext = new ComponentApplyContext();
                componentApplyContext.setMechanismCode(mechanismCode);
                componentApplyContext.setMechanismVariables(allMechaniseVariables);
                this.applyComponent(componentBO, componentApplyContext);
            }
        }
    }

    private boolean TryDeleteVariable(String variableName, String tenantId) {
        try {
            mechanismService.deleteVariable(variableName, tenantId);
            log.info("deleted variable {} of tenant {}", variableName, tenantId);
            return true;
        } catch (Exception e) {
            log.error("error while deleting variable {} of tenant {}", variableName, tenantId);
            return false;
        }
    }

    /**
     * 校验每一个机制的依赖机制是否存在
     *
     * @param appCode        应用code
     * @param tenantId       租户id
     * @param mechanismCodes 应用下有权限的机制code集合
     * @param update         是否更新
     * @return
     */
    private boolean checkDependMechanism(String appCode, String tenantId,List<String> mechanismCodes, boolean update) throws DWBusinessException {
        if(CollectionUtils.isEmpty(mechanismCodes)){
            return false;
        }
        Query query = new Query();
        Criteria criteria = Criteria.where("code").in(mechanismCodes);
        query.addCriteria(criteria);
        List<Mechanism> mechanismList = dataPickService.find(criteria, Mechanism.class, "mechanism");
        List<String> allMechanismCodes = new ArrayList<>();
        allMechanismCodes.addAll(mechanismCodes);
        if(update){ //找一下租户已存在的机制
            Query query2 = new Query();
            Criteria criteria2 = Criteria.where("code").is(appCode).and("tenantId").is(tenantId);
            query2.addCriteria(criteria2);
            MechanismTenantParadigm tenantParadigm = this.mongoTemplate.findOne(query, MechanismTenantParadigm.class);
            if(tenantParadigm != null && CollectionUtils.isNotEmpty(tenantParadigm.getValue())){
                allMechanismCodes.addAll(tenantParadigm.getValue());
            }
        }

        mechanismList.forEach(e->{
            List<String> dependMechanismCodes = e.getDependMechanismCodes();
            if(CollectionUtils.isNotEmpty(dependMechanismCodes)){
                boolean subCollection = CollectionUtils.isSubCollection(dependMechanismCodes, allMechanismCodes);
                if(!subCollection){
                    throw new RuntimeException(I18nUtils.getValue("knowledgegraph.machineNotExist",e.getCode(),e.getName(),dependMechanismCodes));
                }
            }
        });

        return false;
    }


    public FieldDescription createFieldDescriptionByComponent(ComponentBO componentBO) throws DWBusinessException{
//        if(ComponentTypeEnum.MonitorRule.equals(componentBO.getType())){ //侦测
//            FieldDescription monitorRuleParam = this.appService.createMonitorRuleParam(componentBO.getCode(), false);
//            return monitorRuleParam;
//        }else if(ComponentTypeEnum.MechanismParam.equals(componentBO.getType()) || ComponentTypeEnum.ApplicationParam.equals(componentBO.getType())){
//            FieldDescription variableParam = this.appService.createVariableParam(componentBO.getCode());
//            return variableParam;
//        }else if(ComponentTypeEnum.BaseDataEntry.equals(componentBO.getType())){
//            FieldDescription linkParam = this.appService.createLinkParam(componentBO.getCode());
//            return linkParam;
//        }else{
//            FieldDescription defaultParam = this.appService.createDefaultParam(componentBO.getCode());
//            return defaultParam;
//        }
        return this.createFieldDescriptionByComponent(componentBO, null);
    }

    /**
     * 根据机制变量code获取详细信息，例如侦测，机制，基础数据，常规变量
     * @param componentBO 组件
     * @param variableDefinitionsMap 机制变量数据
     * @return
     * @throws DWBusinessException
     */
    public FieldDescription createFieldDescriptionByComponent(ComponentBO componentBO,  Map<String, VariableDefinition> variableDefinitionsMap) throws DWBusinessException{
        if(ComponentTypeEnum.MonitorRule.equals(componentBO.getType())){ //侦测
            FieldDescription monitorRuleParam = this.appService.createMonitorRuleParam(componentBO.getCode(), false, variableDefinitionsMap);
            return monitorRuleParam;
        }else if(ComponentTypeEnum.MechanismParam.equals(componentBO.getType()) || ComponentTypeEnum.ApplicationParam.equals(componentBO.getType())){
            FieldDescription variableParam = this.appService.createVariableParam(componentBO.getCode(), variableDefinitionsMap);
            return variableParam;
        } else if (ComponentTypeEnum.BaseDataEntry.equals(componentBO.getType())) {
            FieldDescription linkParam = this.appService.createLinkParam(componentBO.getCode(), variableDefinitionsMap);
            return linkParam;
        } else {
            FieldDescription defaultParam = this.appService.createDefaultParam(componentBO.getCode(), variableDefinitionsMap);
            return defaultParam;
        }
    }

    /**
     * 获取机制参数的详细信息
     *
     * @param paramDescriptionComponentList 参数展示
     * @return
     * @throws DWBusinessException
     */
    public List<FieldDescription> generateParamDescriptionList(List<ComponentBO> paramDescriptionComponentList) throws DWBusinessException {
        List<FieldDescription> paramDescriptionList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(paramDescriptionComponentList)) {
            String tenantId = AthenaUtils.getTenantId();
            String tenantVersion = kgInnerService.getTenantVersion(tenantId);

            Set<String> collect = paramDescriptionComponentList.stream().map(ComponentBO::getCode).collect(Collectors.toSet());
            Criteria criteria = Criteria.where("code").in(collect);
            List<VariableDefinition> variableDefinitions = dataPickService.find(criteria, VariableDefinition.class, "variableDefinition");
            Map<String, VariableDefinition> variableDefinitionsMap = variableDefinitions.stream().collect(Collectors.toMap(VariableDefinition::getCode, Function.identity(), (a, b) -> a));
            paramDescriptionComponentList.forEach(e->{
                try {
                    FieldDescription fieldDescriptionByComponent = this.createFieldDescriptionByComponent(e, variableDefinitionsMap);
                    if(fieldDescriptionByComponent != null){
                        appService.setFieldDescriptionValue(fieldDescriptionByComponent);
                        paramDescriptionList.add(fieldDescriptionByComponent);
                    }
                } catch (Exception ex) {
                    log.error(ex.getMessage(), ex);
                }
            });
        }
        return paramDescriptionList;
    }

    /**
     * 根据应用code和租户信息清理租户机制组件数据和租户下范式机制数据
     * @param appCode 应用code
     * @param tenantId 租户id
     * @param paradigm 范式code
     * @param mechanismCodes 机制code
     * @throws DWBusinessException
     */
    public void clearTenantApplicationComponentListByMechanism(String appCode, String tenantId, String paradigm, List<String> mechanismCodes) throws DWBusinessException {
        log.info("clearTenantApplicationComponentListByMechanism appCode:{}, tenantId:{}, paradigm:{}, mechanismCodes:{}", appCode, tenantId, paradigm, mechanismCodes);
        //TODO 暂时不判断之前机制的组件是否生效过，并且把生效过的组件停用（因为现在不同范式的机制内容基本一致）
        this.tenantAppComponentListDAO.removeTenantAppComponentList(appCode, tenantId);
        this.clearMechanismTenantParadigm(appCode, tenantId);
    }

    /**
     * 获取需要增加到清单组件表的组件数据
     * @param newComponentBO 待处理的组件数据
     * @param existsComponents 已存在于当前范式组件下的数据
     * @param existsParadigmAppComponent  已存在于应用所有范式下的组件数据
     * @param paradigmCode 范式code
     * @return 返回增加到清单组件表的组件数据
     */
    public List<ComponentBO> getExtraComponent(List<ComponentBO> newComponentBO, List<StructuredComponentBO> existsComponents, Map<String, Set<String>> existsParadigmAppComponent, String paradigmCode){
        List<ComponentBO> result = new ArrayList<>();
        if(CollectionUtils.isEmpty(existsComponents)){
            return newComponentBO;
        }
        List<ComponentBO> mergedList = new ArrayList<ComponentBO>();
        Merge(mergedList,existsComponents, new ArrayList());
        Set<String> collect = mergedList.stream().map(e -> e.getType().toString() + "#" + e.getCode()).collect(Collectors.toSet());
        for(ComponentBO componentBO : newComponentBO){
            if(!collect.contains(componentBO.getType().toString() + "#" + componentBO.getCode())
                    && !existsInOtherParadigm(existsParadigmAppComponent, paradigmCode, componentBO.getType().toString() + "#" + componentBO.getCode())){
                //如果本身不存在，并且也不存在其他范式的组件清单中，则表示这个组件是需要增加的，如果在其他范式的组件清单中存在，则可能这个组件是其他组件特有的，不需要新增
                result.add(componentBO);
            }
        }
        return result;
    }

    private boolean existsInOtherParadigm(Map<String, Set<String>> existsParadigmAppComponent, String paradigmCode, String key){
        if(existsParadigmAppComponent == null){
            return false;
        }
        Set<Map.Entry<String, Set<String>>> entries = existsParadigmAppComponent.entrySet();
        for(Map.Entry<String, Set<String>> entry : entries){
            if(StringUtils.equals(entry.getKey(), paradigmCode)){
                continue;
            }
            Set<String> value = entry.getValue();
            if(value.contains(key)){  //如果存在，直接返回true
                return true;
            }
        }
        return false;
    }

    /**
     * 根据租户授权的机制code集合获取租户级机制组件清单数据（数据表ComponentStructuredList）
     * @param mechanismTenantParadigm 租户范式机制数据
     * @param version 租户版本
     * @return 返回范式机制数据{范式：[机制]}
     */
    public Map<String, List<String>> getTenantParadigm(MechanismTenantParadigm mechanismTenantParadigm, String version, String tenantId) {
        if (CollectionUtils.isNotEmpty(mechanismTenantParadigm.getValue())) {
            List<ComponentStructuredList> mechanismComponentStructuredList = null;
            try {
                mechanismComponentStructuredList = this.findMechanismComponentStructuredList(mechanismTenantParadigm.getValue(), version, tenantId);
            } catch (DWBusinessException e) {
                log.error(e.getMessage(), e);
            }
            if (mechanismComponentStructuredList == null || mechanismComponentStructuredList.isEmpty()) {
                return null;
            } else {
                Map<String, List<String>> collect = mechanismComponentStructuredList.stream().collect(Collectors.toMap(ComponentStructuredList::getParadigm, p -> {
                    List<String> list = new ArrayList<>();
                    list.add(p.getMechanismCode());
                    return list;
                }, (List<String> list1, List<String> list2) -> {
                    list1.addAll(list2);
                    return list1;
                }));
                return collect;
            }
        } else {
            List<String> paradigms = mechanismTenantParadigm.getParadigms();
            if(CollectionUtils.isEmpty(paradigms)){
                return null;
            }
            Map<String,  List<String>> result = new HashMap<>();
            paradigms.forEach(e->{
                result.put(e, new ArrayList<>());
            });
            return result;
        }
    }

    public List<ComponentBO> filterComponentByType(List<ComponentStructuredList> componentStructuredList,
        List<ComponentTypeEnum> mechanismArrayList, String theoryCode) {
        ArrayList<ComponentBO> list = new ArrayList<>();
        for (ComponentStructuredList structuredList : componentStructuredList) {
            List<StructuredComponentBO> components = structuredList.getComponents();
            List<StructuredComponentBO> collect =
                    components.stream().filter(v -> v.getType().equals(ComponentTypeEnum.MechanismPrinciple) && v.getCode().equals(theoryCode)).collect(Collectors.toList());
            recursion(collect, mechanismArrayList, list);
        }
        return list;
    }

    private void recursion(List<StructuredComponentBO> components, List<ComponentTypeEnum> mechanismArrayList,
                           ArrayList<ComponentBO> list) {
        if (CollectionUtils.isNotEmpty(components)) {
            for (StructuredComponentBO component : components) {
                if (mechanismArrayList.contains(component.getType())) {
                    list.add(component);
                }
                if (CollectionUtils.isNotEmpty(component.getSubComponents())) {
                    recursion(component.getSubComponents(), mechanismArrayList, list);
                }
            }
        }
    }
}