package com.digiwin.athena.knowledgegraph.rule.parameter;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.knowledgegraph.domain.VariableDefinition;
import com.digiwin.athena.knowledgegraph.rule.AbstractMonitorRuleMechanismParameterService;
import com.digiwin.athena.knowledgegraph.rule.model.parameter.MechanismParameterReqDO;
import com.digiwin.athena.knowledgegraph.service.impl.MechanismService;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.kg.monitorRule.TriggerDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Author QinQiang
 * @Description 通用频次类型的参数解析
 * @Date 2024/6/6
 **/
@Service
@Slf4j
@MyExceptionHandler
public class FrequencyTypeMechanismParameterService extends
        AbstractMonitorRuleMechanismParameterService<MechanismParameterReqDO, TriggerDTO> {

    @Autowired
    private MechanismService mechanismService;

    @Autowired
    private DataPickService dataPickService;

    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate mongoTemplateSystem;
    private static final String DISPLAY_TYPE = "FREQUENCY";
    private static final String FIX_INTERVAL = "fixInterval";

    @Override
    public boolean accept(String code) {
        // 基于PHP需求之上修改，由于已经使用参数code去匹配，那只能先查一遍所有频次类型的参数，看看是否包含
        Criteria criteria = Criteria.where("displayType").is(DISPLAY_TYPE);
        // 查询VariableDefinition表数据，只需要code字段
        List<VariableDefinition> variableDefinitions = null;
        try {
            variableDefinitions = dataPickService.find(criteria, VariableDefinition.class, "variableDefinition");
        } catch (DWBusinessException e) {
            log.error("控制能力使用侦测类型参数出错：", e);
            variableDefinitions = Collections.emptyList();
        }
        List<String> codes = variableDefinitions.stream().map(VariableDefinition::getCode).collect(Collectors.toList());
        return codes.contains(code);
    }

    @Override
    public TriggerDTO parse(MechanismParameterReqDO mechanismParameterReqDO) {
        try {
            // "value": ["W1;20210801 12:00"] FixInterval
            Map<String, Object> map = (Map<String, Object>) this.mechanismService.getVariable(mechanismParameterReqDO.getVariableCode(), mechanismParameterReqDO.getTenantId());
            Object o = map.get(mechanismParameterReqDO.getVariableCode());
            if(o == null) {
                return null;
            }
            String value = null;
            if(o instanceof List) {
                List<String> list = (List<String>) o;
                if(list.isEmpty()) {
                    return null;
                }
                value = list.get(0);
            } else if (o instanceof String) {
                value = String.valueOf(o);
            }
            if (null == value || value.isEmpty()) {
                return null;
            }
            // W1;20210801 12:00
            String[] split = value.split(";");
            if(split.length != 2) {
                throw new DWBusinessException("参数格式错误");
            }
            TriggerDTO triggerDTO = new TriggerDTO();
            String time = split[1] + ":00";
            // 将20210801 12:00:00这种时间格式转成 2021-08-01 12:00:00
            LocalDateTime localDateTime = LocalDateTime.parse(time, DateTimeFormatter.ofPattern("yyyyMMdd HH:mm:ss"));
            triggerDTO.setStart_time(localDateTime.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
            String frequency = split[0];
            if (frequency.startsWith(FIX_INTERVAL)) {
                frequency = frequency.substring(FIX_INTERVAL.length());
            }
            // 周期 F;H;D;W;M;
            String cycle = frequency.substring(0, 1);
            // 每周/每月的时间点
            String timePoint = frequency.substring(1);
            switch (cycle) {
                case "F":
                    // 分钟
                    triggerDTO.setRepeat_type(1);
                    triggerDTO.setFrequency(Integer.parseInt(timePoint));
                    break;
                case "H":
                    // 小时
                    triggerDTO.setRepeat_type(2);
                    triggerDTO.setFrequency(Integer.parseInt(timePoint));
                    break;
                case "D":
                    // 天
                    triggerDTO.setRepeat_type(3);
                    //兼容区间场景、每天执行的场景
                    triggerDTO.setFrequency(StringUtils.isEmpty(timePoint) ? 1 : Integer.parseInt(timePoint));
                    break;
                case "W":
                    // 周
                    triggerDTO.setWeekly(1);
                    triggerDTO.setByday(convertWeekDay(timePoint));
                    break;
                case "M":
                    // 月
                    triggerDTO.setMonthly(1);
                    triggerDTO.setBymonthday(timePoint);
                    break;
                default:
                    break;
            }
            return triggerDTO;
        } catch (DWBusinessException e) {
            log.error("knowledgegraph,getVariable 异常{},{}", mechanismParameterReqDO.getVariableCode(), e.getMessage());
        }
        return null;
    }

    private static String convertWeekDay(String weekDay) {
        switch (weekDay) {
            case "1":
                return "MO";
            case "2":
                return "TU";
            case "3":
                return "WE";
            case "4":
                return "TH";
            case "5":
                return "FR";
            case "6":
                return "SA";
            case "7":
                return "SU";
        }
        return null;
    }
}
