package com.digiwin.athena.knowledgegraph.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.knowledgegraph.domain.task.CardGroupGroup;
import com.digiwin.athena.knowledgegraph.filter.KgFilter;
import com.digiwin.athena.knowledgegraph.filter.LoginInfo;
import com.digiwin.athena.knowledgegraph.po.DapResponse;
import com.digiwin.athena.knowledgegraph.service.impl.IamService;
import com.digiwin.athena.knowledgegraph.service.model.EmailTemplateReq;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
@Slf4j
public class ATMCService {
    @Autowired
    @Qualifier("routeRestTemplate")
    RestTemplate restTemplate;

    @Value("${atmcUrl}")
    private String atmcUrl;
    @Autowired
    private IamService iamService;


    private String getDefaultLocale(){
        if(StringUtils.endsWithIgnoreCase(atmcUrl, ".cn")){
            return "zh_CN";
        }else{
            return "zh_TW";
        }
    }


    /**
     * {                //类型：Object  必有字段  备注：无
     *             "tenantId":"mock",                //类型：String  必有字段  备注：无
     *             "superadmins": - [                //类型：Array  必有字段  备注：超管用户
     *                  - {                //类型：Object  必有字段  备注：邮箱地址
     *                     "userId":"mock",                //类型：String  必有字段  备注：用户id
     *                     "userName":"mock",                //类型：String  必有字段  备注：用户名称
     *                     "email":"mock",                //类型：String  必有字段  备注：邮箱
     *                     "language":"mock"                //类型：String  必有字段  备注：偏好语系
     *                 }
     *             ],
     *             "userErrors":10,                //类型：Number  必有字段  备注：员工未绑定
     *             "deptErrors":17,                //类型：Number  必有字段  备注：员工未设定部门
     *             "dutyErrors":2                //类型：Number  必有字段  备注：必要职能未有员工数据
     *         }
     * @param param
     */
    public void noticeSuperAdminUserIntegrityCheckResult(JSONObject param){
        log.info("sendUserIntegrityCheckResultMessage param:{}", param);
        try {
            if(param == null){
                return;
            }
            String tenantId = param.getString("tenantId");
            Integer userErrors = param.getInteger("userErrors");
            Integer deptErrors = param.getInteger("deptErrors");
            Integer dutyErrors = param.getInteger("dutyErrors");
            JSONArray superadmins = param.getJSONArray("superadmins");
            if(superadmins == null || superadmins.size() == 0){
                log.info("sendUserIntegrityCheckResultMessage superadmins empty");
                return;
            }
            List<String> zh_tw = this.getUserIntegrityCheckMessage(userErrors, deptErrors, dutyErrors, "zh_TW");
            if(CollectionUtils.isEmpty(zh_tw)){
                log.info("sendUserIntegrityCheckResultMessage no errors");
                return;
            }
            List<String> zh_CN = this.getUserIntegrityCheckMessage(userErrors, deptErrors, dutyErrors, "zh_CN");

            Map<String, Object> messageSendParam = new HashMap<>();
            Map<String, Object> message = new HashMap<>();
            message.put("zhCN", zh_CN);
            message.put("zhTW", zh_tw);
            message.put("enUS", new ArrayList<>()); //暂时不给en_US的内容
            messageSendParam.put("message", message);
            messageSendParam.put("source", "UserIntegrityCheck");

            Map<String, Object> messageDetail = new HashMap<>();
            messageDetail.put("userErrorNum", userErrors);
            messageDetail.put("userErrorDatas", param.get("userErrorDatas"));
            messageDetail.put("deptErrorNum", deptErrors);
            messageDetail.put("deptErrorDatas", param.get("deptErrorDatas"));
            messageDetail.put("dutyErrorNum", dutyErrors);
            messageDetail.put("dutyErrorDatas", param.get("dutyErrorData"));
            messageSendParam.put("messageDetail", messageDetail);

            List<Object> users = new ArrayList<>();
            List<Object> emailUsers = new ArrayList<>();
            for(int i = 0; i< superadmins.size(); i++){
                JSONObject jsonObject = superadmins.getJSONObject(i);
                String language = jsonObject.getString("language");
                if(StringUtils.isEmpty(language)){
                    jsonObject.put("locale", this.getDefaultLocale());
                }else{
                    jsonObject.put("locale", language);
                }

                if(StringUtils.isNotEmpty(jsonObject.getString("email"))
                        && !StringUtils.endsWithIgnoreCase(jsonObject.getString("email"), "@digiwin.com")){  //当邮件字段为空或者非鼎捷用户的时候，排除在发邮件的报文
                    jsonObject.put("emailAddress", jsonObject.getString("email"));
                    emailUsers.add(jsonObject);
                }
//                jsonObject.put("emailAddress", jsonObject.getString("email"));
                users.add(jsonObject);
            }

            String tenantToken = this.iamService.getTenantToken(tenantId);

            try {
                if(CollectionUtils.isNotEmpty(users)){
                    messageSendParam.put("users", users);
                    this.sendIMMessage(messageSendParam, tenantToken, tenantId);
                }
            } catch (Exception e) {
                log.error("sendIMMessage error:{}", e.getMessage());
            }

            try {
                if(CollectionUtils.isNotEmpty(emailUsers)){
                    messageSendParam.put("users", emailUsers);
                    this.sendEmail(messageSendParam, tenantToken, tenantId);
                }
            } catch (Exception e) {
                log.error("sendEmail error:{}", e.getMessage());
            }
        } catch (Exception e) {
            log.error("sendUserIntegrityCheckResultMessage error:{}", e.getMessage(), e);
        }

    }

    private List<String> getUserIntegrityCheckMessage(Integer userErrors, Integer deptErrors, Integer dutyErrors, String locale){
        List<String> result = new ArrayList<>();
        if(userErrors != null && userErrors != 0){
            if(StringUtils.equalsIgnoreCase(locale, "zh_TW")){
                result.add("共"+ userErrors +"位用戶未與員工關聯");
            }else if(StringUtils.equalsIgnoreCase(locale, "zh_CN")){
                result.add("共"+ userErrors +"位用户未与员工关联");
            }else{
                result.add("A total of " + userErrors+ " users are not associated with employees");

            }
        }

        if(deptErrors != null && deptErrors != 0){
            if(StringUtils.equalsIgnoreCase(locale, "zh_TW")){
                result.add("共"+ deptErrors +"位員工未設定所屬部門");
            }else if(StringUtils.equalsIgnoreCase(locale, "zh_CN")){
                result.add("共"+ deptErrors +"位员工未设定所属部门");
            }else{
                result.add("A total of " + deptErrors+ " employees have not set their departments");

            }
        }

        if(dutyErrors != null && dutyErrors != 0){
            if(StringUtils.equalsIgnoreCase(locale, "zh_TW")){
                result.add("共"+ dutyErrors +"個職能無對應員工");
            }else if(StringUtils.equalsIgnoreCase(locale, "zh_CN")){
                result.add("共"+ dutyErrors +"个职能无对应员工");
            }else{
                result.add("There are "+ dutyErrors +" functions without corresponding employees");

            }
        }

        return result;
    }

    public List<CardGroupGroup> getGroupDimension(String routerKey) {
        String tenantToken = DWServiceContext.getContext().getToken();
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("token", tenantToken);
        if (StringUtils.isNotEmpty(routerKey)) {
            headers.add("routerKey", routerKey);
        }

        HttpEntity<Object> requestEntity = new HttpEntity<>(headers);
        log.info("getGroupDimension tenantId:{}, url:{}", routerKey, atmcUrl + "/api/atmc/v1/backlog/group/dimension");
        ResponseEntity<DapResponse> resEntity = restTemplate.exchange(atmcUrl + "/api/atmc/v1/backlog/group/dimension", HttpMethod.GET, requestEntity, DapResponse.class);
        DapResponse body = resEntity.getBody();
        log.info("getGroupDimension result:{}", body);
        List<CardGroupGroup> cardGroupGroups = JSONObject.parseArray(JSON.toJSONString(body.getResponse()), CardGroupGroup.class);

        return cardGroupGroups;
    }

    public Object sendIMMessage(Object message, String token, String routerKey){
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("token", token);
        if(StringUtils.isNotEmpty(routerKey)){
            headers.add("routerKey", routerKey);
        }

        HttpEntity<Object> requestEntity = new HttpEntity<>(message, headers);
        log.info("sendIMMessage tenantId:{}, url:{}, requestParam:{}", routerKey,atmcUrl + "/api/atmc/v1/eoc/notice/sendMessage", message);
        ResponseEntity<Object> resEntity = restTemplate.exchange(atmcUrl + "/api/atmc/v1/eoc/notice/sendMessage", HttpMethod.POST, requestEntity, Object.class);
        Object body = resEntity.getBody();
        log.info("sendIMMessage result:{}", body);
        return body;
    }

    public Object sendEmail(Object message, String token, String routerKey){
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("token", token);
        if(StringUtils.isNotEmpty(routerKey)){
            headers.add("routerKey", routerKey);
        }

        HttpEntity<Object> requestEntity = new HttpEntity<>(message, headers);
        log.info("sendEmail tenantId:{}, url:{}, requestParam:{}", routerKey, atmcUrl + "/api/atmc/v1/eoc/notice/sendEmail", message);
        ResponseEntity<Object> resEntity = restTemplate.exchange(atmcUrl + "/api/atmc/v1/eoc/notice/sendEmail", HttpMethod.POST, requestEntity, Object.class);
        Object body = resEntity.getBody();
        log.info("sendEmail result:{}", body);
        return body;
    }

    /**
     * 添加邮件模板
     * @param params 入参
     * @return 请求结果
     */
    @SuppressWarnings("findsecbugs:CRLF_INJECTION_LOGS")
    public Object addEmailTemplate(EmailTemplateReq params) {
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        String routerKey = request.getHeader("routerKey");
        String token = request.getHeader("token");
        if(null==token){
            token = (String) DWServiceContext.getContext().getRequestHeader().get("token");
        }
        if(null==token){
            LoginInfo loginInfo = KgFilter.getLoginInfo();
            if(null!=loginInfo){
                token = loginInfo.getToken();
                routerKey=loginInfo.getRouterKey();
            }
        }
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("token", token);
        if (StringUtils.isNotEmpty(routerKey)) {
            headers.add("routerKey", routerKey);
        }

        HttpEntity<Object> requestEntity = new HttpEntity<>(params, headers);
        log.info("新增邮件管理模板 tenantId:{}, url:{}, requestParam:{}", routerKey, atmcUrl + "/api/atmc/v1/email/template", JSON.toJSONString(params)); //NOSONAR 无 CRLF characters
        Object body = null;
        try {
            ResponseEntity<Object> resEntity = restTemplate.exchange(atmcUrl + "/api/atmc/v1/email/template", HttpMethod.POST, requestEntity, Object.class);
            body = resEntity.getBody();
            log.info("新增邮件管理模板 result:{}", JSON.toJSONString(body)); //NOSONAR 无 CRLF characters
        } catch (Exception e) {
            log.error("新增邮件管理模板出错：", e);
        }
        return body;
    }

    public Object cacheReset(String routerKey,String pattern){
        String token = DWServiceContext.getContext().getToken();
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        headers.add("token", token);
        if(StringUtils.isNotEmpty(routerKey)){
            headers.add("routerKey", routerKey);
        }

        HttpEntity<Object> requestEntity = new HttpEntity<>(headers);
        log.info("cacheReset tenantId:{}, url:{}, requestParam:{}", routerKey, atmcUrl + "/api/atmc/v1/cache/reset/withPattern",
                pattern);
        ResponseEntity<Object> resEntity = restTemplate.exchange(atmcUrl + "/api/atmc/v1/cache/reset/withPattern" +
                        "?pattern="+pattern,
                HttpMethod.GET, requestEntity, Object.class);
        Object body = resEntity.getBody();
        log.info("sendEmail result:{}", body);
        return body;
    }
}
