package com.digiwin.athena.knowledgegraph.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.resource.DWResourceBundleUtils;
import com.digiwin.athena.domain.common.BaseEntity;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.knowledgegraph.clients.CACUtils;
import com.digiwin.athena.knowledgegraph.dto.report.ReportSceneDTOs;
import com.digiwin.athena.knowledgegraph.service.ICommandService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.LanguageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
@MyExceptionHandler
public class CommandService implements ICommandService {
    @Autowired
    @Qualifier("knowledgegraphSystem")
    MongoTemplate mongoTemplate;
    @Autowired
    private KgInnerService kgInnerService;
    @Autowired
    private CACUtils cacUtils;
    @Autowired
    private ScreenService screenService;

    @Autowired
    private SceneService sceneService;

    @Override
    public Object getQueryCommandIntentions(String userId) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);
        List<String> appCodes = cacUtils.getAppAuth(userId);
        List<Map> commandIntentions = mongoTemplate.find(queryCriteria(appCodes, tenantVersion), Map.class, "agiledata_instruction_set");
        if (CollectionUtils.isEmpty(commandIntentions)) {
            return Collections.emptyList();
        }
        List<String> applicationCodes = commandIntentions.stream()
                .map(commandIntention -> (String) commandIntention.get("applicationCode"))
                .distinct()
                .collect(Collectors.toList());
        List<String> permissionCodes = screenService.getUserPermissionScreen(userId, Arrays.asList("scene","metric"),applicationCodes);
        // 过滤 intentions
        commandIntentions.forEach(commandIntention -> filterIntentions(commandIntention, permissionCodes));
        LanguageUtil.processLocaleLanguage(commandIntentions, DWResourceBundleUtils.getCurrentLocale().toString());
        return commandIntentions;
    }

    @Override
    public Object getDataSetCommandIntentions(String appCode) throws Exception {
        String userId = AthenaUtils.getUserId();
        String tenantVersion =kgInnerService.currentTenantVersion();
        Map application = mongoTemplate.findOne(Query.query(Criteria.where("code").is(appCode).and("version").is(tenantVersion)), Map.class, "application");
        if (Objects.isNull(application)) {
            return Collections.emptyMap();
        }
        Map<String,Object> commandIntentionInfo = new HashMap<>();
        commandIntentionInfo.put("name", application.get("name"));
        commandIntentionInfo.put("introduction", application.get("introduction"));
        commandIntentionInfo.put("prompt", application.get("prompt"));
        commandIntentionInfo.put("commandIntentions", Collections.emptyList());
        commandIntentionInfo.put("appType", application.get("appType"));
        commandIntentionInfo.put("iconUrl", application.get("iconUrl"));
        commandIntentionInfo.put("lang", application.get("lang"));
        List<String> permissionCodes = screenService.getUserPermissionScreen(userId, new ArrayList<>(), Collections.singletonList(appCode));
        if (CollectionUtils.isEmpty(permissionCodes)) {
            return commandIntentionInfo;
        }

        List<ReportSceneDTOs> dataSetList = mongoTemplate.find(Query.query(Criteria.where("version").is(tenantVersion).and("appCode").is(appCode)), ReportSceneDTOs.class, "agiledatainquiry_data_set");
        if (CollectionUtils.isEmpty(dataSetList)) {
            return commandIntentionInfo;
        }
        List<Object> commandIntentions = new ArrayList<>();
        for (ReportSceneDTOs dataSet : dataSetList) {
            if (CollectionUtils.isEmpty(dataSet.getQuestions())) {
                continue;
            }
            Map<String, Object> dataSetInfo = new HashMap<>();
            dataSetInfo.put("code", dataSet.getCode());
            dataSetInfo.put("name", dataSet.getName());
            dataSetInfo.put("questions", dataSet.getQuestions());
            dataSetInfo.put("modelCode", dataSet.getModelCode());
            dataSetInfo.put("lang", dataSet.getLang());
            commandIntentions.add(dataSetInfo);
        }

        commandIntentionInfo.put("commandIntentions", commandIntentions);
        LanguageUtil.processLocaleLanguage(commandIntentionInfo, DWResourceBundleUtils.getCurrentLocale().toString());
        return commandIntentionInfo;
    }


    @Override
    public Object getQueryAllCommandIntentions(String userId) throws Exception {
        String tenantId = AthenaUtils.getTenantId();
        String tenantVersion = kgInnerService.getTenantVersion(tenantId);
        List<String> appCodes = cacUtils.getAppAuth(userId);


//        Criteria criteria = Criteria.where("version").is(tenantVersion).and("appType").in(6,12).and("code").in(uniqueAppCodes);
//        List<Application> applications = dataPickService.find(criteria, Application.class, "application");

        List<Application> applications = mongoTemplate.find(Query.query(Criteria.where("code").in(appCodes)
                .and("version").is(tenantVersion)), Application.class, "application");
        List<String> applicationCodes = applications.stream()
                .map(BaseEntity::getCode)
                .distinct()
                .collect(Collectors.toList());
        List<Map> commandIntentions = mongoTemplate.find(queryCriteria(applicationCodes, tenantVersion), Map.class, "agiledata_instruction_set");

        List<String> permissionCodes = screenService.getUserPermissionScreen(userId, new ArrayList<>(),applicationCodes);
        // 过滤 intentions
        commandIntentions.forEach(commandIntention -> filterIntentions(commandIntention, permissionCodes));
        // 查询1.0数据集
        List<ReportSceneDTOs> reportSceneDTOList = sceneService.querySceneByAppAndTags(
                appCodes, "dataset", tenantVersion);
        commandIntentions.addAll(filterDataSetIntentions(reportSceneDTOList, permissionCodes));
        LanguageUtil.processLocaleLanguage(commandIntentions, DWResourceBundleUtils.getCurrentLocale().toString());
        return commandIntentions;
    }

    private List<Map> filterDataSetIntentions(List<ReportSceneDTOs> reportSceneDTOList, List<String> permissionCodes) {
        return reportSceneDTOList.stream()
                .filter(reportScenes -> CollectionUtils.isNotEmpty(reportScenes.getQuestions()) && permissionCodes.contains(reportScenes.getCode()))
                .map(this::convertToDataSetInfo)
                .collect(Collectors.toList());
    }

    private Map<String, Object> convertToDataSetInfo(ReportSceneDTOs reportScenes) {
        Map<String, Object> dataSetInfo = new HashMap<>();
        dataSetInfo.put("code", reportScenes.getCode());
        dataSetInfo.put("intentionGroupCode", reportScenes.getCode());
        dataSetInfo.put("intentionGroupName", reportScenes.getLang().get("sceneTitle").getOrDefault(AthenaUtils.getCurrentLocale(), reportScenes.getSceneTitle()));
        dataSetInfo.put("version", reportScenes.getVersion());
        dataSetInfo.put("applicationCode", reportScenes.getAppCode());
        dataSetInfo.put("lang", reportScenes.getLang());

        // 对问题列表进行处理
        List<Map<String, Object>> intentions = reportScenes.getQuestions().stream()
                .map(question -> {
                    Map<String, Object> intention = new HashMap<>();
                    intention.put("relationType", 1);
                    intention.put("relationName", reportScenes.getName());
                    intention.put("relationValue", reportScenes.getCode());
                    JSONObject lang = (JSONObject)question.get("lang");
                    Object value = lang.getOrDefault(AthenaUtils.getCurrentLocale(), question.get("question"));
                    intention.put("intention", value);
                    intention.put("lang", question.get("lang"));
                    intention.put("sort", 999);
                    intention.put("relationStatus", 1);
                    return intention;
                })
                .collect(Collectors.toList());

        dataSetInfo.put("intentions", intentions);
        return dataSetInfo;
    }

    private Query queryCriteria(List<String> appCode, String tenantVersion)  {
        Query query = Query.query(Criteria.where("applicationCode").in(appCode)
                .and("version").is(tenantVersion));
        return query;
    }

    private void filterIntentions(Map<String, Object> commandIntention, List<String> permissionCodes) {
        List<Map<String, Object>> intentions = (List<Map<String, Object>>) commandIntention.get("intentions");
        if (CollectionUtils.isEmpty(intentions)) {
            return;
        }
        // 只保留具有权限的 intentions
        List<Map<String, Object>> filteredIntentions = intentions.stream()
                .filter(intention -> (intention.get("relationValue") != null &&
                        permissionCodes.contains(intention.get("relationValue").toString()))
                        || (intention.get("relationValue") == null && intention.get("intention") != null))
                .collect(Collectors.toList());
        commandIntention.put("intentions", filteredIntentions);
    }
}