package com.digiwin.athena.knowledgegraph.service.impl;

import com.digiwin.app.container.exceptions.DWArgumentException;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.resource.DWResourceBundleUtils;
import com.digiwin.athena.domain.component.ComponentTypeEnum;
import com.digiwin.athena.domain.component.bo.ComponentBaseInfoBO;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.knowledgegraph.component.ComponentManager;
import com.digiwin.athena.knowledgegraph.domain.guide.GuideConfig;
import com.digiwin.athena.knowledgegraph.domain.guide.GuideMechanismReference;
import com.digiwin.athena.knowledgegraph.domain.guide.GuideSetting;
import com.digiwin.athena.knowledgegraph.domain.mechanism.Mechanism;
import com.digiwin.athena.knowledgegraph.domain.mechanism.Principle;
import com.digiwin.athena.knowledgegraph.domain.mechanism.PrincipleDetail;
import com.digiwin.athena.knowledgegraph.repoSystem.GuideConfigRepository;
import com.digiwin.athena.knowledgegraph.service.IGuideService;
import com.digiwin.athena.knowledgegraph.service.KgInnerService;
import com.digiwin.athena.knowledgegraph.service.inner.DataPickService;
import com.digiwin.athena.knowledgegraph.utils.AthenaUtils;
import com.digiwin.athena.knowledgegraph.utils.I18nUtils;
import com.digiwin.athena.knowledgegraph.utils.LanguageUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;

/**
 * @program: athena_backend
 * @description: 新手引导服务
 * @author: lingxiang
 * @create: 2022-05-18 16:00
 **/
@Lang
@Slf4j
@Service
@MyExceptionHandler
public class GuideService implements IGuideService {

    @Autowired
    private GuideConfigRepository guideConfigRepository;
    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate mongoTemplateSys;//系统库
    @Autowired
    KgInnerService kgInnerService;
    @Autowired
    private MechanismService mechanismService;
    @Autowired
    private ComponentManager componentManager;
    @Autowired
    private DataPickService dataPickService;

    @Override
    public Object put(GuideConfig guideConfig) throws DWBusinessException {
        if (StringUtils.isEmpty(guideConfig.getAppCode())) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.appCodeEmpty"));
        }
        if (StringUtils.isEmpty(guideConfig.getCode())) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.codeEmpty"));
        }
        if (StringUtils.isEmpty(guideConfig.getType())) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.typeEmpty"));
        }
        String version = guideConfig.getVersion();
        if (StringUtils.isEmpty(version)) {//没有传 开发版本 则使用当前登录租户的开发版本
            version = kgInnerService.getTenantVersion(AthenaUtils.getTenantId());
            guideConfig.setVersion(version);
        }

        try {
            GuideConfig re = guideConfigRepository.findOneByAppCodeAndCodeAndTypeAndVersion(guideConfig.getAppCode(), guideConfig.getCode(), guideConfig.getType(), version);
            if (re == null) {//不存在 则新增
                guideConfigRepository.insert(guideConfig);
            } else {//已存在 则以当前入参版本更新
                re.setSettings(guideConfig.getSettings());
                guideConfigRepository.save(re);
            }
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }
        return null;
    }

    @Override
    public Object delete(String appCode, String code, String type, String version) throws DWBusinessException {
        if (StringUtils.isEmpty(appCode)) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.appCodeEmpty"));
        }
        if (StringUtils.isEmpty(code)) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.codeEmpty"));
        }
        if (StringUtils.isEmpty(type)) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.typeEmpty"));
        }
        String tenantId = AthenaUtils.getTenantId();
        if (StringUtils.isEmpty(version)) {//没有传 开发版本 则使用当前登录租户的开发版本
            version = kgInnerService.getTenantVersion(tenantId);
        }
        //删除条件
        Criteria criteria = new Criteria();
        criteria.and("appCode").is(appCode).and("code").is(code).and("type").is(type).and("version").is(version);
        dataPickService.addTenantCondition(criteria, appCode, tenantId);
        try {
            mongoTemplateSys.remove(new Query(criteria), GuideConfig.class);
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }
        return null;
    }

    @Override
    public Object post(GuideConfig guideConfig) throws DWBusinessException {
        if (StringUtils.isEmpty(guideConfig.getAppCode())) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.appCodeEmpty"));
        }
        if (StringUtils.isEmpty(guideConfig.getCode())) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.codeEmpty"));
        }
        if (StringUtils.isEmpty(guideConfig.getType())) {
            throw new DWArgumentException("guideConfig", I18nUtils.getValue("knowledgegraph.guideConfig.typeEmpty"));
        }
        String version = guideConfig.getVersion();
        if (StringUtils.isEmpty(version)) {//没有传 开发版本 则使用当前登录租户的开发版本
            version = kgInnerService.getTenantVersion(AthenaUtils.getTenantId());
        }

        //更新条件
        Criteria criteria = new Criteria();
        criteria.and("appCode").is(guideConfig.getAppCode()).and("code").is(guideConfig.getCode()).and("type").is(guideConfig.getType()).and("version").is(version);
        dataPickService.addTenantCondition(criteria, guideConfig.getAppCode(), AthenaUtils.getTenantId());
        //更新字段
        Update update = new Update();
        update.set("settings", guideConfig.getSettings());//更新引导内容

        try {
            mongoTemplateSys.upsert(new Query(criteria), update, GuideConfig.class);//不存在则新增
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }
        return null;
    }

    @Override
    public Object postQuery(String appCode, String code, String type) throws DWBusinessException {
        //使用当前登录租户的开发版本
        String version = kgInnerService.getTenantVersion(AthenaUtils.getTenantId());
        GuideConfig re;
        try {
            //条件
            Criteria criteria = new Criteria();
            criteria.and("appCode").is(appCode).and("code").is(code).and("type").is(type);
            re = dataPickService.findOneByCondition(criteria, GuideConfig.class, "guideConfig");

            if (re != null) {
                //筛选组件
                filterByComponentManager(re);

                //处理引用机制原理
                processReferMechismPrinceple(re);

                //处理多语言
                Locale locale = DWResourceBundleUtils.getCurrentLocale();
                String localeStr = locale.toString();
                LanguageUtil.processLocaleLanguage(re, localeStr);
            }
        } catch (Exception e) {
            throw new DWBusinessException(e.getLocalizedMessage());
        }
        return re;
    }

    public void filterByComponentManager(GuideConfig config) throws DWBusinessException {
        List<GuideSetting> list = config.getSettings();
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(list)) {
            Map<String, GuideSetting> map = new HashMap<>();
            List<ComponentBaseInfoBO> componentBaseInfos = new ArrayList<>();
            for (GuideSetting g : list) {
                map.put(g.getCode(), g);
                ComponentBaseInfoBO info = new ComponentBaseInfoBO();
                info.setCode(g.getCode());
                info.setType(ComponentTypeEnum.Guidance);
                componentBaseInfos.add(info);
            }
            //过滤
            List<ComponentBaseInfoBO> filterComponents = componentManager.FilterByTenantApplicationComponentList(config.getAppCode(), AthenaUtils.getTenantId(), componentBaseInfos);
            List<GuideSetting> filterSettings = new ArrayList<>();
            if (org.apache.commons.collections.CollectionUtils.isNotEmpty(filterComponents)) {//过滤掉所有数据
                if (list.size() == filterComponents.size()) {//数量相同无需过滤
                    filterSettings = list;
                } else {
                    for (ComponentBaseInfoBO c : filterComponents) {
                        filterSettings.add(map.get(c.getCode()));
                    }
                }
            }
            config.setSettings(filterSettings);
        }
    }

    public void processReferMechismPrinceple(GuideConfig config) throws DWBusinessException {
        if (!CollectionUtils.isEmpty(config.getSettings())) {
            for (GuideSetting setting : config.getSettings()) {
                GuideMechanismReference ref = setting.getReference();
                if (ref != null && !ref.isEmpty()) {
                    Mechanism m = (Mechanism) mechanismService.postQueryMechanismByCode(ref.getMechanismCode(), null);
                    if (m != null && !CollectionUtils.isEmpty(m.getPrinciples())) {
                        for (Principle p : m.getPrinciples()) {
                            if (StringUtils.equals(ref.getPrincipleCode(), p.getCode())) {
                                if (!CollectionUtils.isEmpty(p.getDetails())) {
                                    for (PrincipleDetail d : p.getDetails()) {
                                        if (StringUtils.equals(ref.getDetailCode(), d.getCode())) {
                                            String c = d.getContent();
                                            Map<String, String> cmap = d.getLang().get("content");
                                            setContent(c, cmap, ref.getLineNo(), setting);
                                            break;
                                        }
                                    }
                                }
                                break;
                            }
                        }
                    }
                }
            }
        }
    }

    public void setContent(String c, Map<String, String> cmap, List<Integer> lineNo, GuideSetting setting) {
        if (!CollectionUtils.isEmpty(lineNo)) {
            c = processLineNo(c, lineNo);
            for (Map.Entry<String, String> entry : cmap.entrySet()) {
                entry.setValue(processLineNo(entry.getValue(), lineNo));
            }
        }

        setting.setContent(c);
        if (setting.getLang() != null) {
            setting.getLang().put("content", cmap);
        } else {
            Map<String, Map<String, String>> lang = new HashMap<>();
            lang.put("content", cmap);
            setting.setLang(lang);
        }
    }

    public String processLineNo(String c, List<Integer> lineNo) {
        String spliter = "\r\n";
        if (!c.contains(spliter)) {//如果找不到 \r\n 替换为 \n分隔 保存过的数据以\n分隔
            spliter = "\n";
        }
        String[] cs = c.split(spliter);
        StringBuilder tmp = new StringBuilder();
        for (Integer i : lineNo) {
            if (i <= cs.length) {
                tmp.append(cs[i-1]+"\r\n");
            }
        }
        return tmp.toString();
    }
}
