package com.digiwin.athena.knowledgegraph.service.impl;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.neo4j.Neo4j1Config;
import com.digiwin.athena.kmservice.neo4j.Neo4j2Config;
import com.digiwin.athena.knowledgegraph.domain.IndicatorNode;
import com.digiwin.athena.knowledgegraph.domain.targets.SystemIndicatorForestDto;
import com.digiwin.athena.knowledgegraph.domain.targets.SystemIndicatorTreeNodeRelation;
import com.digiwin.athena.repository.neo4j.IndicatorRepository;
import com.digiwin.athena.knowledgegraph.service.ISystemTargetService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.neo4j.ogm.session.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import javax.validation.*;
import java.util.*;
import java.util.stream.Collectors;

@Lang
@Service
@Slf4j
public class SystemTargetService implements ISystemTargetService {

    @Autowired
    IndicatorRepository indicatorRepository;

    @Autowired(required = false)
    @Qualifier(Neo4j1Config.SESSION_FACTORY)
    SessionFactory sessionFactory1;

    @Autowired(required = false)
    @Qualifier(Neo4j2Config.SESSION_FACTORY)
    SessionFactory sessionFactory2;

    @Override
    public Object deleteIndicatorForest() throws DWBusinessException {
        //清空旧的forest
        if (null != sessionFactory1) {
            sessionFactory1.openSession().deleteAll(IndicatorNode.class);
        }
        if (null != sessionFactory2) {
            sessionFactory2.openSession().deleteAll(IndicatorNode.class);
        }
        return null;
    }

    @Override
    public Object getIndicatorForest() throws DWBusinessException {
        List<IndicatorNode> roots = indicatorRepository.findRootIndicatorNodes();
        if (CollectionUtils.isEmpty(roots)) {
            return Collections.EMPTY_LIST;
        }
        List<String> rootIndicatorIds = roots.stream().map(IndicatorNode::getIndicatorId).collect(Collectors.toList());
        List<IndicatorNode> indicatorNodes = indicatorRepository.findAllByIndicatorIdIn(rootIndicatorIds);
        List<SystemIndicatorTreeNodeRelation> indicatorRelations = new ArrayList<>();
        indicatorNodes.forEach(item -> {
            SystemIndicatorTreeNodeRelation nodeRelation = buildRelationByNodes(null, item);
            if (!indicatorRelations.contains(nodeRelation)) {
                indicatorRelations.add(nodeRelation);
            }
            if (!CollectionUtils.isEmpty(item.getConsistsOf())) {
                traverseRecursively(item, indicatorRelations);
            }
        });
        SystemIndicatorForestDto systemIndicatorForestDto = new SystemIndicatorForestDto();
        systemIndicatorForestDto.setIndicatorRelations(indicatorRelations);
        return systemIndicatorForestDto;
    }

    private List<IndicatorNode> findForest() {
        List<IndicatorNode> roots = indicatorRepository.findRootIndicatorNodes();
        if (CollectionUtils.isEmpty(roots)) {
            return Collections.EMPTY_LIST;
        }
        List<String> rootIndicatorIds = roots.stream().map(IndicatorNode::getIndicatorId).collect(Collectors.toList());
        List<IndicatorNode> indicatorNodes = indicatorRepository.findAllByIndicatorIdIn(rootIndicatorIds);
        return indicatorNodes;
    }

    private void traverseRecursively(IndicatorNode parent, List<SystemIndicatorTreeNodeRelation> indicatorRelations) {
        if (!CollectionUtils.isEmpty(parent.getConsistsOf())) {
            parent.getConsistsOf().forEach(item -> {
                SystemIndicatorTreeNodeRelation nodeRelation = buildRelationByNodes(parent, item);
                if (!indicatorRelations.contains(nodeRelation)) {
                    indicatorRelations.add(nodeRelation);
                }
                traverseRecursively(item, indicatorRelations);
            });
        }
    }

    private SystemIndicatorTreeNodeRelation buildRelationByNodes(IndicatorNode parent, IndicatorNode child) {
        SystemIndicatorTreeNodeRelation nodeRelation = new SystemIndicatorTreeNodeRelation();
        if (null == child) {
            throw new IllegalArgumentException("child cannot be null");
        }
        nodeRelation.setIndicatorId(child.getIndicatorId());
        nodeRelation.setGroup(child.getGroup());
        nodeRelation.setFeatureSelection(child.getFeatureSelection());
        nodeRelation.setName(child.getName());
        nodeRelation.setTendency(child.getTendency());
        nodeRelation.setUnit(child.getUnit());
        nodeRelation.setParentId(null == parent ? null : parent.getIndicatorId());
        return nodeRelation;
    }

    @Override
    public Object postIndicatorForest(@Valid SystemIndicatorForestDto systemIndicatorForest) throws DWBusinessException {
        ValidatorFactory validatorFactory = Validation.buildDefaultValidatorFactory();
        Validator validator = validatorFactory.getValidator();
        Set<ConstraintViolation<SystemIndicatorForestDto>> violations = validator.validate(systemIndicatorForest);
        List<String> errorMessages = new LinkedList<>();
        for (ConstraintViolation<SystemIndicatorForestDto> violation : violations) {
            errorMessages.add(violation.getPropertyPath() + violation.getMessage() + ";");

        }
        if (!CollectionUtils.isEmpty(errorMessages)) {
            throw new DWBusinessException(errorMessages.toString());
        }
        //清空旧的forest
        //indicatorRepository.deleteAll();
        //构建森林结构
        List<SystemIndicatorTreeNodeRelation> indicatorRelations = systemIndicatorForest.getIndicatorRelations();
        List<IndicatorNode> forestInNeo4j = findForest();
        List<IndicatorNode> forest = buildForestByRelations(forestInNeo4j, indicatorRelations);
        if (!CollectionUtils.isEmpty(forest)) {
            forest.forEach(indicatorNode -> {
                if (null != sessionFactory1) {
                    sessionFactory1.openSession().save(indicatorNode);
                }
                if (null != sessionFactory2) {
                    sessionFactory2.openSession().save(indicatorNode);
                }
            });
        }
        return null;
    }

    private List<IndicatorNode> buildForestByRelations(List<IndicatorNode> forestInNeo4j, List<SystemIndicatorTreeNodeRelation> indicatorRelations) {
        Map<String, IndicatorNode> indicatorMap = buildMapByForest(forestInNeo4j);
        if (!CollectionUtils.isEmpty(indicatorRelations)) {
            //初始化,新值会覆盖旧值
            indicatorRelations.forEach(item -> {
                if (!indicatorMap.keySet().contains(item.getIndicatorId())) {
                    //初始化node
                    IndicatorNode indicatorNode = new IndicatorNode();
                    indicatorNode.setIndicatorId(item.getIndicatorId());
                    indicatorNode.setName(item.getName());
                    indicatorNode.setGroup(item.getGroup());
                    indicatorNode.setTendency(item.getTendency());
                    indicatorNode.setUnit(item.getUnit());
                    indicatorNode.setFeatureSelection(item.getFeatureSelection());
                    indicatorMap.put(item.getIndicatorId(), indicatorNode);
                } else {
                    IndicatorNode indicatorNode = indicatorMap.get(item.getIndicatorId());
                    //更新指标属性
                    indicatorNode.setName(item.getName());
                    indicatorNode.setGroup(item.getGroup());
                    indicatorNode.setTendency(item.getTendency());
                    indicatorNode.setUnit(item.getUnit());
                    indicatorNode.setFeatureSelection(item.getFeatureSelection());
                }
            });
            List<IndicatorNode> rootIndicatorNodes = new LinkedList<>();
            rootIndicatorNodes.addAll(forestInNeo4j);
            //建立关联
            indicatorRelations.forEach(item -> {
                if (!StringUtils.isEmpty(item.getParentId())) {
                    if (indicatorMap.keySet().contains(item.getParentId())) {
                        //为父节点添加子节点
                        indicatorMap.get(item.getParentId()).addRelation(indicatorMap.get(item.getIndicatorId()));
                    }
                } else {
                    rootIndicatorNodes.add(indicatorMap.get(item.getIndicatorId()));
                }
            });
            return rootIndicatorNodes;
        }
        return Collections.EMPTY_LIST;
    }

    private Map<String, IndicatorNode> buildMapByForest(List<IndicatorNode> forestInNeo4j) {
        Map<String, IndicatorNode> indicatorNodeMap = new HashMap<>();
        if (CollectionUtils.isEmpty(forestInNeo4j)) {
            return indicatorNodeMap;
        }
        forestInNeo4j.forEach(item -> {
            indicatorNodeMap.put(item.getIndicatorId(), item);
            if (!CollectionUtils.isEmpty(item.getConsistsOf())) {
                traverseAndPutMap(item, indicatorNodeMap);
            }
        });
        return indicatorNodeMap;
    }

    private void traverseAndPutMap(IndicatorNode parent, Map<String, IndicatorNode> indicatorNodeMap) {
        parent.getConsistsOf().forEach(child -> {
            indicatorNodeMap.put(child.getIndicatorId(), child);
            if (!CollectionUtils.isEmpty(child.getConsistsOf())) {
                traverseAndPutMap(child, indicatorNodeMap);
            }
        });
    }
}
