package com.digiwin.athena.knowledgegraph.synonym.dao;


import com.digiwin.athena.knowledgegraph.synonym.domain.BusinessType;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.DeleteVo;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.PageVo;
import com.digiwin.athena.knowledgegraph.synonym.exception.BusinessException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;

/**
 * @Author: zjz
 * @CreateTime: 2024-03-26  13:08
 * @Description:
 * @Version: 1.0
 */
@Repository
public class BusinessTypeDao {
   @Autowired
    @Qualifier("knowledgegraphTenant")
    private MongoTemplate mongoTemplate;
    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate systemMongoTemplate;

    private static final String APP_CODE = "appCode";

    private static final String TENANT_ID = "tenantId";

    public void batchInsert(List<BusinessType> businessTypes){
        mongoTemplate.insertAll(businessTypes);
    }

    public void update(BusinessType businessType){
        businessType.setEditTime(new Date());
        mongoTemplate.save(businessType);
    }

    public void batchDelete(DeleteVo deleteVo){
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.and(TENANT_ID).is(deleteVo.getTid());
        criteria.and(APP_CODE).is(deleteVo.getAppCode());
        if (!CollectionUtils.isEmpty(deleteVo.getIds())) {
            criteria.and("id").in(deleteVo.getIds());
        }
        query.addCriteria(criteria);
        this.mongoTemplate.remove(query, BusinessType.class);
    }

    public List<BusinessType> findList(String appCode, String tenantId){
        if (StringUtils.isEmpty(tenantId)){
            throw new BusinessException("租户id不能为空");
        }
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where(TENANT_ID).is(tenantId));
        criteria.and(APP_CODE).is(appCode);
        query.addCriteria(criteria);
        return this.mongoTemplate.find(query, BusinessType.class);
    }

    public PageVo<BusinessType> getPage(String appCode, String tid, int pageNum, int pageSize) {

        if (StringUtils.isEmpty(tid)){
            throw new BusinessException("租户id不能为空");
        }
        if (pageSize==0){
            pageSize = 10;
        }
        pageNum = pageNum-1;
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.and(TENANT_ID).is(tid);
        criteria.and(APP_CODE).is(appCode);
        query.addCriteria(criteria);
        Pageable pageable = PageRequest.of(pageNum, pageSize, Sort.Direction.DESC, "editTime");
        long totalCount = mongoTemplate.count(query, BusinessType.class);

        List<BusinessType> businessTypes = mongoTemplate.find(query.with(pageable), BusinessType.class);
        int totalPage = (int) (totalCount % pageSize == 0 ? totalCount / pageSize : totalCount / pageSize + 1);

        return new PageVo<>((int) totalCount,businessTypes,pageNum,pageSize,totalPage);
    }

    public List<BusinessType> querySystemBusinessTypeByVersion(String appCode, String tenantVersion){
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("version").is(tenantVersion));
        criteria.and(APP_CODE).is(appCode);
        query.addCriteria(criteria);
        return this.systemMongoTemplate.find(query, BusinessType.class);
    }

    public BusinessType querySystemBusinessType(String appCode, String code, String tenantVersion){
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where(APP_CODE).is(appCode));
        criteria.and("code").is(code);
        criteria.and("version").is(tenantVersion);
        query.addCriteria(criteria);
        return this.systemMongoTemplate.findOne(query, BusinessType.class);
    }
}
