package com.digiwin.athena.knowledgegraph.synonym.dao;

import com.digiwin.athena.knowledgegraph.synonym.domain.BusinessVariables;
import com.digiwin.athena.knowledgegraph.synonym.domain.BusinessVariablesValue;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.DeleteVo;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.PageVo;
import com.digiwin.athena.knowledgegraph.synonym.enums.SourceEnum;
import com.digiwin.athena.knowledgegraph.synonym.exception.BusinessException;
import com.mongodb.bulk.BulkWriteResult;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.data.util.Pair;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * @Author: wangao
 * @CreateTime: 2024-08-28  11:17
 * @Description:
 * @Version: 1.0
 */
@Repository
public class BusinessVariablesDao {
    @Autowired
    @Qualifier("knowledgegraphTenant")
    private MongoTemplate mongoTemplate;
    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate systemMongoTemplate;

    public void insertBusinessVariables(List<BusinessVariablesValue> businessVariables) {
        mongoTemplate.insertAll(businessVariables);
    }

    public void updateBusinessVariables(BusinessVariablesValue businessVariables) {
        if(StringUtils.isEmpty(businessVariables.getSource())){
            businessVariables.setSource(SourceEnum.CUSTOM.getType());
        }
        mongoTemplate.save(businessVariables);
    }
    public int saveBusinessVariables(List<BusinessVariablesValue> list){
        List<Pair<Query, Update>> updateList = new ArrayList<>(list.size());
        BulkOperations operations = mongoTemplate.bulkOps(BulkOperations.BulkMode.UNORDERED, "agiledata_business_variables_value");
        list.forEach(data -> {
            //如果query查询到有数据就更新
            Query query = new Query(new Criteria("_id").is(data.getId()));
            Update update = new Update();
            update.set("source",data.getSource());

            Pair<Query, Update> updatePair = Pair.of(query, update);
            updateList.add(updatePair);
        });
        operations.upsert(updateList);
        BulkWriteResult result = operations.execute();
        return result.getModifiedCount();
    }
    public void deleteBusinessVariables(DeleteVo deleteVo) {
        if (StringUtils.isEmpty(deleteVo.getTid())){
            throw new BusinessException("租户id不能为空");
        }
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.and("tenantId").is(deleteVo.getTid());
        criteria.and("appCode").is(deleteVo.getAppCode());
        if (!CollectionUtils.isEmpty(deleteVo.getIds())) {
            criteria.and("id").in(deleteVo.getIds());
        }
        query.addCriteria(criteria);
        this.mongoTemplate.remove(query, BusinessVariablesValue.class);
    }

    public PageVo<BusinessVariablesValue> queryBusinessVariables(String appCode,String tid, int pageNum, int pageSize) {

        if (StringUtils.isEmpty(tid)){
            throw new BusinessException("租户id不能为空");
        }
        if (pageSize==0){
            pageSize = 10;
        }
        pageNum = pageNum-1;
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(tid));
        criteria.and("appCode").is(appCode);
        criteria.and("source").ne(SourceEnum.SYSTEM.getType());
        query.addCriteria(criteria);
        Pageable pageable = PageRequest.of(pageNum, pageSize,Sort.Direction.DESC, "data_name");
        long totalCount = mongoTemplate.count(query, BusinessVariablesValue.class);

        List<BusinessVariablesValue> entityTypes = this.mongoTemplate.find(query.with(pageable), BusinessVariablesValue.class);
        int totalPage = (int) (totalCount % pageSize == 0 ? totalCount / pageSize : totalCount / pageSize + 1);

        return new PageVo((int) totalCount,entityTypes,pageNum,pageSize,totalPage);
    }
    public List<BusinessVariablesValue> queryBusinessVariablesValue(String tid,String appCode) {
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(tid));
        criteria.and("appCode").is(appCode);
        query.addCriteria(criteria);
        return this.mongoTemplate.find(query, BusinessVariablesValue.class);
    }
    public List<BusinessVariablesValue> queryAll() {
        Query query = new Query();
        Criteria criteria = new Criteria();
        query.addCriteria(criteria);
        return this.mongoTemplate.find(query, BusinessVariablesValue.class);
    }
    public List<BusinessVariables> queryBusinessVariables(String appCode, String tenantVersion) {
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("version").is(tenantVersion));
        criteria.and("appCode").is(appCode);
        query.addCriteria(criteria);
        return this.systemMongoTemplate.find(query, BusinessVariables.class);
    }
}
