package com.digiwin.athena.knowledgegraph.synonym.dao;


import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.knowledgegraph.synonym.domain.LocalAppInfo;
import com.digiwin.athena.knowledgegraph.synonym.domain.Synonym;
import com.digiwin.athena.knowledgegraph.synonym.domain.po.PageVo;
import com.digiwin.athena.knowledgegraph.synonym.exception.BusinessException;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.aggregation.GroupOperation;
import org.springframework.data.mongodb.core.aggregation.MatchOperation;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Author: wangao
 * @CreateTime: 2024-03-26  13:08
 * @Description: TODO
 * @Version: 1.0
 */
@Repository
public class SynonymDao {
    @Autowired
    @Qualifier("knowledgegraphTenant")
    private MongoTemplate mongoTemplate;
    @Autowired
    @Qualifier("knowledgegraphSystem")
    private MongoTemplate systemMongoTemplate;

    public void insertOrUpdate(Synonym synonym){
        if (synonym==null){
            return;
        }
        synonym.setEditDate(new Date());
        mongoTemplate.save(synonym);
    }

    public void saveAll(List<Synonym> synonyms){
        if (CollectionUtils.isEmpty(synonyms)){
            return;
        }
        //先删除再新增
        List<String> collect = synonyms.stream().map(Synonym::getObjectId).collect(Collectors.toList());
        delete(collect);
        insertAll(synonyms);
    }
    public void insertAll(List<Synonym> synonyms){
        mongoTemplate.insertAll(synonyms);
    }
    public PageVo findListPage(String tenantId, String properNoun, String entityTypeName, String des, int page, int pageSize){
        if (StringUtils.isEmpty(tenantId)){
            throw new BusinessException("租户id不能为空");
        }
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(tenantId));
        if (StringUtils.isNotEmpty(des)){
            criteria.and("description").regex(des);
        }
        if (StringUtils.isNotEmpty(properNoun)){
            criteria.orOperator(Criteria.where("properNoun").regex(properNoun),Criteria.where("zh_CN_Str").regex(properNoun),Criteria.where("zh_TW_Str").regex(properNoun));
        }
        if (StringUtils.isNotEmpty(entityTypeName)){
            criteria.and("entityTypeName").regex(entityTypeName);
        }
        query.addCriteria(criteria);
        Pageable pageable = PageRequest.of(page, pageSize);
        long totalCount = mongoTemplate.count(query,Synonym.class);

        List<Synonym> synonyms = this.mongoTemplate.find(query.with(pageable), Synonym.class);
        int totalPage = (int) (totalCount % pageSize == 0 ? totalCount / pageSize : totalCount / pageSize + 1);

        return new PageVo((int) totalCount,synonyms,page,pageSize,totalPage);
    }
    public Map<String,Synonym> findMap(String tenantId, List<String> ids){
        if (StringUtils.isEmpty(tenantId)){
            throw new BusinessException("租户id不能为空");
        }
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(tenantId));
        if (!CollectionUtils.isEmpty(ids)){
            criteria.andOperator(Criteria.where("_id").in(ids));
        }
        query.addCriteria(criteria);
        List<Synonym> synonyms = this.mongoTemplate.find(query, Synonym.class);
        HashMap<String, Synonym> map = new HashMap<>();
        for (Synonym synonym:synonyms){
            map.put(synonym.getObjectId(),synonym);
        }
        return map;
    }
    public List<Synonym> findList(String tenantId, String properNoun, String entityTypeName, String des){
        if (StringUtils.isEmpty(tenantId)){
            throw new BusinessException("租户id不能为空");
        }
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(tenantId));
        if (StringUtils.isNotEmpty(des)){
            criteria.and("description").regex(des);
        }
        if (StringUtils.isNotEmpty(properNoun)){
            criteria.orOperator(Criteria.where("properNoun").regex(properNoun),Criteria.where("zh_CN_Str").regex(properNoun),Criteria.where("zh_TW_Str").regex(properNoun));
        }
        if (StringUtils.isNotEmpty(entityTypeName)){
            criteria.and("entityTypeName").regex(entityTypeName);
        }
        query.addCriteria(criteria);
        return this.mongoTemplate.find(query, Synonym.class);
    }
    public List<Synonym> findList(String tenantId, List<String> ids){
        if (StringUtils.isEmpty(tenantId)){
            throw new BusinessException("租户id不能为空");
        }
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(tenantId));
        if (!CollectionUtils.isEmpty(ids)){
            criteria.and("_id").in(ids);
        }
        query.addCriteria(criteria);
        return this.mongoTemplate.find(query, Synonym.class);
    }
    public Synonym findOne(String tenantId,String properNoun){
        if (StringUtils.isEmpty(tenantId)||StringUtils.isEmpty(properNoun)){
            throw new BusinessException("租户id/专有名词/实体类型不能为空");
        }
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.andOperator(Criteria.where("tenantId").is(tenantId),Criteria.where("properNoun").is(properNoun));
        query.addCriteria(criteria);
        return this.mongoTemplate.findOne(query, Synonym.class);
    }

    public void delete(List<String> list, String tenantId){
        if (CollectionUtils.isEmpty(list)){
            return;
        }
        if (StringUtils.isEmpty(tenantId)){
            throw new BusinessException("租户id不能为空");
        }
        Query query = new Query(Criteria.where("objectId").in(list).andOperator(Criteria.where("tenantId").is(tenantId)));
        this.mongoTemplate.remove(query,Synonym.class);
    }
    public void delete(List<String> list){
        if (CollectionUtils.isEmpty(list)){
            return;
        }
        Query query = new Query(Criteria.where("objectId").in(list));
        this.mongoTemplate.remove(query,Synonym.class);
    }

    public Synonym getOne(String tenantId, String properNoun, String entityTypeName) {
        Query query = new Query();
        Criteria criteria = new Criteria();
        criteria.and("tenantId").is(tenantId).and("properNoun").is(properNoun).and("entityTypeName").is(entityTypeName);
        query.addCriteria(criteria);
        Synonym one = this.mongoTemplate.findOne(query, Synonym.class);
        return one;
    }

    public List<Synonym> getTwIsNotNullList(){
        Query query = new Query(Criteria.where("zh_TW_Str").exists(true).ne(""));
        return this.mongoTemplate.find(query,Synonym.class);
    }
    @Async
    public void dataMigration(){
        long count = systemMongoTemplate.count(new Query(), Synonym.class);
        if (count > 0){
            int totalPage = (int) (count/1000);
            for (int i = 0; i <=totalPage ; i++) {
                Pageable pageable = PageRequest.of(i, 1000);
                List<Synonym> synonyms = this.systemMongoTemplate.find(new Query().with(pageable), Synonym.class);
                if (!CollectionUtils.isEmpty(synonyms)){
                    mongoTemplate.insertAll(synonyms);
                }
            }
        }
    }

    public List<Synonym> findAll(int page, int pageSize) {
        Pageable pageable = PageRequest.of(page, pageSize);
        List<Synonym> synonyms = this.mongoTemplate.find(new Query().with(pageable), Synonym.class);
        return synonyms;
    }

    public JSONObject groupByEntityCount(String tenantId) {
        JSONObject jsonObject = new JSONObject();
        MatchOperation matchOperation = Aggregation.match(Criteria.where("tenantId").is(tenantId));

        GroupOperation groupOperation = Aggregation.group("entityTypeName").count().as("count");

        Aggregation aggregation = Aggregation.newAggregation(matchOperation,groupOperation);

        AggregationResults<Map> results = this.mongoTemplate.aggregate(aggregation, "synonym", Map.class);

        List<Map> mappedResults = results.getMappedResults();

        if (CollectionUtils.isEmpty(mappedResults)){
            return jsonObject;
        }else {
            for (Map mappedResult : mappedResults) {
                jsonObject.put(mappedResult.get("_id").toString(), mappedResult.get("count"));
            }
        }
        return jsonObject;
    }
}
