package com.digiwin.metadatacache.apiservice;

import com.digiwin.app.service.DWServiceContext;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.dao.ApiRelationDao;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.exception.TokenNotRetrievedException;
import com.digiwin.metadatacache.model.ApiRelation;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.services.LocalCacheService;
import com.digiwin.metadatacache.util.IamVerifyServiceUtil;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Map;

/**
 * 取得API映射关系
 * Path:[/restful/standard/mdc/ApiRelation/Delete]
 */
@Service
public class ApiRelationDeleteService extends AbstractApiService {

	@Autowired
	private ApiRelationDao apiRelationDao;

	@Autowired
	private LocalCacheService localCacheService;

	public ApiRelationDeleteService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.API_RELATION_DELETE_SCHEMA;
	}

	/**
	 * 删除映射关系信息
	 * @param validatorResult validatorResult
	 * @return map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		// 組成回傳訊息
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
				+ ", 原始訊息: " + tRequestJsonNode);
		// 1、获取token信息
		String tToken = (String) DWServiceContext.getContext().getRequestHeader().get(MdcConstant.TOKEN);
		if (null == tToken) {
			throw new TokenNotRetrievedException(getClass());
		}
		String userName = IamVerifyServiceUtil.getUserName(tToken);
		// 2、删除对应的关联关系
		String apiName = tRequestJsonNode.get(MdcConstant.API_NAME).asText();
		String tenantId = null;
		if (null != tRequestJsonNode.get(MdcConstant.TENANT_ID)
		      && !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.TENANT_ID).asText())) {
			tenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
		}
		deleteApiRelation(userName, apiName, tenantId);
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription,null);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 删除关联关系
	 * @param userName  用户名称
	 * @param apiName   Api名称
	 * @param tenantId  租户Id
	 */
	public void deleteApiRelation(String userName, String apiName, String tenantId) {
		// 3、查询对应的关联关系，并删除
		ApiRelation apiRelation = apiRelationDao.getRelationBySubInfo(apiName, tenantId);
		if (null != apiRelation) {
			apiRelation.setUpdateId(userName);
			apiRelation.setDeleteFlag(1);
			apiRelation.setUpdateTime(LocalDateTime.now());
			apiRelationDao.saveOrupdate(apiRelation);
			localCacheService.remove(CacheMapTypeEnum.api_relation_idempotency_cachemap, apiRelation.getSubApiName());
		}
	}
}
