package com.digiwin.metadatacache.apiservice;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.util.CacVerifyServiceUtil;
import com.digiwin.metadatacache.util.IamVerifyServiceUtil;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.digiwin.app.service.DWServiceContext;
import com.digiwin.metadatacache.dao.EocIntgMappingDao;
import com.digiwin.metadatacache.dao.TenantEocSetInfoDao;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.EocIntgMapping;
import com.digiwin.metadatacache.model.TenantEocSetInfo;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;

/**
 * 检查该租户是否需要被提醒配置运营单元
 * Path:[/restful/standard/mdc/tenant/EocSetInfo]
 */
@org.springframework.stereotype.Service
public class EocSetInfoCheckService extends AbstractApiService {

    @Autowired
    private TenantEocSetInfoDao tenantEocSetInfoDao;

    @Autowired
    private EocIntgMappingDao eocIntgMappingDao;

    @Autowired
    private ProductDao productDao;

    public EocSetInfoCheckService() {
        super();
        jsonSchemaFileName = JsonSchemaFileConstant.EOC_SET_INFO_CHECK_SCHEMA;
    }

    /**
     * 提醒运营单元配置设置查询
     *
     * @param validatorResult * @param validatorResult
     * @return map
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
                + ", 原始訊息: " + tRequestJsonNode);
        return checkEocSetInfo(tRequestJsonNode);
    }

    private Map<String, Object> checkEocSetInfo(JsonNode pMessageNode) throws Exception {
        Map<String, Object> tReturn = new HashMap<>();
        String tToken = DWServiceContext.getContext().getRequestHeader().get(MdcConstant.TOKEN).toString();
        String tAction = null;
        if (pMessageNode.get(MdcConstant.ACTION) != null) {
            tAction = pMessageNode.get(MdcConstant.ACTION).asText();
        }
        String tenantId = pMessageNode.get(MdcConstant.TENANT_ID).asText();
        String userId = pMessageNode.get(MdcConstant.USER_ID).asText();
        if (tAction != null && tAction.equals(MdcConstant.DELETE)) {
            tenantEocSetInfoDao.deleteIgnoreSetting(tenantId);
            tReturn.put("result", "delete tenantId " + tenantId + " ignore setting success.");
        } else {
            List<String> productIds = CacVerifyServiceUtil.getEnabledModules(tenantId, tToken);
            if (CollectionUtils.isEmpty(productIds)) {
                tReturn.put("remindUnitSetting", false);
            } else {
                // 所有appId都可以检索到值且settingEocFlag设置为1 则无需提醒
                boolean judge = true;
                // 查询product表判定是否需要提醒
                for (String appId : productIds) {
                    log.info("cac获取的产品appId：" + appId);
                    List<Product> productList = productDao.queryByAppId(appId);
                    log.info("cac获取的产品appId对应的产品列表：" + productList);
                    if (CollectionUtils.isEmpty(productList)
                            || (CollectionUtils.isNotEmpty(productList) && productList.stream().anyMatch(x -> x.getSettingEocFlag() == 0))) {
                        judge = false;
                        break;
                    }
                }
                if (judge) {
                    tReturn.put("remindUnitSetting", false);
                } else {
                    // 檢查該租戶是否有設忽略配置， 有配置則remindUnitSetting = false，前端不需要顯示「提醒進行運營配置」畫面
                    Map<String, String> condition = new HashMap<>();
                    condition.put(MdcConstant.TENANT_ID, tenantId);
                    List<TenantEocSetInfo> tenantEocSetInfos = tenantEocSetInfoDao.fetch(condition);
                    if (CollectionUtils.isNotEmpty(tenantEocSetInfos)) {
                        tReturn.put("remindUnitSetting", false);
                    } else {
                        boolean remindUnitSetting = true;

                        //获取当前租户的版本
                        String version = IamVerifyServiceUtil.getCurrentTenantVersion(tenantId);
                        //老版本处理
                        if(MdcConstant.TENANT_VERSION_V1.equalsIgnoreCase(version)){
                            // 檢查該租戶是否有配置任何的運營，層級須為公司(company_id)或工廠(site_id)
                            Map<String, String> eocCondition = new HashMap<>();
                            eocCondition.put(MdcConstant.TENANT_ID, tenantId);
                            List<EocIntgMapping> eocIntgMappings = eocIntgMappingDao.fetch(eocCondition);

                            // 有配置則remindUnitSetting = false，前端不需要顯示「提醒進行運營配置」畫面
                            if (CollectionUtils.isNotEmpty(eocIntgMappings)) {
                                for (EocIntgMapping mapping : eocIntgMappings) {
                                    if (mapping.getEocLevel().equals(MdcConstant.COMPANY_ID) || mapping.getEocLevel().equals(MdcConstant.SITE_ID)) {
                                        remindUnitSetting = false;
                                        break;
                                    }
                                }
                            }
                        }else {
                            // 檢查該租戶是否有配置任何的運營，層級須為公司(company_id)或工廠(site_id) V2租户逻辑查询
                            Map<String, String> eocCondition = new HashMap<>();
                            eocCondition.put(MdcConstant.TENANT_ID, tenantId);
                            eocCondition.put(MdcConstant.EOC_LEVEL, MdcConstant.CLOUD_MAPPING_ID);
                            List<EocIntgMapping> eocIntgMappings = eocIntgMappingDao.fetch(eocCondition);

                            // 有配置則remindUnitSetting = false，前端不需要顯示「提醒進行運營配置」畫面
                            if (CollectionUtils.isNotEmpty(eocIntgMappings)) {
                                for (EocIntgMapping mapping : eocIntgMappings) {
                                    if (mapping.getCloudMappingType().equals(MdcConstant.COMPANY_ID) ||
                                            mapping.getCloudMappingType().equals(MdcConstant.SITE_ID)) {
                                        remindUnitSetting = false;
                                        break;
                                    }
                                }
                            }
                        }
                        // 如果remindUnitSetting仍是true再判斷租戶類型
                        if (remindUnitSetting) {
                            // 根據token查詢租戶資料，如果是個人租戶則remindUnitSetting = false
                            if (IamVerifyServiceUtil.isPersonalTenant(tToken)) {
                                remindUnitSetting = false;
                            }
                        }
                        tReturn.put("remindUnitSetting", remindUnitSetting);
                    }
                }
            }
            // 查询user_id對應的角色是否為superadmin
            tReturn.put("userRole", IamVerifyServiceUtil.isSuperAdmin(userId, tToken) ? "super" : "common");
        }
        return tReturn;
    }
}