package com.digiwin.metadatacache.apiservice;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.digiwin.metadatacache.MdcApplicationParameter;
import com.digiwin.metadatacache.annotate.AppTokenVerify;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.dao.ServiceDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.exception.AddProdNotExistException;
import com.digiwin.metadatacache.exception.AddProdNotFoundException;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.Service;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 雲中台整合設定同步產品清單(增量)
 * Path:[/restful/standard/mdc/ProductInfo/Add]
 */
@org.springframework.stereotype.Service
@AppTokenVerify
public class ProductInfoAddService extends AbstractApiService {

	@Autowired
	private ProductDao productDao;

	@Autowired
	private ServiceDao serviceDao;

	public ProductInfoAddService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.TENANT_PRD_INFO_ADD_SCHEMA;
	}

	/**
	 * 新增产品 信息
	 * @param validatorResult      * @param validatorResult
	 * @return map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		String eaiUId = null;
		String tGatewayId = null;
		if (tRequestJsonNode.has(MdcConstant.EAI_UID)) {
			eaiUId = tRequestJsonNode.get(MdcConstant.EAI_UID).asText();
		}
		if (tRequestJsonNode.get(MdcConstant.GATEWAY_ID) != null && tRequestJsonNode.get(MdcConstant.GATEWAY_ID).asText().length() != 0) {
			tGatewayId = tRequestJsonNode.get(MdcConstant.GATEWAY_ID).asText();
		}
		// 检查需要新增的服务对应的产品是否存在
		ArrayNode tProductAryNode = (ArrayNode) tRequestJsonNode.get(MdcConstant.PRODUCT);
		if (tProductAryNode.size() == 0) {
			throw new AddProdNotExistException(getClass());
		}
		// 保存服务
		addProductAndService(tProductAryNode, eaiUId, tGatewayId);
		// 根据code响应返回
		return returnByCode(validatorResult.getState().getCode());
	}

	/**
	 * 新增服务  -> 新增产品与服务
	 * @param tProductAryNode 节点
	 */
	private void addProductAndService(ArrayNode tProductAryNode, String eaiUId, String tGatewayId) throws BaseException, InterruptedException {
		// 全量更新快取键值
		List<Service> addServiceList = new ArrayList<>();
//		String tKey = CacheMapTypeEnum.tenant_product_list_update_cachemap.getCode() + MdcSymbolConstant.COLON + MdcConstant.ESP;
//		if (StringUtils.isNotBlank(eaiUId)) {
//			tKey = CacheMapTypeEnum.tenant_product_list_update_cachemap.getCode() + MdcSymbolConstant.COLON + eaiUId;
//		}
		// 对需要新增的服务进行添加
		for (JsonNode tProductNode : tProductAryNode) {
			// 产品名称
			String name = tProductNode.get(MdcConstant.NAME).asText();
			// 产品UID
			String uid = tProductNode.get(MdcConstant.UID).asText();
			// 锁键值
			String pKey = name + MdcSymbolConstant.COLON + uid + MdcSymbolConstant.COLON + eaiUId;
			try {
				if (!cacheService.lock(pKey + MdcConstant.CACHE)) {
					return;
				}
				// 获取当前产品下已经存在的服务与新增服务进行比对检查是否已经新增
				Map<String, String> tProductCondition = new HashMap<>();
				tProductCondition.put(MdcConstant.EAI_UID, eaiUId);
				tProductCondition.put(MdcConstant.UID, uid);
				tProductCondition.put(MdcConstant.NAME, name);
				List<Product> productList = productDao.fetch(tProductCondition);
				if (productList.size() > 1) { // 如果不存在或者存在多个则提示异常信息
					throw new AddProdNotFoundException("UID(" + uid + ")" + ",name(" + name + ")", getClass());
				}
				Product product;
				if (CollectionUtils.isEmpty(productList)) {
					product = new Product();
					product.setName(name);
					product.setUid(uid);
					product.setEaiUid(eaiUId);
					product.setVersion(tProductNode.get(MdcConstant.VERSION).asText());
					product.setIp(tProductNode.get(MdcConstant.IP).asText());
					product.setApid(tProductNode.get(MdcConstant.ID).asText());
					product.setGatewayId(tGatewayId);
					if (!StringUtil.isEmptyOrSpace(tProductNode.get(MdcConstant.SETTING_EOC_FLAG))) {
						product.setSettingEocFlag(tProductNode.get(MdcConstant.SETTING_EOC_FLAG).asInt());
					}
					if (!StringUtil.isEmptyOrSpace(tProductNode.get(MdcConstant.APP_ID))) {
						product.setAppId(tProductNode.get(MdcConstant.APP_ID).asText());
					}
					if (tProductNode.get(MdcConstant.URL) != null) {
						product.setUrl(tProductNode.get(MdcConstant.URL).asText());
					}
					product.setBuildTime(Calendar.getInstance());
					product.setLastUpdateTime(Calendar.getInstance());
					if (eaiUId == null) {
						product.setProductType(ApiTypeEnum.agile.toString());
					} else {
						product.setProductType(ApiTypeEnum.standard.toString());
					}
					product = productDao.save(product);
				} else {
					product = productList.get(0);
				}
				// 获取对应产品的服务列表
				List<Service> tCurrentServiceList = serviceDao.getByPrdId(product.getId());
				// 需要新增的服务
				ArrayNode tSrvInfoAryNode = (ArrayNode) tProductNode.get(MdcConstant.SERVICES);
				for (JsonNode tSrvNode : tSrvInfoAryNode) {
					String serviceName = tSrvNode.get(MdcConstant.NAME).asText();
					String version = null;
					if (tSrvNode.has(MdcConstant.VERSION)) {
						version = tSrvNode.get(MdcConstant.VERSION).asText();
					}
					boolean isExist = false;
					for (Service service : tCurrentServiceList) {
						if (serviceName.equals(service.getName())) {
							isExist = true;
							break;
						}
					}
					// 不存在则进行新增
					if (!isExist) {
						// 加入待增加列表
						// 新增service
						Service tService = new Service();
						tService.setBuildTime(Calendar.getInstance());
						tService.setProduct(product);
						tService.setName(serviceName);
						tService.setVersion(version);
						addServiceList.add(tService);
					}
				}
				// 保存
				if (addServiceList.size() > 0) {
					serviceDao.saveServices(addServiceList);
					// 添加缓存 eaiUId为空是敏态 否则是稳态
					String prKey;
					if (StringUtils.isNotBlank(eaiUId)) {
						prKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + product.getTenantId();
						Map<String, List<String>> tProductServiceMapFromCache = (Map<String, List<String>>) cacheService.get(prKey);
						addServiceFromProductInCache(tProductServiceMapFromCache, addServiceList, product, prKey);
					} else {
						for (Service service : addServiceList) {
							prKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON
									+ MdcConstant.AGILE + MdcSymbolConstant.COLON + service.getName();
							Map<String, String> tProductMap = (Map<String, String>) cacheService.get(prKey);
							addProductInCache(tProductMap, product, null == service.getVersion()? "" : service.getVersion() ,prKey);
						}
					}
				}
			}finally {
				cacheService.unLock(pKey + MdcConstant.CACHE);
			}
		}
//		cacheService.remove(tKey);
	}

	/**
	 * 执行结果返回
	 * @param code code
	 * @return map
	 */
	private Map<String, Object> returnByCode(String code){
		StateCode tStateCode = getStateCode(code);
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 添加服务进入缓存
	 * @param pProduct pProduct
	 * @param services services
	 * @param tProductServiceMapFromCache tProductServiceMapFromCache
	 * @param tKey tKey
	 */
	private void addServiceFromProductInCache(Map<String, List<String>> tProductServiceMapFromCache,
											  List<Service> services, Product pProduct, String tKey) {
		String tProductKey = pProduct.getName() + MdcSymbolConstant.COLON + pProduct.getUid() + MdcSymbolConstant.COLON + pProduct.getEaiUid();
		if (tProductServiceMapFromCache != null && tProductServiceMapFromCache.get(tProductKey) != null) {
			log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "更新快取(產品中的服務)：" + "tenant = "
					+ pProduct.getTenantId() + ", product = " + tProductKey + ", service = " + services);
			tProductServiceMapFromCache.get(tProductKey).addAll(services.stream()
					.map(x->x.getName() + MdcSymbolConstant.LEFT_DASH + (null == x.getVersion()? "" : x.getVersion())).collect(Collectors.toList()));
			cacheService.setSync(tKey, tProductServiceMapFromCache);
		}
	}

	/**
	 * 添加产品进入缓存
	 * @param pProduct pProduct
	 * @param tProductMap tProductMap
	 * @param tKey tKey
	 */
	private void addProductInCache(Map<String, String> tProductMap, Product pProduct, String serviceVersion, String tKey) {
		String tProductKey = pProduct.getName() + MdcSymbolConstant.COLON + pProduct.getUid();
		if (null != tProductMap) {
			log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "更新快取(產品中的服務)：" + "pProduct = "
					+ pProduct.getTenantId() + ", product = " + tProductKey + ", service = "+ tKey);
			String tProductFullInfo = pProduct.getName()
					+ MdcSymbolConstant.LEFT_DASH + pProduct.getUid()
					+ MdcSymbolConstant.LEFT_DASH + pProduct.getEaiUid()
					+ MdcSymbolConstant.LEFT_DASH + pProduct.getVersion()
					+ MdcSymbolConstant.LEFT_DASH + pProduct.getIp()
					+ MdcSymbolConstant.LEFT_DASH + pProduct.getApid()
					+ MdcSymbolConstant.LEFT_DASH + pProduct.getGatewayId()
					+ ":::" + serviceVersion;
			tProductMap.put(tProductKey,tProductFullInfo);
			cacheService.setSyncTimeOut(tKey, tProductMap, MdcApplicationParameter._TIME_OUT + (int) (Math.random() * MdcApplicationParameter._TIME_RANDOM));
		}
	}
}
