package com.digiwin.metadatacache.apiservice;

import com.digiwin.app.service.DWServiceContext;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.dao.ApiDao;
import com.digiwin.metadatacache.dao.ApiMetadataDao;
import com.digiwin.metadatacache.dao.ApiVersionDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.Api;
import com.digiwin.metadatacache.model.ApiMetadata;
import com.digiwin.metadatacache.model.ApiVersion;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.services.LoadCacheService;
import com.digiwin.metadatacache.services.LocalCacheService;
import com.digiwin.metadatacache.util.DateUtil;
import com.digiwin.metadatacache.util.MQUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 更新API元數據
 * Path:[/restful/standard/mdc/StdApiMetadata/Update]
 */
@Service
public class StdApiMetadataUpdateService extends AbstractApiService {

	@Autowired
	private ApiDao apiDao;

	@Autowired
	private ApiVersionDao apiVersionDao;

	@Autowired
	private ApiMetadataDao apiMetadataDao;

	@Autowired
	private ApiRelationSaveService apiRelationSaveService;

	@Autowired
	private LoadCacheService loadCacheService;

	@Autowired
	private LocalCacheService localCacheService;

	@Autowired
	private AssetRegisterService assetRegisterService;

	public StdApiMetadataUpdateService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.STANDARD_API_UPDATE_SCHEMA;
	}

	/**
	 * 更新API元数据
	 * @param validatorResult      * @param validatorResult
	 * @return Map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		String tUserId = (String) DWServiceContext.getContext().getRequestHeader().get(MdcConstant.USERID);
		ArrayNode tApiArrayNode = (ArrayNode) tRequestJsonNode.get(MdcConstant.API_DATA);
		if (tApiArrayNode.isArray()) {
			for (JsonNode tApi : tApiArrayNode) {
				update(tApi, tUserId);
			}
		}
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 新增Api
	 * @param tApiData tApiData
	 */
	private void update(JsonNode tApiData, String tUserId) throws Exception {
		Map<String, Object> tDataMap = new HashMap<>();
		List<String> versionList = new ArrayList<>();
		for (String tApiInfoKey : MdcConstant.API_INFO_SPEC_KEY) {
			ArrayNode tArrayNode;
			JsonNode tData = tApiData.get(tApiInfoKey);
			// 取得字段內容放入tDataMap
			switch (tApiInfoKey) {
				case MdcConstant.API_NAME:
				case MdcConstant.INVOKED_TYPE:
				case MdcConstant.CATEGORY:
				case MdcConstant.PROVIDER:
					tDataMap.put(tApiInfoKey, tData.asText());
					break;
				case MdcConstant.APP_NAME:
				case MdcConstant.TENANT_ID:
				case MdcConstant.PACKAGE_NAME:
				case MdcConstant.PARENT_API_NAME:
				case MdcConstant.API_EXTEND:
					if (tData != null) {
						tDataMap.put(tApiInfoKey, tData.asText());
					}
					break;
				case MdcConstant.API_VERSION_INFO:
					ObjectNode tVersionDataNode = mapper.createObjectNode();
					tArrayNode = (ArrayNode) tData;
					if (tArrayNode.isArray()) {
						for (JsonNode tArrayElement : tArrayNode) {
							tDataMap.put(MdcConstant.APPROVED_STATUS, tArrayElement.get(MdcConstant.APPROVED_STATUS).asText());
							if (tArrayElement.get(MdcConstant.IDEMPOTENCY) != null) {
								tDataMap.put(MdcConstant.IDEMPOTENCY, tArrayElement.get(MdcConstant.IDEMPOTENCY).asText());
							}
							tVersionDataNode.set(tArrayElement.get(MdcConstant.API_VERSION).asText(),
									tArrayElement.get(MdcConstant.DATA_METADATA));
							versionList.add(tArrayElement.get(MdcConstant.API_VERSION).asText());
						}
					}
					tDataMap.put(MdcConstant.API_VERSION_INFO, tVersionDataNode);
					break;
				case MdcConstant.DESCRIPTION:
					if (tData.isObject()) {
						tDataMap.put(MdcConstant.DESCRIPTION_ZH_TW, tData.get(MdcConstant.ZH_TW).asText());
						tDataMap.put(MdcConstant.DESCRIPTION_ZH_CN, tData.get(MdcConstant.ZH_CN).asText());
						tDataMap.put(MdcConstant.DESCRIPTION_EN, tData.get(MdcConstant.EN).asText());
					}
					break;
				case MdcConstant.REMARK:
					if (tData != null && tData.isObject()) {
						tDataMap.put(MdcConstant.REMARK_ZH_TW, tData.get(MdcConstant.ZH_TW).asText());
						tDataMap.put(MdcConstant.REMARK_ZH_CN, tData.get(MdcConstant.ZH_CN).asText());
						tDataMap.put(MdcConstant.REMARK_EN, tData.get(MdcConstant.EN).asText());
					}
					break;
				default:
					break;
			}
		}
		if (CollectionUtils.isNotEmpty(versionList)) {
			String tTenantId = null;
			if (null != tDataMap.get(MdcConstant.TENANT_ID)) {
				tTenantId = String.valueOf(tDataMap.get(MdcConstant.TENANT_ID));
			}
			for (String version : versionList) {
				deleteApi(String.valueOf(tDataMap.get(MdcConstant.API_NAME)), version, tTenantId);
			}
		}
		if (tDataMap.get(MdcConstant.APP_NAME) != null && tDataMap.get(MdcConstant.APP_NAME).toString().length() != 0) {
			addApi(tDataMap, ApiTypeEnum.agile.toString(), tUserId);
		} else {
			// 儲存API資訊
			addApi(tDataMap, ApiTypeEnum.standard.toString(), tUserId);
		}
		// 組成發送至MQ訊息
		log.info("Append Send MQ: apiName=" + tDataMap.get(MdcConstant.API_NAME) + ",action=add");
		String tApiMetadataStr = sendMqMessage(tDataMap);
		// 發送訊息
		MQUtil.sendApiMetadata(tApiMetadataStr);
		log.info("Send AddApi MQ Success.");
		//保存API关联关系
		log.info("创建API关联关系开始");
		if (null != tDataMap.get(MdcConstant.PARENT_API_NAME)) {
			apiRelationSaveService.saveApiRelation(
					tDataMap.get(MdcConstant.PARENT_API_NAME).toString(), tDataMap.get(MdcConstant.API_NAME).toString(),
					null != tDataMap.get(MdcConstant.TENANT_ID)? tDataMap.get(MdcConstant.TENANT_ID).toString() : null,
					tUserId);
		}
		log.info("创建API关联关系结束");
	}

	/**
	 * 删除
	 * @param apiName api名称
	 * @param apiVersion api版本
	 * @param tenantId 租户Id
	 */
	private void deleteApi(String apiName,String apiVersion,String tenantId) throws Exception {
		apiDao.removeByNameAndVersionTenantId(apiName, apiVersion, tenantId);
		cacheService.remove(CacheMapTypeEnum.api_version_list.getCode() + MdcSymbolConstant.COLON + apiName);
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "清除快取(清除服務版本號)：" + "key = "
				+ CacheMapTypeEnum.api_version_list.getCode() + MdcSymbolConstant.COLON + apiName);
        loadCacheService.removeData(CacheMapTypeEnum.api_base_info_cachemap.getCode() + MdcSymbolConstant.COLON + apiName);
		localCacheService.remove(CacheMapTypeEnum.api_relation_idempotency_cachemap, apiName);
        log.info("删除本地缓存: " + CacheMapTypeEnum.api_base_info_cachemap.getCode() + MdcSymbolConstant.COLON + apiName);
		// 建立data節點
		ObjectNode tApiMetadataNode = mapper.createObjectNode();
		tApiMetadataNode.put(MdcConstant.API_NAME, apiName);
		tApiMetadataNode.put(MdcConstant.API_VERSION, apiVersion);
		if (tenantId != null) {
			tApiMetadataNode.put(MdcConstant.TENANT_ID, tenantId);
		}
		tApiMetadataNode.put(MdcConstant.TIMESTAMP, DateUtil.formatToDate(new Date()));
		tApiMetadataNode.put(MdcConstant.ACTION, "delete");
		// 發送訊息
        log.info("同步删除信息至MQ开始_API名称:"+ apiName + ",API版本:" + apiVersion + ",API租户:" + tenantId);
		MQUtil.sendApiMetadata(tApiMetadataNode.toString());
		log.info("Send DeleteApi MQ Success.");
//		log.info("删除资产表API信息开始");
//		assetRegisterService.deletePropertyAndES(apiName, PropertyTypeEnum.API.toString(), tenantId);
//		log.info("删除资产表API信息结束");
	}

	/**
	 * 存贮API咨询(表：api, api_version, api_metadata)
	 * @param tDataMap tDataMap
	 * @param pApiType pApiType
	 * @throws IOException IOException
	 */
	private void addApi(Map<String, Object> tDataMap, String pApiType, String userName) throws Exception {
		Api tApi = new Api();
		List<ApiVersion> apiVersionList = new ArrayList<>();
		ApiVersion tApiVersion = new ApiVersion();
		ApiMetadata tApiMetadata = new ApiMetadata();
		String tVersionMapString = tDataMap.get(MdcConstant.API_VERSION_INFO).toString();
		// convert JSON string to Map
		Map<String, Object> tVersionMap = mapper.readValue(tVersionMapString, Map.class);
		
		for (String tVersion : tVersionMap.keySet()) {
			tApiVersion.setVersion(tVersion);
			tApiVersion.setApprovedStatus(Long.valueOf(tDataMap.get(MdcConstant.APPROVED_STATUS).toString()));
			if (tDataMap.get(MdcConstant.IDEMPOTENCY) != null) {
				tApiVersion.setIdempotency(Boolean.parseBoolean(tDataMap.get(MdcConstant.IDEMPOTENCY).toString()));
			}
			tApiVersion.setBuildTime(Calendar.getInstance());
			tApiVersion.setApi(tApi);
			tApi.addApiVersion(tApiVersion);
			
			tApiMetadata.setBuildTime(Calendar.getInstance());
			tApiMetadata.setMetadata(mapper.writeValueAsString(tVersionMap.get(tVersion)));
			tApiVersion.setApiMetadata(tApiMetadata);
			
			apiVersionList.add(tApiVersion);
			tApi.setApiVersions(apiVersionList);
			
			tApi.setName(tDataMap.get(MdcConstant.API_NAME).toString().trim());
			if (null != tDataMap.get(MdcConstant.API_EXTEND)) {
				tApi.setApiExtend(tDataMap.get(MdcConstant.API_EXTEND).toString());
			}
			if (tDataMap.get(MdcConstant.INVOKED_TYPE) == null) {
				tApi.setInvokedType(MdcConstant.SYNC);
			} else {
				tApi.setInvokedType(tDataMap.get(MdcConstant.INVOKED_TYPE).toString().trim());
			}
			tApi.setProvider(tDataMap.get(MdcConstant.PROVIDER).toString().trim());
			tApi.setDescriptionZhTw(tDataMap.get(MdcConstant.DESCRIPTION_ZH_TW).toString().trim());
			tApi.setDescriptionZhCn(tDataMap.get(MdcConstant.DESCRIPTION_ZH_CN).toString().trim());
			tApi.setDescriptionEn(tDataMap.get(MdcConstant.DESCRIPTION_EN).toString().trim());
			if (tDataMap.containsKey(MdcConstant.REMARK_ZH_TW)) {
				tApi.setRemarkZhTw(tDataMap.get(MdcConstant.REMARK_ZH_TW).toString().trim());
			}
			if (tDataMap.containsKey(MdcConstant.REMARK_ZH_CN)) {
				tApi.setRemarkZhCn(tDataMap.get(MdcConstant.REMARK_ZH_CN).toString().trim());
			}
			if (tDataMap.containsKey(MdcConstant.REMARK_EN)) {
				tApi.setRemarkEn(tDataMap.get(MdcConstant.REMARK_EN).toString().trim());
			}
			tApi.setCategory(tDataMap.get(MdcConstant.CATEGORY).toString().trim());
			tApi.setApi_type(pApiType);
			
			String tTenantId = null;
			if (tDataMap.get(MdcConstant.TENANT_ID) != null && tDataMap.get(MdcConstant.TENANT_ID).toString().length() != 0) {
				tTenantId = tDataMap.get(MdcConstant.TENANT_ID).toString().trim();
				tApi.setTenantId(tTenantId);
			}
			if (tDataMap.get(MdcConstant.PACKAGE_NAME) != null && tDataMap.get(MdcConstant.PACKAGE_NAME).toString().length() != 0) {
				tApi.setPackageName(tDataMap.get(MdcConstant.PACKAGE_NAME).toString().trim());
			}
			tApi.setBuildTime(Calendar.getInstance());
			apiDao.save(tApi);
//			ApiVersion tSavedApiVersion = apiVersionDao.getByApiNameTenantVer(tApi.getName(), tTenantId, tApiVersion.getVersion());
			cacheService.remove(CacheMapTypeEnum.api_version_list.getCode() + MdcSymbolConstant.COLON + tApi.getName());
			log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "清除快取(清除服務版本號)：" + "key = "
					+ CacheMapTypeEnum.api_version_list.getCode() + MdcSymbolConstant.COLON + tApi.getName());
            loadCacheService.removeData(CacheMapTypeEnum.api_base_info_cachemap.getCode() + MdcSymbolConstant.COLON + tApi.getName());
			localCacheService.remove(CacheMapTypeEnum.api_relation_idempotency_cachemap, tApi.getName());
            log.info("删除本地缓存: " + CacheMapTypeEnum.api_base_info_cachemap.getCode() + MdcSymbolConstant.COLON + tApi.getName());
//			log.info("更新资产表开始");
//			List<ApiMetadata> apiMetadataList = apiMetadataDao.getByApiVersionId(tSavedApiVersion.getId());
//			if (CollectionUtils.isNotEmpty(apiMetadataList)) {
//				assetRegisterService.saveAssetProperty(tApi.getName(), tApi.getDescriptionZhCn(), tApi.getProvider(),
//						apiMetadataList.get(0), tApi.getTenantId(), userName);
//			}
//			log.info("更新资产表结束");
		}
	}
}
