package com.digiwin.metadatacache.apiservice;

import com.digiwin.metadatacache.apiservice.request.TenantApiProductListGetRequest;
import com.digiwin.metadatacache.apiservice.response.TenantApiProductListGetResponse;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.dao.ApiDao;
import com.digiwin.metadatacache.dao.ApiVersionDao;
import com.digiwin.metadatacache.dao.EaiTenantMappingDao;
import com.digiwin.metadatacache.dao.EocIntgMappingDao;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.ApiRelation;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.services.MdcProductService;
import com.digiwin.metadatacache.util.JsonUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 取得租户产品清单
 * Path:[/restful/standard/mdc/TenantApiProduct/batch/get]
 */
@org.springframework.stereotype.Service
public class TenantApiProductListGetService extends AbstractApiService {

    @Autowired
    private ApiDao apiDao;

    @Autowired
    private ApiVersionDao apiVersionDao;

    @Autowired
    private EocIntgMappingDao eocIntgMappingDao;

    @Autowired
    private MdcProductService productService;

    @Autowired
    private EaiTenantMappingDao eaiTenantMappingDao;

    public TenantApiProductListGetService() {
        super();
        jsonSchemaFileName = JsonSchemaFileConstant.TENANT_API_PRD_LIST_GET_SCHEMA;
    }

    /**
     * 取得租户产品清单逻辑处理
     *
     * @param validatorResult validatorResult
     * @return Map
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
                + ", 原始訊息: " + tRequestJsonNode);
        TenantApiProductListGetRequest request = JsonUtil.toJavaObj(tRequestJsonNode, TenantApiProductListGetRequest.class);
        String tenantId = request.getTenantId();
        List<String> apiNames = request.getApiNames();
        List<ApiRelation> apiRelations = apiRelationDao.getRelationBySub(apiNames, tenantId);
        Map<String, String> apiRelationMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(apiRelations)) {
            apiRelations.forEach(one -> {
                if (StringUtils.isEmpty(one.getSubTenantId())) {
                    apiRelationMap.putIfAbsent(one.getSubApiName(), one.getParentApiName());
                } else {
                    apiRelationMap.put(one.getSubApiName(), one.getParentApiName());
                }
            });
        }
        //缓存获取
        Map<String, List<Product>> productMap = getProductFromCache(tenantId, apiNames, apiRelationMap);
        List<String> notFoundProdApis = apiNames.stream().filter(api -> !productMap.containsKey(api)).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(notFoundProdApis)) {
            Map<String, List<Product>> dbProductMap = productService.getProductFromDb(tenantId, notFoundProdApis, apiRelationMap);
            if (MapUtils.isNotEmpty(dbProductMap)) {
                productMap.putAll(dbProductMap);
            }
        }
        Map<String, List<TenantApiProductListGetResponse.ProductVO>> productVOMap = productMap.entrySet().stream()
                .collect(Collectors.toMap(entry -> entry.getKey(), entry -> covertToProductVO(entry.getValue())));
        TenantApiProductListGetResponse response = new TenantApiProductListGetResponse(tenantId, productVOMap);
        String tDescription = getDescriptionByLocale(SUCCESS_STATE, locale);
        return createResponseMap(SUCCESS_STATE.getCode(), tDescription, response.toMap());
    }

    private Map<String, List<Product>> getProductFromCache(String tenantId, List<String> apiNames, Map<String, String> apiRelationMap) {
        String tStandardKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + tenantId;
        Map<String, List<String>> standardCache = (Map<String, List<String>>) cacheService.get(tStandardKey);
        Map<String, List<Product>> productMap = new HashMap<>();
        apiNames.forEach(api -> {
            String realApi = apiRelationMap.containsKey(api) ? apiRelationMap.get(api) : api;
            List<Product> products = productService.getStandardProductCache(tenantId, standardCache, realApi);
            if (CollectionUtils.isEmpty(products)) {
                products = productService.getAgileProductCache(tenantId, realApi);
                if (CollectionUtils.isNotEmpty(products)) {
                    productMap.put(realApi, products);
                }
            } else {
                log.info("命中快取: " + tStandardKey);
                productMap.put(realApi, products);
            }
        });
        return productMap;
    }

    private static List<TenantApiProductListGetResponse.ProductVO> covertToProductVO(List<Product> products) {
        if (null == products) {
            return null;
        }
        return products.stream().map(product -> {
            TenantApiProductListGetResponse.ProductVO vo = new TenantApiProductListGetResponse.ProductVO();
            vo.setVersion(product.getVersion());
            vo.setUid(product.getUid());
            vo.setId(product.getApid());
            vo.setIp(product.getIp());
            vo.setName(product.getName());
            vo.setGatewayId(product.getGatewayId());
            return vo;
        }).collect(Collectors.toList());
    }
}
