package com.digiwin.metadatacache.apiservice;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.MdcApplicationParameter;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.digiwin.metadatacache.dao.EaiTenantMappingDao;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.enums.ProdTypeEnum;
import com.digiwin.metadatacache.enums.ValidateStateEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.exception.ProdTypeNotFoundException;
import com.digiwin.metadatacache.model.EaiTenantMapping;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.Service;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 取得租戶的ERP產品主機資訊<br>
 * Path:[/restful/standard/mdc/TenantProductType/Get]
 */
@org.springframework.stereotype.Service
public class TenantProductTypeGetService extends AbstractApiService {

	@Autowired
	private ProductDao productDao;

	@Autowired
	private EaiTenantMappingDao eaiTenantMappingDao;

	public TenantProductTypeGetService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.TENANT_PRD_TYPE_GET_SCHEMA;
	}

	/**
	 * 取得租戶的ERP產品主機資訊
	 * @param validatorResult      * @param validatorResult
	 * @return Map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		// 取出入參tenant, prod_type
		String tTenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
		String tProdType = tRequestJsonNode.get(MdcConstant.PROD_TYPE).asText();
		return processTypeProductGet(tTenantId, tProdType);
	}

	/**
	 * 根據租戶ID、產品類型取得符合條件的產品主機清單 <br>
	 * 回傳清單=[該租戶符合產品類型的產品主機清單]
	 * 
	 * @param pTenantId 租户Id
	 * @param pProdType 主机类型  ERP
	 * @return Map
	 */
	private Map<String, Object> processTypeProductGet(String pTenantId, String pProdType) throws Exception {
		// 組成回傳訊息必要節點
		ObjectNode tDataNode = mapper.createObjectNode();
		tDataNode.put(MdcConstant.TENANT_ID, pTenantId);
		ArrayNode tGatewayArrayNode = mapper.createArrayNode();
		tDataNode.set(MdcConstant.GATEWAY, tGatewayArrayNode);
		// 檢查輸入的產品類型是否存在
		boolean exist = false;
		for (ProdTypeEnum prodTypeEnum : ProdTypeEnum.values()) {
			if (pProdType.equals(prodTypeEnum.toString())) {
				exist = true;
			}
		}
		if (!exist) {
			throw new ProdTypeNotFoundException(getClass());
		}
		// 先取得該租戶所屬地端中台UID
		Map<String, String> tEaiTenantMappingCondition = new HashMap<>();
		tEaiTenantMappingCondition.put(MdcConstant.TENANT, pTenantId);
		List<EaiTenantMapping> tEaiTenantMappings = eaiTenantMappingDao.fetch(tEaiTenantMappingCondition);
		// 有對應中台UID
		if (!tEaiTenantMappings.isEmpty()) {
			// 取得該中台UID上的所有產品(tProductList)
			Map<String, String> pProductCondition = new HashMap<>();
			pProductCondition.put(MdcConstant.EAI_UID, tEaiTenantMappings.get(0).getEaUid());
			List<Product> tEaiAllProductList = productDao.fetch(pProductCondition);
			// 將產品主機依照產品名稱分群放入各List
			Map<String, List<Product>> tProductMapGroupByProdName = new HashMap<>();
			for (Product tProduct : tEaiAllProductList) {
				List<Product> tProducts = tProductMapGroupByProdName.get(tProduct.getName());
				if (CollectionUtils.isEmpty(tProducts)) {
					tProducts = new ArrayList<>();
					tProductMapGroupByProdName.put(tProduct.getName(), tProducts);
				}
				tProducts.add(tProduct);
			}
			List<Product> tReturnProductList = new ArrayList<>();
			// 根據各產品名稱找出有對應到租戶ID的產品主機
			for (String tProductType : tProductMapGroupByProdName.keySet()) {
				for (Product tProduct : tProductMapGroupByProdName.get(tProductType)) {
					if (tProduct.getTenantId() != null && tProduct.getTenantId().equals(pTenantId)) {
						// 如果輸入產品類型是ERP，則找出ERP類產品主機
						if (pProdType.equals(ProdTypeEnum.ERP.toString())) {
							if (MdcApplicationParameter._ERP_LIST.contains(tProduct.getName())) {
								tReturnProductList.add(tProduct);
							}
						}
					}
				}
			}
			if (!tReturnProductList.isEmpty()) {
				// 將產品清單轉成分群Map
				Map<String, Map<String, Map<String, List<Product>>>> tProductsMap
						= transformListToGroupMap(tReturnProductList, pTenantId);
				// Gateway
				for (String tGatewayId : tProductsMap.get(pTenantId).keySet()) {
					ObjectNode tGatewayItemNode = mapper.createObjectNode();
					tGatewayItemNode.put(MdcConstant.GATEWAY_ID, tGatewayId);
					tGatewayArrayNode.add(tGatewayItemNode);
					ArrayNode tProductArrayNode = mapper.createArrayNode();
					// Product Name
					for (String tProductName : tProductsMap.get(pTenantId).get(tGatewayId).keySet()) {
						// Product
						for (Product tProduct : tProductsMap.get(pTenantId).get(tGatewayId).get(tProductName)) {
							ObjectNode tProductItemNode = mapper.createObjectNode();
							tProductItemNode.put(MdcConstant.NAME, tProduct.getName());
							tProductItemNode.put(MdcConstant.VERSION, tProduct.getVersion());
							tProductItemNode.put(MdcConstant.IP, tProduct.getIp());
							tProductItemNode.put(MdcConstant.APID, tProduct.getApid());
							tProductItemNode.put(MdcConstant.UID, tProduct.getUid());
							if (MdcApplicationParameter._ERP_LIST.contains(tProduct.getName())) {
								tProductItemNode.put(MdcConstant.TYPE, MdcConstant.ERP);
							}
							tProductArrayNode.add(tProductItemNode);
						}
					}
					tGatewayItemNode.set(MdcConstant.PRODUCT, sortProductArray(tProductArrayNode));
				}
			}
		}
		StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
		String tDescription =getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 將List轉成Map，並依照租戶ID、GatewayID及產品名稱分群 <br>
	 * 第一層key: 租戶ID <br>
	 * 第二層key: GatewayID <br>
	 * 第三層key: 產品名稱
	 * 
	 * @param pList pList
	 * @param pTenant pTenant
	 * @return map
	 */
	private <T> Map<String, Map<String, Map<String, List<T>>>> transformListToGroupMap(List<T> pList, String pTenant) {
		Map<String, Map<String, Map<String, List<T>>>> tTenantMap = new HashMap<>();
		for (T tObject : pList) {
			Product tProduct = null;
			if (tObject instanceof Product) {
				tProduct = ((Product) tObject);
			} else if (tObject instanceof Service) {
				tProduct = ((Service) tObject).getProduct();
			}
			if (tProduct != null) {
				// 第一層key: 租戶ID
				Map<String, Map<String, List<T>>> tGatewayMap = tTenantMap.get(pTenant);
				if (tGatewayMap == null) {
					// 建立一個此租戶的gateway Map
					tGatewayMap = new HashMap<>();
					tTenantMap.put(pTenant, tGatewayMap);
				}
				// 第二層key: GatewayID
				Map<String, List<T>> tProductNameMap = tGatewayMap.get(tProduct.getGatewayId());
				if (tProductNameMap == null) {
					// 建立一個此gateway的產品名稱 Map
					tProductNameMap = new HashMap<>();
					tGatewayMap.put(tProduct.getGatewayId(), tProductNameMap);
				}
				// 第三層key: 產品名稱
				List<T> tServiceList = tProductNameMap.get(tProduct.getName());
				if (tServiceList == null) {
					// 建立產品名稱的產品主機list
					tServiceList = new ArrayList<T>();
					tProductNameMap.put(tProduct.getName(), tServiceList);
				}
				tServiceList.add(tObject);
			}
		}
		return tTenantMap;
	}

	public ArrayNode sortProductArray(ArrayNode pArrayNode) {
		// 先將產品分成ERP ERPII
		List<JsonNode> erpList = new ArrayList<>();
		List<JsonNode> erpIIList = new ArrayList<>();
		for (JsonNode node : pArrayNode) {
			if (node.get(MdcConstant.TYPE).asText().equals(MdcConstant.ERP)) {
				erpList.add(node);
			} else {
				erpIIList.add(node);
			}
		}
		Collections.sort(erpList, new Comparator() {
			@Override
			public int compare(Object o1, Object o2) {
				String name1;
				String name2;
				name1 = ((ObjectNode) o1).get(MdcConstant.NAME).asText();
				name2 = ((ObjectNode) o2).get(MdcConstant.NAME).asText();
				return name1.compareTo(name2);
			}
		});
		Collections.sort(erpIIList, new Comparator() {
			@Override
			public int compare(Object o1, Object o2) {
				String name1;
				String name2;
				name1 = ((ObjectNode) o1).get(MdcConstant.NAME).asText();
				name2 = ((ObjectNode) o2).get(MdcConstant.NAME).asText();
				return name1.compareTo(name2);
			}
		});
		// 排序後的json array
		ArrayNode sortedJsonArray = mapper.createArrayNode();
		for (JsonNode jsonNode : erpList) {
			sortedJsonArray.add(jsonNode);
		}
		for (JsonNode jsonNode : erpIIList) {
			sortedJsonArray.add(jsonNode);
		}
		return sortedJsonArray;
	}
}
