package com.digiwin.metadatacache.dao.impl;

import java.util.Calendar;
import java.util.List;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;
import org.hibernate.query.Query;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.digiwin.metadatacache.dao.ApiVersionDao;
import com.digiwin.metadatacache.model.ApiVersion;

@Service
@Transactional("mdcTransactionManager")
public class ApiVersionDaoImpl extends GenericDaoImpl<ApiVersion, Long> implements ApiVersionDao {

	public ApiVersionDaoImpl() {
		super(ApiVersion.class);
	}

	@Override
	public List<ApiVersion> getByApiId(final Long pId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ApiVersion>>() {
			@SuppressWarnings("unchecked")
			@Override
			public List<ApiVersion> doInHibernate(Session session) throws HibernateException {
				Query<ApiVersion> tQuery = session.createQuery("from ApiVersion where api_id = :param ");
				tQuery.setParameter("param", pId);
				List<?> list = tQuery.list();
				List<ApiVersion> tApiVersions = (List<ApiVersion>) list;
				if (list.size() == 0) {
					return null;
				} else {
					return tApiVersions;
				}
			}
		});
	}

	@Override
	public ApiVersion getByApiIdAndVer(Long pId, String pVer) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(ApiVersion.class);
		tDetachedCriteria.add(Restrictions.eq("Api.id", pId));
		tDetachedCriteria.add(Restrictions.eq("version", pVer));
		List<?> tApiVersions = getHibernateTemplate().findByCriteria(tDetachedCriteria);
		if (tApiVersions.size() == 0) {
			return null;
		} else {
			return (ApiVersion) tApiVersions.get(0);
		}
	}

	@Override
	public ApiVersion getByApiNameAndVer(String pApiName, String pVer) {
		String hql = "SELECT av FROM ApiVersion av JOIN av.Api a WHERE a.name = :pApiName AND av.version = :pVer";
		List<?> results = getHibernateTemplate().execute(session -> {
			Query query = session.createQuery(hql);
			query.setParameter("pApiName", pApiName);
			query.setParameter("pVer", pVer);
			return query.list();
		});
		if (results.isEmpty()) {
			return null;
		} else {
			return (ApiVersion) results.get(0);
		}
	}

	@Override
	public ApiVersion getByApiNameTenantVer(String pApiName, String pTenant, String pVer) {
		if (pTenant != null) {
			String hql = "SELECT av FROM ApiVersion av JOIN av.Api a WHERE a.name = :pApiName AND a.tenantId = :pTenantId AND av.version = :pVer";
			List<?> results = getHibernateTemplate().execute(session -> {
				Query query = session.createQuery(hql);
				query.setParameter("pApiName", pApiName);
				query.setParameter("pTenantId", pTenant);
				query.setParameter("pVer", pVer);
				return query.list();
			});
			if (results.isEmpty()) {
				return null;
			} else {
				return (ApiVersion) results.get(0);
			}
		} else {
			String hql = "SELECT av FROM ApiVersion av JOIN av.Api a WHERE a.name = :pApiName AND a.tenantId is null AND av.version = :pVer";
			List<?> results = getHibernateTemplate().execute(session -> {
				Query query = session.createQuery(hql);
				query.setParameter("pApiName", pApiName);
				query.setParameter("pVer", pVer);
				return query.list();
			});
			if (results.isEmpty()) {
				return null;
			} else {
				return (ApiVersion) results.get(0);
			}
		}
	}

	@Override
	public List<ApiVersion> getByApiNameAndTenantId(String apiName, String tenantId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ApiVersion>>() {
			@SuppressWarnings("unchecked")
			@Override
			public List<ApiVersion> doInHibernate(Session session) throws HibernateException {
				StringBuilder tStrBuild = new StringBuilder();
				tStrBuild.append("SELECT ver FROM ApiVersion ver JOIN ver.Api a ");
				tStrBuild.append("WHERE a.name = :name ");
				if (StringUtils.isNotBlank(tenantId)) {
					tStrBuild.append("AND a.tenantId = :tenantId");
				} else {
					tStrBuild.append("AND a.tenantId is null");
				}
				Query<Object> tQuery = session.createQuery(tStrBuild.toString());
				tQuery.setParameter("name", apiName);
				if (StringUtils.isNotBlank(tenantId)) {
					tQuery.setParameter("tenantId", tenantId);
				}
				List<?> list = tQuery.list();
				List<ApiVersion> tApiVersions = (List<ApiVersion>) list;
				if (tApiVersions.size() == 0) {
					return null;
				} else {
					return tApiVersions;
				}
			}
		});
	}

	@Override
	public List<ApiVersion> getBaseInfoByApiNameAndTenantId(String apiName, String tenantId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ApiVersion>>() {
			@SuppressWarnings("unchecked")
			@Override
			public List<ApiVersion> doInHibernate(Session session) throws HibernateException {
				StringBuilder tStrBuild = new StringBuilder();
				tStrBuild.append("SELECT ver FROM ApiVersion ver JOIN ver.Api a ");
				tStrBuild.append("WHERE a.name = :name ");
				if (StringUtils.isNotBlank(tenantId)) {
					tStrBuild.append("AND (a.tenantId = :tenantId OR a.tenantId is null)");
				} else {
					tStrBuild.append("AND a.tenantId is null");
				}
				Query<Object> tQuery = session.createQuery(tStrBuild.toString());
				tQuery.setParameter("name", apiName);
				if (StringUtils.isNotBlank(tenantId)) {
					tQuery.setParameter("tenantId", tenantId);
				}
				List<?> list = tQuery.list();
				return (List<ApiVersion>) list;
			}
		});
	}

	// 找出API最新版本
	@Override
	public ApiVersion getLatestVerByApiId(long pId) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(ApiVersion.class);
		tDetachedCriteria.add(Restrictions.eq("Api.id", pId));
		List<ApiVersion> tApiVersions = (List<ApiVersion>) getHibernateTemplate().findByCriteria(tDetachedCriteria);
		// 取得版本清單
		if (tApiVersions.size() == 0) {
			return null;
		} else {
			String[] versionSplit;
			String[] maxVersionSplit;
			String versionIntegar;
			String versionDecimal;
			String maxVersionIntegar;
			String maxVersionDecimal;
			String tMaxVer = "0.0";
			ApiVersion tMaxApiVer = null;
			for (ApiVersion tApiVersion : tApiVersions) {
				versionSplit = tApiVersion.getVersion().split("\\.");
				maxVersionSplit = tMaxVer.split("\\.");
				versionIntegar = versionSplit[0];
				versionDecimal = versionSplit[1];
				maxVersionIntegar = maxVersionSplit[0];
				maxVersionDecimal = maxVersionSplit[1];
				if (Integer.parseInt(versionIntegar) > Integer.parseInt(maxVersionIntegar)) {
					tMaxVer = tApiVersion.getVersion();
					tMaxApiVer = tApiVersion;
				} else if (Integer.valueOf(versionIntegar).equals(Integer.valueOf(maxVersionIntegar))) {
					if (Integer.parseInt(versionDecimal) > Integer.parseInt(maxVersionDecimal)) {
						tMaxVer = tApiVersion.getVersion();
						tMaxApiVer = tApiVersion;
					}
				}
			}
			return tMaxApiVer;

		}
	}

	@Override
	public List<ApiVersion> getApiInfo(Calendar pBuildTime, String pProvider) {
		return getHibernateTemplate().execute(new HibernateCallback<List<ApiVersion>>() {
			@SuppressWarnings("unchecked")
			@Override
			public List<ApiVersion> doInHibernate(Session session) throws HibernateException {
				StringBuilder tStrBuild = new StringBuilder();
				tStrBuild.append("SELECT ver FROM ApiVersion ver JOIN ver.Api a ");
				tStrBuild.append("WHERE ver.buildTime > :time ");
				if (pProvider != null) {
					tStrBuild.append("AND a.provider like :provider ");
				}
				Query<Object> tQuery = session.createQuery(tStrBuild.toString());
				tQuery.setParameter("time", pBuildTime);
				if (pProvider != null) {
					tQuery.setParameter("provider", "%" + pProvider + "%");
				}
				List<?> list = tQuery.list();
				List<ApiVersion> tApiVersions = (List<ApiVersion>) list;

				if (tApiVersions.size() == 0) {
					return null;
				} else {
					return tApiVersions;
				}
			}
		});
	}
}
