package com.digiwin.metadatacache.dao.impl;

import com.digiwin.metadatacache.dao.ServiceDao;
import com.digiwin.metadatacache.model.Service;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.criterion.CriteriaSpecification;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@org.springframework.stereotype.Service
@Transactional("mdcTransactionManager")
public class ServiceDaoImpl extends GenericDaoImpl<Service, Long> implements ServiceDao {

	public ServiceDaoImpl() {
		super(Service.class);
	}

	// 查詢資料
	@SuppressWarnings("unchecked")
	@Override
	public List<Service> fetch(Map<String, String> pConditions) {
		return getHibernateTemplate().execute(new HibernateCallback<List<Service>>() {
			@Override
			public List<Service> doInHibernate(Session pSession) throws HibernateException {
				// Hibernate - Criteria Query 標準查詢
				Criteria tCriteria = pSession.createCriteria(Service.class);
				tCriteria.setResultTransformer(CriteriaSpecification.DISTINCT_ROOT_ENTITY);
				// 設查詢條件
				// containsKey - boolean containsKey(Object key) 判斷map是否包含指定的key
				// MatchMode.ANYWHERE 等同於 like="%name%"
				if (pConditions.containsKey("version")) {
					tCriteria.add(Restrictions.eq("version", pConditions.get("version")));
				}
				if (pConditions.containsKey("name")) {
					tCriteria.add(Restrictions.eq("name", pConditions.get("name")));
				}
				return tCriteria.list();
			}
		});
	}

	@Override
	public List<Service> getByPrdId(Long pPrdId) {
		return getHibernateTemplate().execute(new HibernateCallback<List<Service>>() {
			@SuppressWarnings("unchecked")
			@Override
			public List<Service> doInHibernate(Session pSession) throws HibernateException {
				Criteria tCriteria = pSession.createCriteria(Service.class);
				tCriteria.add(Restrictions.eq("product.id", pPrdId));
				return tCriteria.list();
			}
		});
	}

	// 查詢資料是否存在
	@Override
	public Service getByPrdIdAndServiceName(Long pPrdId, final String pServiceName) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(Service.class);
		tDetachedCriteria.add(Restrictions.eq("product.id", pPrdId));
		tDetachedCriteria.add(Restrictions.eq("name", pServiceName));
		@SuppressWarnings("unchecked")
		List<?> tServices = getHibernateTemplate().findByCriteria(tDetachedCriteria);
		if (tServices.size() == 0) {
			return null;
		} else {
			return (Service) tServices.get(0);
		}
	}

	@Override
	public void removeByPrdId(final Long pPrdId) {
		getHibernateTemplate().execute(new HibernateCallback<Service>() {
			@Override
			public Service doInHibernate(Session pSession) throws HibernateException {
				Query tQ = pSession.createQuery("delete Service where product.id is :productId");
				tQ.setParameter("productId", pPrdId);
				tQ.executeUpdate();
				return null;
			}
		});
	}

	@Override
	public void saveServices(final List<Service> pServices) {
		getHibernateTemplate().execute(new HibernateCallback() {
			@Override
			public Object doInHibernate(Session pSession) throws HibernateException {
				for (Service service : pServices) {
					pSession.merge(service);
				}
				return pSession;
			}
		});
	}

	// 查詢資料
	@SuppressWarnings("unchecked")
	@Override
	public List<Service> getByServiceNames(List<String> serviceNames) {
		return getHibernateTemplate().<List<Service>>execute(pSession -> {
			// Hibernate - Criteria Query 標準查詢
			Criteria tCriteria = pSession.createCriteria(Service.class);
			tCriteria.setResultTransformer(CriteriaSpecification.DISTINCT_ROOT_ENTITY);
			tCriteria.add(Restrictions.in("name", serviceNames));
			return tCriteria.list();
		});
	}

	@Override
	public List<Service> getByPrdAndServiceName(String prod, String prodUid, String pServiceName) {
		return getHibernateTemplate().execute(session -> {
			StringBuilder tStrBuild = new StringBuilder();
			tStrBuild.append("SELECT ver FROM Service ver JOIN ver.product p ");
			Map<String, Object> condition = new HashMap<>();
			tStrBuild.append("WHERE ver.name = :serviceName ");
			condition.put("serviceName", pServiceName);
			if (StringUtils.isNotBlank(prod)) {
				tStrBuild.append("AND p.name = :prod ");
				condition.put("prod", prod);
			}
			if (StringUtils.isNotBlank(prodUid)) {
				tStrBuild.append("AND p.uid = :prodUid");
				condition.put("prodUid", prodUid);
			}
			org.hibernate.query.Query<Service> tQuery = session.createQuery(tStrBuild.toString());
			condition.forEach((key,val) -> tQuery.setParameter(key, val));
			return tQuery.list();
		});
	}
}
