package com.digiwin.metadatacache.services;

import com.alibaba.fastjson.JSON;
import com.digiwin.metadatacache.constant.AssetConstant;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.search.TotalHits;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.List;
import java.util.Map;

@Service
public class ElasticsearchService {

    private final Log log = LogFactory.getLog(getClass());

    ObjectMapper objectMapper = new ObjectMapper();

    @Autowired
    private RestHighLevelClient restHighLevelClient;

    @Value("${asset_index_name:asset}")
    private String assertIndexName;

    /**
     * 资产中心-数据存贮
     */
    public void insertAssetDataDoc(String docId, Map<String, Object> docMessage) throws Exception {
        IndexRequest indexRequest = new IndexRequest(assertIndexName);
        indexRequest.source(JSON.toJSONString(docMessage), XContentType.JSON);
        indexRequest.id(docId);
        try {
            restHighLevelClient.index(indexRequest, RequestOptions.DEFAULT);
        } catch (IOException e) {
            log.error("新增文档失败,id: " + docId + " 异常 " + e);
        }
    }

    /**
     * 数据查询
     */
    public ObjectNode searchAssetDataDoc(Map<String, String> mapCondition) {
        ObjectNode objectNode = objectMapper.createObjectNode();
        //1、组转请求体
        SearchRequest searchRequest = new SearchRequest();
        searchRequest.indices(assertIndexName);
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        searchSourceBuilder.from(Integer.parseInt(mapCondition.get(AssetConstant.FROM)));
        searchSourceBuilder.size(Integer.parseInt(mapCondition.get(AssetConstant.SIZE)));
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        List<QueryBuilder> queryBuilders = boolQueryBuilder.must();
        // 资产来源
        if (null != mapCondition.get(AssetConstant.PROPERTY_SOURCE)) {
            queryBuilders.add(QueryBuilders.termQuery(AssetConstant.PROPERTY_SOURCE + AssetConstant.KEYWORD,
                    mapCondition.get(AssetConstant.PROPERTY_SOURCE)));
        }
        // 资产分类
        if (null != mapCondition.get(AssetConstant.PROPERTY_TYPE)) {
            queryBuilders.add(QueryBuilders.termQuery(AssetConstant.PROPERTY_TYPE + AssetConstant.KEYWORD,
                    mapCondition.get(AssetConstant.PROPERTY_TYPE)));
        }
        // 编号
        if (null != mapCondition.get(AssetConstant.CODE)) {
            queryBuilders.add(QueryBuilders.termQuery(AssetConstant.CODE + AssetConstant.KEYWORD,
                    mapCondition.get(AssetConstant.CODE)));
        }
        // 名称
        if (null != mapCondition.get(AssetConstant.NAME)) {
            queryBuilders.add(QueryBuilders.matchQuery(AssetConstant.NAME,
                    mapCondition.get(AssetConstant.NAME)));
            queryBuilders.add(QueryBuilders.matchQuery(AssetConstant.CONTENT_TABLES_NAME,
                    mapCondition.get(AssetConstant.NAME)));
            queryBuilders.add(QueryBuilders.matchQuery(AssetConstant.CONTENT_TABLES_FIELDS_NAME,
                    mapCondition.get(AssetConstant.NAME)));
        }
        // 资产归属
        if (null != mapCondition.get(AssetConstant.OWNER_PRODUCT)) {
            queryBuilders.add(QueryBuilders.termQuery(AssetConstant.OWNER_PRODUCT + AssetConstant.KEYWORD,
                    mapCondition.get(AssetConstant.OWNER_PRODUCT)));
        }
        // 内容
        if (null != mapCondition.get(AssetConstant.CONTENT)) {
            queryBuilders.add(QueryBuilders.matchPhraseQuery(AssetConstant.CONTENT,
                    mapCondition.get(AssetConstant.CONTENT)));
        }
        // 资产描述
        if (null != mapCondition.get(AssetConstant.PROPERTY_COMMENT)) {
            queryBuilders.add(QueryBuilders.matchPhraseQuery(AssetConstant.PROPERTY_COMMENT,
                    mapCondition.get(AssetConstant.PROPERTY_COMMENT)));
        }
        searchSourceBuilder.query(boolQueryBuilder);
        searchRequest.source(searchSourceBuilder);
        log.info("查询语句：" + searchRequest.toString());
        //2、执行请求获取数据
        try {
            SearchResponse search = restHighLevelClient.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits searchHits = search.getHits();
            if (null != searchHits) {
                TotalHits totalHits = searchHits.getTotalHits();
                long count = totalHits.value;
                objectNode.put(AssetConstant.COUNT, count);
                objectNode.put(AssetConstant.FROM, Integer.parseInt(mapCondition.get(AssetConstant.FROM)));
                objectNode.put(AssetConstant.SIZE, Integer.parseInt(mapCondition.get(AssetConstant.SIZE)));
                ArrayNode arrayNode = objectMapper.createArrayNode();
                objectNode.set(AssetConstant.DATA, arrayNode);
                SearchHit[] searchHitsHits = searchHits.getHits();
                for (SearchHit searchHit : searchHitsHits) {
                    Map<String, Object> map = searchHit.getSourceAsMap();
                    ObjectNode jsonNodes = objectMapper.createObjectNode();
                    jsonNodes.put(AssetConstant.CODE, null == map.get(AssetConstant.CODE) ? null : map.get(AssetConstant.CODE).toString());
                    jsonNodes.put(AssetConstant.NAME, null == map.get(AssetConstant.NAME) ? null : map.get(AssetConstant.NAME).toString());
                    jsonNodes.put(AssetConstant.OWNER_PRODUCT, null == map.get(AssetConstant.OWNER_PRODUCT) ? null : map.get(AssetConstant.OWNER_PRODUCT).toString());
                    jsonNodes.put(AssetConstant.PROPERTY_SOURCE, null == map.get(AssetConstant.PROPERTY_SOURCE) ? null : map.get(AssetConstant.PROPERTY_SOURCE).toString());
                    jsonNodes.put(AssetConstant.PROPERTY_TYPE, null == map.get(AssetConstant.PROPERTY_TYPE) ? null : map.get(AssetConstant.PROPERTY_TYPE).toString());
                    jsonNodes.put(AssetConstant.PROPERTY_CONTENT, null == map.get(AssetConstant.CONTENT) ? null : map.get(AssetConstant.CONTENT).toString());
                    arrayNode.add(jsonNodes);
                }
            }
        } catch (IOException exception) {
            exception.printStackTrace();
            log.error("查询es失败：" + exception.getMessage());
        }
        return objectNode;
    }

    /**
     * 删除文档
     * 异常如何处理
     *
     * @param docId 文档ID
     */
    public void deleteAssetDataDoc(String docId) throws Exception {
        DeleteRequest deleteRequest = new DeleteRequest(assertIndexName, docId);
        try {
            restHighLevelClient.delete(deleteRequest, RequestOptions.DEFAULT);
        } catch (IOException e) {
            log.info("删除文档失败,id: " + docId + " 异常 " + e);
        }
    }
}
