package com.digiwin.metadatacache.services;

import com.digiwin.metadatacache.MdcApplicationParameter;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.dao.ServiceDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.Service;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2024/7/22 15:40
 */
@org.springframework.stereotype.Service
public class MdcProductService {
    private static final Log log = LogFactory.getLog(MdcProductService.class);
    @Autowired
    private ServiceDao serviceDao;
    @Autowired
    private MdcCacheService cacheService;

    public Map<String, List<Product>> getProductFromDb(String tenantId, List<String> apiNames, Map<String, String> apiRelationMap) {
        List<String> realApiNames = apiNames.stream().map(api -> apiRelationMap.containsKey(api) ? apiRelationMap.get(api) : api).collect(Collectors.toList());
        List<Service> tServices = serviceDao.getByServiceNames(realApiNames);
        Map<String, List<Product>> productMap = new HashMap<>();
        if (CollectionUtils.isEmpty(tServices)) {
            return productMap;
        }
        boolean isStandard = true;
        List<Service> serviceList = tServices.stream().filter(x -> (x.getProduct().getProductType().equals(ApiTypeEnum.standard.toString())
                && null != x.getProduct().getTenantId() && tenantId.equals(x.getProduct().getTenantId()))).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(serviceList)) {
            isStandard = false;
            serviceList = tServices.stream().filter(x -> x.getProduct().getProductType().equals(ApiTypeEnum.agile.toString())).collect(Collectors.toList());
        }
        if (CollectionUtils.isEmpty(serviceList)) {
            return productMap;
        }
        //保存到缓存
        saveToCache(serviceList, tenantId, isStandard);
        Map<String, List<Product>> realApiProdMap = new HashMap<>();
        serviceList.forEach(srv -> {
            List<Product> prod = realApiProdMap.computeIfAbsent(srv.getName(), k -> new ArrayList<>());
            prod.add(srv.getProduct());
        });
        apiNames.stream().forEach(api -> {
            String realName = apiRelationMap.containsKey(api) ? apiRelationMap.get(api) : api;
            if (realApiProdMap.containsKey(realName)) {
                productMap.put(api, realApiProdMap.get(realName));
            }
        });
        return productMap;
    }

    public List<Product> getProductFromDb(String tenantId, String apiName, String prod, String uid) {
        List<Service> tServices = serviceDao.getByPrdAndServiceName(prod, uid, apiName);
        if (CollectionUtils.isEmpty(tServices)) {
            return null;
        }
        boolean isStandard = true;
        List<Service> serviceList = tServices.stream().filter(x -> (x.getProduct().getProductType().equals(ApiTypeEnum.standard.toString())
                && null != x.getProduct().getTenantId() && tenantId.equals(x.getProduct().getTenantId()))).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(serviceList)) {
            isStandard = false;
            serviceList = tServices.stream().filter(x -> x.getProduct().getProductType().equals(ApiTypeEnum.agile.toString())).collect(Collectors.toList());
        }
        if (CollectionUtils.isEmpty(serviceList)) {
            return null;
        }
        //保存到缓存
        saveToCache(serviceList, tenantId, isStandard);
        return serviceList.stream().map(Service::getProduct).collect(Collectors.toList());
    }

    public List<Product> getProductFromDbNoCache(String tenantId, String apiName, String prod, String uid) {
        List<Service> tServices = serviceDao.getByPrdAndServiceName(prod, uid, apiName);
        if (CollectionUtils.isEmpty(tServices)) {
            return null;
        }
        List<Service> serviceList = tServices.stream().filter(x -> (x.getProduct().getProductType().equals(ApiTypeEnum.standard.toString())
                && null != x.getProduct().getTenantId() && tenantId.equals(x.getProduct().getTenantId()))).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(serviceList)) {
            serviceList = tServices.stream().filter(x -> x.getProduct().getProductType().equals(ApiTypeEnum.agile.toString())).collect(Collectors.toList());
        }
        if (CollectionUtils.isEmpty(serviceList)) {
            return null;
        }
        return serviceList.stream().map(Service::getProduct).collect(Collectors.toList());
    }

    public void saveToCache(List<Service> services, String tenantId, boolean isStandard) {
        if (isStandard) {
            String tStandardKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + tenantId;
            Map<String, List<String>> standardCache = (Map<String, List<String>>) cacheService.get(tStandardKey);
            if (null == standardCache) {
                standardCache = new HashMap<>();
            }
            for (Service srv : services) {
                Product product = srv.getProduct();
                String key = buildStandardProductKey(product);
                List<String> srvCache = standardCache.computeIfAbsent(key, k -> new ArrayList<>());
                if (CollectionUtils.isEmpty(srvCache)) {
                    srvCache.add(buildProductInfo(product));
                }
                srvCache.add(srv.getName() + MdcSymbolConstant.LEFT_DASH);
            }
            cacheService.setSync(tStandardKey, standardCache);
        } else {
            String tAgileKeyPrefix = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON
                    + MdcConstant.AGILE + MdcSymbolConstant.COLON;
            services.stream().collect(Collectors.groupingBy(Service::getName)).forEach((srvName, srvs) -> {
                String key = tAgileKeyPrefix + srvName;
                Map<String, String> tAgileProductMap = (Map<String, String>) cacheService.get(key);
                if (null == tAgileProductMap) {
                    tAgileProductMap = new HashMap<>();
                }
                for (Service srv : srvs) {
                    String productKey = buildAgileProductKey(srv.getProduct());
                    String productInfo = buildProductInfo(srv.getProduct()) + ":::";
                    tAgileProductMap.put(productKey, productInfo);
                }
                cacheService.setSyncTimeOut(key, tAgileProductMap, MdcApplicationParameter._TIME_OUT + (int) (Math.random() * MdcApplicationParameter._TIME_RANDOM));
            });
        }
    }

    /**
     * 获取稳态产品信息 回传
     * 产品信息 + 服务信息
     *
     * @param standardCache      standardCache
     * @param pApiName pApiName
     * @return map
     */
    public List<Product> getStandardProductCache(String tenantId, Map<String, List<String>> standardCache, String pApiName) {
        // 進元數據快取中找出該租戶的產品服務清單Map List第一个对象是产品信息拼接,后面的是服务信息拼接
        if (null == standardCache) {
            return null;
        }
        Set<String> productCacheInfos = new HashSet<>();
        for (String tProductInfoKey : standardCache.keySet()) {
            List<String> apiInfoList = standardCache.get(tProductInfoKey);
            if (CollectionUtils.isNotEmpty(apiInfoList)) {
                //apiInfoList：第一条记录是产品信息，后面的记录是产品下的服务信息
                for (String tApiInfo : apiInfoList.subList(1, apiInfoList.size())) {
                    // 服务信息截断
                    String[] tApiInfoSplitAry = tApiInfo.split(MdcSymbolConstant.LEFT_DASH);
                    String tApiName = tApiInfoSplitAry[0];
                    if (pApiName.matches(tApiName)) {
                        // 产品详情
                        productCacheInfos.add(apiInfoList.get(0));
                        break;
                    }
                }
            }
        }
        return CollectionUtils.isEmpty(productCacheInfos) ? null : parseToProductInfo(tenantId, pApiName, productCacheInfos, ApiTypeEnum.standard.toString());
    }

    /**
     * 获取稳态产品信息 回传
     * 产品信息 + 服务信息
     *
     * @param tenantId tenantId
     * @param pApiName pApiName
     * @return map
     */
    public List<Product> getAgileProductCache(String tenantId, String pApiName) {
        // 查询敏态产品对应快取键值
        String tAgileKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON
                + MdcConstant.AGILE + MdcSymbolConstant.COLON + pApiName;
        Map<String, String> tProductServiceList = (Map<String, String>) cacheService.get(tAgileKey);
        if (null == tProductServiceList) {
            return null;
        }
        log.info("命中快取: " + tAgileKey);
        Set<String> productCacheInfos = new HashSet<>();
        for (String tProductInfoKey : tProductServiceList.keySet()) {
            String[] tApiInfoSplitAry = tProductServiceList.get(tProductInfoKey).split(":::");
            productCacheInfos.add(tApiInfoSplitAry[0]);
        }
        return CollectionUtils.isEmpty(productCacheInfos) ? null : parseToProductInfo(tenantId, pApiName, productCacheInfos, ApiTypeEnum.agile.toString());
    }

    private List<Product> parseToProductInfo(String tenantId, String apiName, Set<String> productCacheInfos, String productType) {
        Map<String, Product> productVOS = new HashMap<>();
        for (String productCacheInfo : productCacheInfos) {
            String[] tProductInfoSplitAry = productCacheInfo.split(MdcSymbolConstant.LEFT_DASH);
            String tProductName = tProductInfoSplitAry[0];
            String tProductUid = tProductInfoSplitAry[1];
            String tProductVersion = tProductInfoSplitAry[3];
            String tProductIp = tProductInfoSplitAry[4];
            String tProductId = tProductInfoSplitAry[5];
            String tProductGatewayId = tProductInfoSplitAry[6];
            String title = tProductName + tProductUid + tProductVersion + tProductIp + tProductId + tProductGatewayId;
            if (productVOS.containsKey(title)) {
                log.info("重复数据展示,租户Id: " + tenantId + "api名称: " + apiName + "redis内容解析: " + title);
                continue;
            }
            Product product = new Product();
            productVOS.put(title, product);
            product.setGatewayId(tProductGatewayId);
            product.setName(tProductName);
            product.setVersion(tProductVersion);
            product.setIp(tProductIp);
            product.setApid(tProductId);
            product.setUid(tProductUid);
            product.setProductType(productType);
        }
        return Lists.newArrayList(productVOS.values());
    }

    private String buildAgileProductKey(Product product) {
        return product.getName() + MdcSymbolConstant.COLON + product.getUid();
    }

    private String buildStandardProductKey(Product product) {
        return product.getName() + MdcSymbolConstant.COLON + product.getUid() + MdcSymbolConstant.COLON + product.getEaiUid();
    }

    private String buildProductInfo(Product product) {
        return  product.getName()
                + MdcSymbolConstant.LEFT_DASH + product.getUid()
                + MdcSymbolConstant.LEFT_DASH + product.getEaiUid()
                + MdcSymbolConstant.LEFT_DASH + product.getVersion()
                + MdcSymbolConstant.LEFT_DASH + product.getIp()
                + MdcSymbolConstant.LEFT_DASH + product.getApid()
                + MdcSymbolConstant.LEFT_DASH + product.getGatewayId();
    }

}
