package com.digiwin.metadatacache.apiservice;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.digiwin.app.common.DWApplicationConfigUtils;
import com.digiwin.app.service.DWServiceChainContext;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.http.client.exception.DWHttpFailedException;
import com.digiwin.metadatacache.annotate.AppTokenVerify;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.dao.ApiRelationDao;
import com.digiwin.metadatacache.dao.StateCodeDao;
import com.digiwin.metadatacache.enums.ErrorCodeEnum;
import com.digiwin.metadatacache.enums.ValidateStateEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.exception.TokenNotRetrievedException;
import com.digiwin.metadatacache.model.ApiRelation;
import com.digiwin.metadatacache.model.ApiVersion;
import com.digiwin.metadatacache.model.EocIntgMapping;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.services.MdcCacheService;
import com.digiwin.metadatacache.util.DateUtil;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.JsonSchemaValidator;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public abstract class AbstractApiService {

	protected final Log log = LogFactory.getLog(getClass());

	protected static final StateCode SUCCESS_STATE = createSuccessStateCode();

    /**
     * json訊息驗證器
     */
	final JsonSchemaValidator jsonSchemaValidator = new JsonSchemaValidator();

    /**
     *  mapper(parse or deserialize JSON content)
     */
	ObjectMapper mapper = new ObjectMapper();

    /**
     * 入参验证格式
     */
	protected String jsonSchemaFileName;

    protected String requestMsg;

    protected String locale = "";

	@Autowired
	protected MdcCacheService cacheService;

	@Autowired
	private StateCodeDao stateCodeDao;

	@Autowired
	protected ApiRelationDao apiRelationDao;

	public Map<String, Object> execute(String pRequestMsg) throws Exception {
		StateCode tStateCode;
		StringBuilder tDescription = new StringBuilder();
		requestMsg = pRequestMsg;
		if (DWServiceContext.getContext().getRequestHeader().get("locale") != null) {
			locale = DWServiceContext.getContext().getRequestHeader().get("locale").toString();
		}
		try {
			for (Annotation annotation : this.getClass().getAnnotations()) {
				if (AppTokenVerify.class.isAssignableFrom(annotation.annotationType())) {
					// 判定appToken 是否透传
					Object appToken = DWServiceContext.getContext().getRequestHeader().get(MdcConstant.DIGI_MIDDLEWARE_AUTH_APP);
					log.info("appToken: " + appToken);
					if (null == appToken) {
						throw new TokenNotRetrievedException("app_token鉴权失败",this.getClass());
					} else {
						// 解析token
						DecodedJWT jwt = JWT.decode(appToken.toString());
						String appId = jwt.getClaim(MdcConstant.ID).asString();
						if (StringUtils.isBlank(appId)) {
							throw new TokenNotRetrievedException("app_token鉴权失败",this.getClass());
						}
					}
				}
			}
			ValidatorResult tValidatorResult = validate(pRequestMsg);
			return processData(tValidatorResult);

		} catch (DWHttpFailedException e) {
			String exceptionEntity = e.getEntity(String.class);
			log.error("exceptionEntity = " + exceptionEntity + ";requestMsg = " + pRequestMsg, e);
            throw new Exception(exceptionEntity);
		} catch (BaseException e) {
			log.error(e.getMessage() + ";requestMsg = " + pRequestMsg, e);
			tStateCode = getStateCode(e.getStateEnum().getCode());
			tDescription.append(getDescriptionByLocale(tStateCode, locale));
			if (!StringUtil.isEmptyOrSpace(e.getMessage())) {
				tDescription.append(MdcSymbolConstant.COLON_CN).append(e.getMessage());
			}
		} catch (Exception e) {
			log.error(e.getMessage() + ";requestMsg = " + pRequestMsg, e);
			tStateCode = getStateCode(ValidateStateEnum.UNEXPECTED_ERROR.getCode());
			tDescription.append(getDescriptionByLocale(tStateCode, locale));
		}
		return converJsonNodeToMap(createResponseJsonNode(tStateCode.getCode(), tDescription.toString()));
	}

	protected ValidatorResult validate(String pMsg) throws Exception {
		return jsonSchemaValidator.validate(pMsg, jsonSchemaFileName);
	}

	/**
	 * 將運營單元對應依照租戶ID、產品名稱與產品UID分群
	 */
	protected Map<String, Map<String, Map<String, List<EocIntgMapping>>>> groupEocMappingByTenantAndProdNameAndUid(
			List<EocIntgMapping> pEocIntgMappings, String pTenantId) {
		// 總map
		Map<String, Map<String, Map<String, List<EocIntgMapping>>>> tTotalMap = new HashMap<>();
		for (EocIntgMapping mapping : pEocIntgMappings) {
			// 第一層key: 租戶ID
			Map<String, Map<String, List<EocIntgMapping>>> tProductNameMap = tTotalMap.get(pTenantId);
			if (MapUtils.isEmpty(tProductNameMap)) {
				// 建立一個此租戶的產品名稱 Map
				tProductNameMap = new HashMap<>();
				tTotalMap.put(pTenantId, tProductNameMap);
			}
			// 第二層key: 產品名稱
			Map<String, List<EocIntgMapping>> tProductUidMap = tProductNameMap.get(mapping.getProducName());
			if (MapUtils.isEmpty(tProductUidMap)) {
				// 建立一個此產品名稱的產品UID Map
				tProductUidMap = new HashMap<>();
				tProductNameMap.put(mapping.getProducName(), tProductUidMap);
			}
			// 第三層key: 產品UID
			List<EocIntgMapping> tEocIntgMappingList = tProductUidMap.get(mapping.getProductUid());
			if (CollectionUtils.isEmpty(tEocIntgMappingList)) {
				// 建立產品UID的EOC對應清單
				tEocIntgMappingList = new ArrayList<>();
				tProductUidMap.put(mapping.getProductUid(), tEocIntgMappingList);
			}
			tEocIntgMappingList.add(mapping);
		}
		return tTotalMap;
	}

	protected Map<String, Object> converJsonNodeToMap(JsonNode pJsonNode) {
		return mapper.convertValue(pJsonNode, new TypeReference<Map<String, Object>>() {
		});
	}

	protected Map<String, Object> converJsonStringToMap(String pJsonString) throws IOException {
		return converJsonNodeToMap(mapper.readTree(pJsonString));
	}

	protected StateCode getStateCode(String pStateCode) {
        return stateCodeDao.getStateCodeByCode(pStateCode);
	}

	protected ObjectNode createResponseJsonNode(String pCode, String pDescription, JsonNode pDataNode) {
		ObjectNode tResponseJsonNode = mapper.createObjectNode();
		ObjectNode tExecutionNode = mapper.createObjectNode();
		tResponseJsonNode.set(MdcConstant.EXECUTION, tExecutionNode);
		tExecutionNode.put(MdcConstant.CODE, pCode);
		if (!StringUtil.isEmptyOrSpace(pDescription)) {
			tExecutionNode.put(MdcConstant.DESCRIPTION, pDescription);
		}
		if (pDataNode != null) {
			tResponseJsonNode.set(MdcConstant.DATA, pDataNode);
		}
		// by 20230608 增加异常码处理
		checkErrorCode(pCode, pDescription, tResponseJsonNode);
		return tResponseJsonNode;
	}

	protected Map<String, Object> createResponseMap(String pCode, String pDescription, Map<String, Object> pDataNode) {
		Map<String, Object> response = new HashMap<>();
		Map<String, Object> tExecutionNode = new HashMap<>();
		response.put(MdcConstant.EXECUTION, tExecutionNode);
		tExecutionNode.put(MdcConstant.CODE, pCode);
		if (!StringUtil.isEmptyOrSpace(pDescription)) {
			tExecutionNode.put(MdcConstant.DESCRIPTION, pDescription);
		}
		if (pDataNode != null) {
			response.put(MdcConstant.DATA, pDataNode);
		}
		// by 20230608 增加异常码处理
		checkErrorCode(pCode, pDescription, response);
		return response;
	}

	/**
	 * 针对异常码响应进行检查，如果码是异常码进行添加链路信息
	 * @param code code
	 * @param description description
	 * @param tResponseJsonNode tResponseJsonNode
	 */
	private void checkErrorCode(String code, String description, ObjectNode tResponseJsonNode) {
		ErrorCodeEnum errorCode = ErrorCodeEnum.ERRORCODEENUM_CODE_MAP.get(code);
		if (null != errorCode) {
			tResponseJsonNode.put("errorCode", errorCode.getErrorCode());
            // 获取码值对应的异常信息
			tResponseJsonNode.put("errorMessage", description);
			String chainInfo = DWServiceChainContext.getContext().getTraceElement().getTraceInfo(true);
			chainInfo = StringUtils.substringAfter(chainInfo, MdcConstant.SUBSTRING);
			ObjectNode chainInfoNode = mapper.createObjectNode();
			chainInfoNode.put("chainInfo", chainInfo);
            // 获取信息链路信息
			tResponseJsonNode.set("errorInstructors", chainInfoNode);
		}
	}

	/**
	 * 针对异常码响应进行检查，如果码是异常码进行添加链路信息
	 * @param code code
	 * @param description description
	 * @param tResponseJsonNode tResponseJsonNode
	 */
	private void checkErrorCode(String code, String description, Map<String, Object> tResponseJsonNode) {
		ErrorCodeEnum errorCode = ErrorCodeEnum.ERRORCODEENUM_CODE_MAP.get(code);
		if (null != errorCode) {
			tResponseJsonNode.put("errorCode", errorCode.getErrorCode());
			// 获取码值对应的异常信息
			tResponseJsonNode.put("errorMessage", description);
			String chainInfo = DWServiceChainContext.getContext().getTraceElement().getTraceInfo(true);
			chainInfo = StringUtils.substringAfter(chainInfo, MdcConstant.SUBSTRING);
			ObjectNode chainInfoNode = mapper.createObjectNode();
			chainInfoNode.put("chainInfo", chainInfo);
			// 获取信息链路信息
			tResponseJsonNode.put("errorInstructors", chainInfoNode);
		}
	}

	private static StateCode createSuccessStateCode() {
		StateCode stateCode = new StateCode();
		stateCode.setCode(ValidateStateEnum.SUCCESS.getCode());
		stateCode.setDescription("執行成功");
		stateCode.setDescriptionEnUs("执行成功");
		stateCode.setDescriptionZhCn("执行成功");
		return stateCode;
	}

	private ObjectNode createResponseJsonNode(String pCode, String pDescription) {
		return createResponseJsonNode(pCode, pDescription, null);
	}

    /**
     * 钩子方法-由子类实现
     * @param validatorResult      * @param validatorResult
     * @return map
     * @throws BaseException BaseException
     * @throws Exception Exception
     */
	protected abstract Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception;

	/**
	 * 根据对应的类型获取描述信息
	 * @param pStateCode pStateCode
	 * @param pLocale pLocale
	 * @return String
	 */
	protected String getDescriptionByLocale(StateCode pStateCode, String pLocale) {
		String tDescriptionByLocale;
		switch (pLocale) {
		case MdcConstant.ZH_TW:
			tDescriptionByLocale = pStateCode.getDescription();
			break;
		case MdcConstant.EN_US:
			tDescriptionByLocale = pStateCode.getDescriptionEnUs();
			break;
		case MdcConstant.ZH_CN:
		default:
			tDescriptionByLocale = pStateCode.getDescriptionZhCn();
			break;
		}
		return tDescriptionByLocale;
	}

	/**
	 * 推送mq消息体
	 * @param pDataMap pDataMap
	 * @return String
	 */
	protected String sendMqMessage(Map<String, Object> pDataMap) {
		// 建立data節點
		ObjectNode tApiMetadataNode = mapper.createObjectNode();
		tApiMetadataNode.put(MdcConstant.API_NAME, pDataMap.get(MdcConstant.API_NAME).toString());
		// 如果API的tenant_id欄位有值則外顯
		if (pDataMap.get(MdcConstant.TENANT_ID) != null) {
			tApiMetadataNode.put(MdcConstant.TENANT_ID, pDataMap.get(MdcConstant.TENANT_ID).toString());
			// 如果package_name欄位有值則外顯
			if (pDataMap.get(MdcConstant.PACKAGE_NAME) != null) {
				tApiMetadataNode.put(MdcConstant.PACKAGE_NAME, pDataMap.get(MdcConstant.PACKAGE_NAME).toString());
			}
		}
		if (null != pDataMap.get(MdcConstant.API_EXTEND)) {
			tApiMetadataNode.put(MdcConstant.API_EXTEND, pDataMap.get(MdcConstant.API_EXTEND).toString());
		}
		String tDate = DateUtil.formatToDate(new Date());
		tApiMetadataNode.put(MdcConstant.TIMESTAMP, tDate);
		tApiMetadataNode.put(MdcConstant.ACTION, "add");
		tApiMetadataNode.put(MdcConstant.INVOKED_TYPE, pDataMap.get(MdcConstant.INVOKED_TYPE).toString());
		tApiMetadataNode.put(MdcConstant.CATEGORY, pDataMap.get(MdcConstant.CATEGORY).toString());
		ObjectNode tDescriptionNode = mapper.createObjectNode();
		tDescriptionNode.put(MdcConstant.ZH_TW, pDataMap.get(MdcConstant.DESCRIPTION_ZH_TW).toString());
		tDescriptionNode.put(MdcConstant.ZH_CN, pDataMap.get(MdcConstant.DESCRIPTION_ZH_CN).toString());
		tDescriptionNode.put(MdcConstant.EN, pDataMap.get(MdcConstant.DESCRIPTION_EN).toString());
		tApiMetadataNode.set(MdcConstant.DESCRIPTION, tDescriptionNode);
		ObjectNode tRemarkNode = mapper.createObjectNode();
		tRemarkNode.put(MdcConstant.ZH_TW, pDataMap.get(MdcConstant.REMARK_ZH_TW).toString());
		tRemarkNode.put(MdcConstant.ZH_CN, pDataMap.get(MdcConstant.REMARK_ZH_CN).toString());
		tRemarkNode.put(MdcConstant.EN, pDataMap.get(MdcConstant.REMARK_EN).toString());
		tApiMetadataNode.set(MdcConstant.REMARK, tRemarkNode);
		// API版本機制後
		ObjectNode tApiVersionInfo = (ObjectNode) pDataMap.get(MdcConstant.API_VERSION_INFO);
		List<String> keys = new ArrayList<>();
		Iterator<String> iterator = tApiVersionInfo.fieldNames();
		iterator.forEachRemaining(keys::add);
		tApiMetadataNode.put(MdcConstant.API_VERSION, keys.get(0));
		tApiMetadataNode.put(MdcConstant.URL, DWApplicationConfigUtils.getProperty("eaiUrl") + "/CROSS/RESTful");
		tApiMetadataNode.set(MdcConstant.DATA_METADATA, tApiVersionInfo.get(keys.get(0)));
		tApiMetadataNode.put(MdcConstant.IDEMPOTENCY, pDataMap.get(MdcConstant.IDEMPOTENCY).toString());
		return tApiMetadataNode.toString();
	}

	/**
	 * 通过租户+api查询父类Api名称
	 * @param apiName 子类Api名称
	 * @param tenantId 租户Id
	 * @return 父类Api名称
	 */
	protected String getParentName(String apiName, String tenantId) {
		String parentName = null;
		// 通过租户Id与api名称 查询影身关系
		List<ApiRelation> apiRelations = apiRelationDao.getRelationBySub(apiName, tenantId);
		if (CollectionUtils.isNotEmpty(apiRelations)) {
			List<ApiRelation> apiRelationList
					= apiRelations.stream().filter(x-> StringUtils.isNotBlank(x.getSubTenantId())
					&& tenantId.equals(x.getSubTenantId())).collect(Collectors.toList());
			if (CollectionUtils.isNotEmpty(apiRelationList)) {
				parentName = apiRelationList.get(0).getParentApiName();
			}else{
				parentName = apiRelations.get(0).getParentApiName();
			}
		}
		return parentName;
	}

	/**
	 * 查询最大版本API
	 * @param apiVersions 版本集
	 * @return ApiVersion
	 */
	protected ApiVersion getLastedVersion(List<ApiVersion> apiVersions) {
		String[] versionSplit;
		String[] maxVersionSplit;
		String versionInteger;
		String versionDecimal;
		String maxVersionInteger;
		String maxVersionDecimal;
		String tMaxVer = "0.0";
		ApiVersion tMaxApiVer = null;
		for (ApiVersion tApiVersion : apiVersions) {
			versionSplit = tApiVersion.getVersion().split("\\.");
			maxVersionSplit = tMaxVer.split("\\.");
			versionInteger = versionSplit[0];
			versionDecimal = versionSplit[1];
			maxVersionInteger = maxVersionSplit[0];
			maxVersionDecimal = maxVersionSplit[1];
			if (Integer.parseInt(versionInteger) > Integer.parseInt(maxVersionInteger)) {
				tMaxVer = tApiVersion.getVersion();
				tMaxApiVer = tApiVersion;
			} else if (Integer.valueOf(versionInteger).equals(Integer.valueOf(maxVersionInteger))) {
				if (Integer.parseInt(versionDecimal) > Integer.parseInt(maxVersionDecimal)) {
					tMaxVer = tApiVersion.getVersion();
					tMaxApiVer = tApiVersion;
				}
			}
		}
		return tMaxApiVer;
	}
}
