package com.digiwin.metadatacache.apiservice;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.util.DateUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.digiwin.metadatacache.dao.ApiDao;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.Api;
import com.digiwin.metadatacache.model.ApiVersion;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 根据租户取得API清單列表
 * Path:[/restful/standard/mdc/ApiList/Get]
 */
@Service
public class ApiListGetService extends AbstractApiService {

	@Autowired
	private ApiDao apiDao;

	public ApiListGetService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.API_LIST_GET_SCHEMA;
	}

	/**
	 * 查询API
	 * @param validatorResult      * @param validatorResult
	 * @return map
	 * @throws BaseException BaseException
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
				+ ", 原始訊息: " + tRequestJsonNode);
		// 建立data節點
		ObjectNode tDataNode = mapper.createObjectNode();
		// 建立api_info節點(陣列，多筆API資料)
		ArrayNode tApiInfoAryNode = mapper.createArrayNode();
		tDataNode.set(MdcConstant.API_INFO, tApiInfoAryNode);
		// 如果入參有tenant_id
		String tTenantId = null;
		if (tRequestJsonNode.get(MdcConstant.TENANT_ID) != null) {
			tTenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
		}
		List<Api> tList = getApiList(tTenantId);
		for (Api tApi : tList) {
			// 建立API資訊的節點
			ObjectNode tApiObject = mapper.createObjectNode();
			// 紀錄API名稱
			tApiObject.put(MdcConstant.NAME, tApi.getName().trim());
			// 建立API描述的節點(description)
			ObjectNode tDescriptionNode = mapper.createObjectNode();
			tDescriptionNode.put(MdcConstant.ZH_TW, tApi.getDescriptionZhTw());
			tDescriptionNode.put(MdcConstant.ZH_CN, tApi.getDescriptionZhCn());
			tDescriptionNode.put(MdcConstant.EN, tApi.getDescriptionEn());
			tApiObject.set(MdcConstant.DESCRIPTION, tDescriptionNode);
			// 建立API備注的節點(remark)
			ObjectNode tRemarkNode = mapper.createObjectNode();
			tRemarkNode.put(MdcConstant.ZH_TW, tApi.getRemarkZhTw());
			tRemarkNode.put(MdcConstant.ZH_CN, tApi.getRemarkZhCn());
			tRemarkNode.put(MdcConstant.EN, tApi.getRemarkEn());
			tApiObject.set(MdcConstant.REMARK, tRemarkNode);
			// 紀錄API分類
			tApiObject.put(MdcConstant.CATEGORY, tApi.getCategory());
			// 建立API版本陣列
			ArrayNode tVersionAryNode = mapper.createArrayNode();
			tApiObject.set(MdcConstant.VERSION, tVersionAryNode);
			List<ApiVersion> tApiVersions = tApi.getApiVersions();
			if (tApiVersions != null && !tApiVersions.isEmpty()) {
				int size = tApi.getApiVersions().size();
				for (ApiVersion tApiVersion : tApiVersions) {
					tVersionAryNode.add(tApiVersion.getVersion());
				}
				tApiObject.put(MdcConstant.LAST_UPDATE_TIME,
						DateUtil.formatToSecond(tApi.getApiVersions().get(size - 1).getBuildTime().getTime()));
			}
			// api_info陣列節點加入一筆API節點
			tApiInfoAryNode.add(tApiObject);
		}
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 取得所有API
	 * @param pTenantId 租户Id
	 * @return list
	 */
	private List<Api> getApiList(String pTenantId) {
		Map<String, String> pCondition = new HashMap<>();
		if (pTenantId != null) {
			pCondition.put(MdcConstant.TENANT_ID, pTenantId);
		}
		return apiDao.fetch(pCondition);
	}
}
