
package com.digiwin.metadatacache.apiservice;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.MdcApplicationParameter;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.digiwin.app.common.DWApplicationConfigUtils;
import com.digiwin.metadatacache.dao.ApiDao;
import com.digiwin.metadatacache.dao.ApiMetadataDao;
import com.digiwin.metadatacache.dao.ApiVersionDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.exception.ApiMetadataNotFoundException;
import com.digiwin.metadatacache.exception.ApiNameNotExistException;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.Api;
import com.digiwin.metadatacache.model.ApiMetadata;
import com.digiwin.metadatacache.model.ApiVersion;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 取得API元數據
 * Path:[/restful/standard/mdc/ApiMetadata/Get]
 */
@Service
public class ApiMetadataGetService extends AbstractApiService {

	@Autowired
	private ApiDao apiDao;

	@Autowired
	private ApiVersionDao apiVersionDao;

	@Autowired
	private ApiMetadataDao apiMetadataDao;

	public ApiMetadataGetService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.API_METADATA_GET_SCHEMA;
	}

	/**
	 * 获取API元数据
	 * @param validatorResult      * @param validatorResult
	 * @return Map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
				+ ", 原始訊息: " + tRequestJsonNode);
		// 建立data節點
		ObjectNode tDataNode = mapper.createObjectNode();
		// 建立api_info節點(陣列，多筆API資料)
		ArrayNode tApiInfoAryNode = mapper.createArrayNode();
		tDataNode.set(MdcConstant.API_INFO, tApiInfoAryNode);
		// 取得api_name、product_name、product_version
		String tApiName = tRequestJsonNode.get(MdcConstant.API_NAME).asText();
		String tTenantId = null;
		String tApiVersion = null;
		if (tRequestJsonNode.get(MdcConstant.TENANT_ID) != null) {
			tTenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
		}
		if (tRequestJsonNode.get(MdcConstant.API_VERSION) != null
				&& !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.API_VERSION).asText())) {
			tApiVersion = tRequestJsonNode.get(MdcConstant.API_VERSION).asText();
		}
		String tKey;
		if (tTenantId == null && tApiVersion == null) {
			tKey = CacheMapTypeEnum.api_metadata_cachemap.getCode() + MdcSymbolConstant.COLON + tApiName;
		} else if (tTenantId == null) {
			tKey = CacheMapTypeEnum.api_metadata_cachemap.getCode()+ MdcSymbolConstant.COLON + tApiName
					+ MdcSymbolConstant.COLON + tApiVersion;
		} else if (tApiVersion == null) {
			tKey = CacheMapTypeEnum.api_metadata_cachemap.getCode()+ MdcSymbolConstant.COLON + tTenantId
					+ MdcSymbolConstant.COLON + tApiName;
		} else {
			tKey = CacheMapTypeEnum.api_metadata_cachemap.getCode() + MdcSymbolConstant.COLON + tTenantId
					+ MdcSymbolConstant.COLON + tApiName + MdcSymbolConstant.COLON + tApiVersion;
		}
		// 1、進元數據快取中找出對應資料
		String tDataFromCache = (String) cacheService.get(tKey);
		if (tDataFromCache != null) {
			log.info("命中快取: " + tKey);
			return converJsonStringToMap(tDataFromCache);
		}
		// 2、設定DB尋找條件 pCondition
		Map<String, String> pCondition = new HashMap<>();
		pCondition.put(MdcConstant.NAME, tApiName);
		if (tTenantId != null) {
			pCondition.put(MdcConstant.TENANT_ID, tTenantId);
		}
		// 2.1、查询API
		List<Api> tApiList = apiDao.fetch(pCondition);
		Api tApi;
		// 找不到對應名稱的API
		if (CollectionUtils.isEmpty(tApiList)) {
			throw new ApiNameNotExistException(getClass());
		} else {
			tApi = tApiList.get(0);
		}
		// 2.2、根據API名稱、產品名稱、產品版本、API類型(穩or敏)找到對應版本元數據
		ApiMetadata tApiMetadata = getApiMetadata(tApi, tApiVersion);
		ApiVersion apiVersion = apiVersionDao.get(tApiMetadata.getApiVersionId());
		tDataNode = mapper.createObjectNode();
		tDataNode.put(MdcConstant.API_NAME, tApiName);
		// 如果API的tenant_id欄位有值則外顯
		if (!StringUtil.isEmptyOrSpace(tApi.getTenantId())) {
			tDataNode.put(MdcConstant.TENANT_ID, tApi.getTenantId());
			// 如果package_name欄位有值則外顯
			if (!StringUtil.isEmptyOrSpace(tApi.getPackageName())) {
				tDataNode.put(MdcConstant.PACKAGE_NAME, tApi.getPackageName());
			}
		}
		tDataNode.put(MdcConstant.INVOKED_TYPE, tApi.getInvokedType());
		tDataNode.put(MdcConstant.CATEGORY, tApi.getCategory());
		ObjectNode tDescriptionNode = mapper.createObjectNode();
		tDescriptionNode.put(MdcConstant.ZH_TW, tApi.getDescriptionZhTw());
		tDescriptionNode.put(MdcConstant.ZH_CN, tApi.getDescriptionZhCn());
		tDescriptionNode.put(MdcConstant.EN, tApi.getDescriptionEn());
		tDataNode.set(MdcConstant.DESCRIPTION, tDescriptionNode);
		ObjectNode tRemarkNode = mapper.createObjectNode();
		tRemarkNode.put(MdcConstant.ZH_TW, tApi.getRemarkZhTw());
		tRemarkNode.put(MdcConstant.ZH_CN, tApi.getRemarkZhCn());
		tRemarkNode.put(MdcConstant.EN, tApi.getRemarkEn());
		tDataNode.set(MdcConstant.REMARK, tRemarkNode);
		// API版本機制後
		tDataNode.put(MdcConstant.API_VERSION, apiVersion.getVersion());
		if (tApi.getApi_type().equals(ApiTypeEnum.agile.toString()) && tApi.getUrl() != null) {
			tDataNode.put(MdcConstant.URL, tApi.getUrl());
		}
		if (tApi.getApi_type().equals(ApiTypeEnum.standard.toString())) {
			tDataNode.put(MdcConstant.URL, DWApplicationConfigUtils.getProperty("eaiUrl") + "/CROSS/RESTful");
		}
		// convert JSON string to Map
		JsonNode actualObj = mapper.readTree(tApiMetadata.getMetadata());
		tDataNode.set(MdcConstant.DATA_METADATA, actualObj);
		tDataNode.put(MdcConstant.IDEMPOTENCY, apiVersion.isIdempotency());
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
		if (tStateCode.getCode().equals("000")) {
			cacheService.remove(tKey);
			cacheService.setSyncTimeOut(tKey, tResponseJsonNode.toString(), MdcApplicationParameter._TIME_OUT + (int) (Math.random() * MdcApplicationParameter._TIME_RANDOM));
		}
		return converJsonNodeToMap(tResponseJsonNode);
	}

	private ApiMetadata getApiMetadata(Api pApi, String pApiVersion) throws ApiMetadataNotFoundException {
		ApiVersion tApiVersion;
		// 沒有給API版本(新)
		if (pApiVersion == null) {
			// 找出最新的API版本
			tApiVersion = apiVersionDao.getLatestVerByApiId(pApi.getId());
		} else {
			tApiVersion = apiVersionDao.getByApiIdAndVer(pApi.getId(), pApiVersion);
		}
		// 有找到對應API版本
		if (tApiVersion != null) {
			// return tApiVersion.getApiMetadata();
			List<ApiMetadata> tApiMetadatas = apiMetadataDao.getByApiVersionId(tApiVersion.getId());
			if (tApiMetadatas != null) {
				return tApiMetadatas.get(0);
			}
		}
		// 都沒找到則拋出ApiMetadataNotFoundException
		throw new ApiMetadataNotFoundException(getClass());
	}
}
