package com.digiwin.metadatacache.apiservice;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.digiwin.metadatacache.MdcApplicationParameter;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.util.IamVerifyServiceUtil;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.digiwin.metadatacache.dao.ApiDao;
import com.digiwin.metadatacache.dao.ApiVersionDao;
import com.digiwin.metadatacache.dao.EaiTenantMappingDao;
import com.digiwin.metadatacache.dao.EocIntgMappingDao;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.dao.ServiceDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.enums.ValidateStateEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.Api;
import com.digiwin.metadatacache.model.ApiVersion;
import com.digiwin.metadatacache.model.EaiTenantMapping;
import com.digiwin.metadatacache.model.EocIntgMapping;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.Service;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 取得租户产品清单 Path:[/restful/standard/mdc/TenantProductList/Get]
 */
@org.springframework.stereotype.Service
public class TenantProductListGetService extends AbstractApiService {

	@Autowired
	private ApiDao apiDao;

	@Autowired
	private ProductDao productDao;

	@Autowired
	private ApiVersionDao apiVersionDao;

	@Autowired
	private ServiceDao serviceDao;

	@Autowired
	private EocIntgMappingDao eocIntgMappingDao;

	@Autowired
	private EaiTenantMappingDao eaiTenantMappingDao;

	public TenantProductListGetService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.TENANT_PRD_LIST_GET_SCHEMA;
	}

	/**
	 * 取得租户产品清单逻辑处理
	 *
	 * @param validatorResult validatorResult
	 * @return Map
	 * @throws BaseException BaseException
	 * @throws Exception     Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass() + ", 原始訊息: " + tRequestJsonNode);
		// 取出入參tenant, api_name
		String tTenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
		if (tRequestJsonNode.get(MdcConstant.API_NAME) != null && !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.API_NAME).asText())) {
			String tApiName = tRequestJsonNode.get(MdcConstant.API_NAME).asText();
			String parentApiName = getParentName(tApiName, tTenantId);
			return processApiImplementProductListGet(tTenantId, null == parentApiName ? tApiName : parentApiName);
		} else {
			//获取当前租户的版本
			String version = IamVerifyServiceUtil.getCurrentTenantVersion(tTenantId);
			return processProductListGet(tTenantId,version);
		}
	}

	/**
	 * 根據租戶ID取得符合條件的產品主機清單 回傳清單=[若該產品有某台主機綁定為該租戶->該主機]+[若該產品沒有任何主機綁定該租戶->該產品所有主機]
	 *
	 * @param pTenantId 租户Id
	 * @return map
	 */
	private Map<String, Object> processProductListGet(String pTenantId,String version) {
		// 組成回傳訊息必要節點
		ObjectNode tDataNode = mapper.createObjectNode();
		tDataNode.put(MdcConstant.TENANT_ID, pTenantId);
		ArrayNode tGatewayArrayNode = mapper.createArrayNode();
		tDataNode.set(MdcConstant.GATEWAY, tGatewayArrayNode);
		// 先取得該租戶所屬地端中台UID
		Map<String, String> tEaiTenantMappingCondition = new HashMap<>();
		tEaiTenantMappingCondition.put(MdcConstant.TENANT, pTenantId);
		List<EaiTenantMapping> tEaiTenantMappings = eaiTenantMappingDao.fetch(tEaiTenantMappingCondition);
		// 有對應中台UID
		if (!tEaiTenantMappings.isEmpty()) {
			// 取得該中台UID上的所有產品(tProductList)
			String eaiUid = tEaiTenantMappings.get(0).getEaUid();
			Map<String, String> pProductCondition = new HashMap<>();
			pProductCondition.put(MdcConstant.EAI_UID, eaiUid);
			List<Product> tEaiAllProductList = productDao.fetch(pProductCondition);
			// 將產品主機依照產品名稱分群放入各List
			Map<String, List<Product>> tProductMapGroupByProdName = new HashMap<>();
			for (Product tProduct : tEaiAllProductList) {
				List<Product> tProducts = tProductMapGroupByProdName.computeIfAbsent(tProduct.getName(), k -> new ArrayList<>());
				tProducts.add(tProduct);
			}
			List<Product> tReturnProductList = new ArrayList<>();
			// 根據各產品名稱找出有對應到租戶ID的產品主機
			for (String tProductType : tProductMapGroupByProdName.keySet()) {
				List<String> tExistUidList = new ArrayList<>();
				for (Product product : tProductMapGroupByProdName.get(tProductType)) {
					if (!tExistUidList.contains(product.getUid())) {
						tReturnProductList.add(product);
					}
					if (!StringUtil.isEmptyOrSpace(product.getUid())) {
						tExistUidList.add(product.getUid());
					}
				}
			}
			if (!tReturnProductList.isEmpty()) {
				// 將產品清單轉成分群Map
				Map<String, Map<String, Map<String, List<Product>>>> tProductsMap = transformListToGroupMap(tReturnProductList, pTenantId);
				// Gateway
				for (String tGatewayId : tProductsMap.get(pTenantId).keySet()) {
					ObjectNode tGatewayItemNode = mapper.createObjectNode();
					tGatewayItemNode.put(MdcConstant.GATEWAY_ID, tGatewayId);
					tGatewayItemNode.put(MdcConstant.EAI_UID, eaiUid);
					tGatewayArrayNode.add(tGatewayItemNode);
					ArrayNode tProductArrayNode = mapper.createArrayNode();
					// Product Name
					for (String tProductName : tProductsMap.get(pTenantId).get(tGatewayId).keySet()) {
						// Product
						for (Product tProduct : tProductsMap.get(pTenantId).get(tGatewayId).get(tProductName)) {
							ObjectNode tProductItemNode = mapper.createObjectNode();
							tProductItemNode.put(MdcConstant.NAME, tProduct.getName());
							tProductItemNode.put(MdcConstant.VERSION, tProduct.getVersion());
							tProductItemNode.put(MdcConstant.IP, tProduct.getIp());
							tProductItemNode.put(MdcConstant.APID, tProduct.getApid());
							tProductItemNode.put(MdcConstant.UID, tProduct.getUid());

							boolean tOmEnable = false;
							//老版本处理
							if(MdcConstant.TENANT_VERSION_V1.equalsIgnoreCase(version)){
								// 用產品名稱 產品UID去找有無配置運營
								Map<String, String> tEocIntgMappingCondition = new HashMap<>();
								tEocIntgMappingCondition.put(MdcConstant.PROD_NAME, tProduct.getName());
								tEocIntgMappingCondition.put(MdcConstant.PRODUCT_UID, tProduct.getUid());
								tEocIntgMappingCondition.put(MdcConstant.TENANT_ID, pTenantId);
								List<EocIntgMapping> tEocIntgMappings = eocIntgMappingDao.fetch(tEocIntgMappingCondition);
								// 有找到運營配置
								if (tEocIntgMappings != null && !tEocIntgMappings.isEmpty()) {
									for (EocIntgMapping mapping : tEocIntgMappings) {
										if (mapping.getEocLevel().equals(MdcConstant.COMPANY_ID) || mapping.getEocLevel().equals(MdcConstant.SITE_ID)) {
											tOmEnable = true;
											break;
										}
									}
								}
							}else {
								// 用產品名稱 產品UID去找有無配置運營 v2版本处理
								Map<String, String> tEocIntgMappingCondition = new HashMap<>();
								tEocIntgMappingCondition.put(MdcConstant.PROD_NAME, tProduct.getName());
								tEocIntgMappingCondition.put(MdcConstant.PRODUCT_UID, tProduct.getUid());
								tEocIntgMappingCondition.put(MdcConstant.TENANT_ID, pTenantId);
								tEocIntgMappingCondition.put(MdcConstant.EOC_LEVEL, MdcConstant.CLOUD_MAPPING_ID);
								List<EocIntgMapping> tEocIntgMappings = eocIntgMappingDao.fetch(tEocIntgMappingCondition);
								// 有找到運營配置
								if (tEocIntgMappings != null && !tEocIntgMappings.isEmpty()) {
									tOmEnable = tEocIntgMappings.stream()
											.anyMatch(mapping -> mapping.getCloudMappingType().equals(MdcConstant.COMPANY_ID)
													|| mapping.getCloudMappingType().equals(MdcConstant.SITE_ID));
								}
							}
							tProductItemNode.put(MdcConstant.OPERATION_MANAGE_ENABLE, tOmEnable);
							if (MdcApplicationParameter._ERP_LIST.contains(tProduct.getName())) {
								tProductItemNode.put(MdcConstant.TYPE, MdcConstant.ERP);
							} else {
								tProductItemNode.put(MdcConstant.TYPE, MdcConstant.ERPII);
							}
							tProductArrayNode.add(tProductItemNode);
						}
					}
					tGatewayItemNode.set(MdcConstant.PRODUCT, sortProductArray(tProductArrayNode));
				}
			}
		}
		StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 根據租戶ID、服務名稱取得符合條件的產品主機清單 回傳清單=[該租戶有實作該API的產品主機清單](API名稱須在元數據清單中) 结构分化
	 * 由于敏态数据量大，对敏态快取进行结构优化
	 *
	 * @param pTenantId 租户Id
	 * @param pApiName  Api名称
	 * @return map
	 */
	private Map<String, Object> processApiImplementProductListGet(String pTenantId, String pApiName) {
		// 查询稳态产品对应快取键值
		boolean judgeAgile = false;
		String tStandardKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + pTenantId;
		Map<String, String> tImplementProductMap = getStandardProductCatch(tStandardKey, pApiName);
		// 查询敏态产品对应快取键值
		String tAgileKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + MdcConstant.AGILE + MdcSymbolConstant.COLON + pApiName;
		if (tImplementProductMap.isEmpty()) {
			judgeAgile = true;
			tImplementProductMap = getAgileProductCache(tAgileKey);
		}
		// 构建返回对象主体
		ObjectNode tDataNode = mapper.createObjectNode();
		tDataNode.put(MdcConstant.TENANT_ID, pTenantId);
		ArrayNode tGatewayArrayNode = mapper.createArrayNode();
		tDataNode.set(MdcConstant.GATEWAY, tGatewayArrayNode);
		if (!tImplementProductMap.isEmpty()) {
			log.info("命中快取: " + (judgeAgile ? tAgileKey : tStandardKey));
			List<String> list = new ArrayList<>();
			for (String tProductInfoString : tImplementProductMap.keySet()) {
				ObjectNode tGatewayItemNode = mapper.createObjectNode();
				tGatewayArrayNode.add(tGatewayItemNode);
				ArrayNode tProductArrayNode = mapper.createArrayNode();
				tGatewayItemNode.set(MdcConstant.PRODUCT, tProductArrayNode);
				String[] tProductInfoSplitAry = tProductInfoString.split(MdcSymbolConstant.LEFT_DASH);
				String tProductName = tProductInfoSplitAry[0];
				String tProductUid = tProductInfoSplitAry[1];
				String tProductVersion = tProductInfoSplitAry[3];
				String tProductIp = tProductInfoSplitAry[4];
				String tProductId = tProductInfoSplitAry[5];
				String tProductGatewayId = tProductInfoSplitAry[6];
				String title = tProductName + tProductUid + tProductVersion + tProductIp + tProductId + tProductGatewayId;
				if (list.contains(title)) {
					log.info("重复数据展示,租户Id: " + pTenantId + "api名称: " + pApiName + "redis内容解析: " + title);
					continue;
				} else {
					list.add(title);
				}
				tGatewayItemNode.put(MdcConstant.GATEWAY_ID, tProductGatewayId);
				ObjectNode tProductItemNode = mapper.createObjectNode();
				tProductItemNode.put(MdcConstant.NAME, tProductName);
				tProductItemNode.put(MdcConstant.VERSION, tProductVersion);
				tProductItemNode.put(MdcConstant.IP, tProductIp);
				tProductItemNode.put(MdcConstant.APID, tProductId);
				tProductItemNode.put(MdcConstant.UID, tProductUid);
				// 如果產品沒有提供服務版本，則從服務版本快取中找
				if (StringUtil.isEmpty(tImplementProductMap.get(tProductInfoString))) {
					tProductItemNode.put(MdcConstant.API_VERSION, getImplementServiceVersion(pApiName, pTenantId));
				} else {
					// 如果產品有提供服務版本，則使用產品提供的版本號
					tProductItemNode.put(MdcConstant.API_VERSION, tImplementProductMap.get(tProductInfoString));
				}
				tProductArrayNode.add(tProductItemNode);
			}
			// 取得狀態碼及描述文字
			StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
			String tDescription = getDescriptionByLocale(tStateCode, locale);
			ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
			return converJsonNodeToMap(tResponseJsonNode);
		}
		// 查询服务信息
		Api tApi;
		// 如果服務名稱是以"uc."開頭的，表示是客製服務，則從資料庫中用租戶ID和API名稱找出對應API
		// 如果服務名稱不是以"uc."開頭的，則從資料庫中用服務名稱找出對應API
		if (pApiName.startsWith("uc.")) {
			tApi = apiDao.getByNameAndTenantId(pApiName, pTenantId);
		} else {
			tApi = apiDao.getByName(pApiName);
		}
		// 儲存API最新版本號的變數，初始值為空字串
		String tLatestVer = "";
		// 如果成功從資料庫中取得對應的API物件，則取得API最新版本號
		if (null != tApi) {
			// 取得API最新版本物件
			ApiVersion tApiLatestVersion = apiVersionDao.getLatestVerByApiId(tApi.getId());
			// 如果成功取得API最新版本物件，則將版本號儲存到tLatestVer變數中
			if (null != tApiLatestVersion) {
				tLatestVer = tApiLatestVersion.getVersion();
			}
		}
		Map<String, String> pServiceCondition = new HashMap<>();
		pServiceCondition.put(MdcConstant.NAME, pApiName.trim());
		List<Service> tServices = serviceDao.fetch(pServiceCondition);
		// 將符合條件的Service加入tReturnServiceList清單中即存在稳态也存在敏态
		// 如果是穩態產品，則必須符合租戶ID才能加入清單|如果是敏態應用，則直接加入清單
		List<Service> tReturnServiceList = tServices.stream()
				.filter(x -> (x.getProduct().getProductType().equals(ApiTypeEnum.standard.toString()) && null != x.getProduct().getTenantId() && pTenantId.equals(x.getProduct().getTenantId()))
						|| x.getProduct().getProductType().equals(ApiTypeEnum.agile.toString()))
				.collect(Collectors.toList());
		// 默认稳态
		boolean judgeType = false;
		if (CollectionUtils.isNotEmpty(tReturnServiceList)) {
			// 区分稳态||敏态
			List<Service> tServiceList = tReturnServiceList.stream().filter(x -> x.getProduct().getProductType().equals(ApiTypeEnum.standard.toString())).collect(Collectors.toList());
			if (CollectionUtils.isEmpty(tServiceList)) {
				judgeType = true;
				tServiceList = tReturnServiceList.stream().filter(x -> x.getProduct().getProductType().equals(ApiTypeEnum.agile.toString())).collect(Collectors.toList());
			}
			// 將服務清單按照分群轉換成Map<租户,gatewayId,产品,服务>
			Map<String, Map<String, Map<String, List<Service>>>> tServicesMap = transformListToGroupMap(tServiceList, pTenantId);
			// 將服務清單轉成分群Map後，根據不同的Gateway Id進行分類
			for (String tGatewayId : tServicesMap.get(pTenantId).keySet()) {
				// 創建一個ObjectNode，設定Gateway Id
				ObjectNode tGatewayItemNode = mapper.createObjectNode();
				tGatewayItemNode.put(MdcConstant.GATEWAY_ID, tGatewayId);
				// 加入tGatewayItemNode到Gateway Array Node中
				tGatewayArrayNode.add(tGatewayItemNode);
				// 創建一個Product Array Node，並設定到tGatewayItemNode中
				ArrayNode tProductArrayNode = mapper.createArrayNode();
				tGatewayItemNode.set(MdcConstant.PRODUCT, tProductArrayNode);
				// 根據Gateway Id分類後，再根據不同的Product Name進行分類
				for (String tProductName : tServicesMap.get(pTenantId).get(tGatewayId).keySet()) {
					// 根據Gateway Id和Product Name分類後，再依序設定Product相關資訊到ObjectNode中
					for (Service tService : tServicesMap.get(pTenantId).get(tGatewayId).get(tProductName)) {
						ObjectNode tProductItemNode = mapper.createObjectNode();
						tProductItemNode.put(MdcConstant.NAME, tService.getProduct().getName());
						tProductItemNode.put(MdcConstant.VERSION, tService.getProduct().getVersion());
						tProductItemNode.put(MdcConstant.IP, tService.getProduct().getIp());
						tProductItemNode.put(MdcConstant.APID, tService.getProduct().getApid());
						tProductItemNode.put(MdcConstant.UID, tService.getProduct().getUid());
						tProductItemNode.put(MdcConstant.PROD_TYPE, tService.getProduct().getProductType());
						tProductItemNode.put(MdcConstant.EAI_UID, tService.getProduct().getEaiUid());
						tProductItemNode.put(MdcConstant.SERVICE_NAME, tService.getName());
						if (!StringUtil.isEmptyOrSpace(tService.getVersion())) {
							tProductItemNode.put(MdcConstant.PROVIDER_API_VERSION, tService.getVersion());
						} else {
							tProductItemNode.put(MdcConstant.PROVIDER_API_VERSION, "");
						}
						// 判斷服務版本是否存在，並設定到tProductItemNode中
						if (null == tService.getVersion()) {
							// 如果產品沒有提供服務版本，則視為最新版本
							tProductItemNode.put(MdcConstant.API_VERSION, tLatestVer);
						} else {
							ApiVersion tApiVersion = null;
							if (null != tApi) {
								tApiVersion = apiVersionDao.getByApiIdAndVer(tApi.getId(), tService.getVersion());
							}
							if (tApiVersion == null) {
								// 如果服務版本不存在，則視為最新版本
								tProductItemNode.put(MdcConstant.API_VERSION, tLatestVer);
							} else {
								// 如果服務版本存在，則設定服務版本
								tProductItemNode.put(MdcConstant.API_VERSION, tService.getVersion());
							}
						}
						tProductArrayNode.add(tProductItemNode);
					}
				}
			}
		}
		StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
		// 將回傳訊息加入快取中
		if ("000".equals(tStateCode.getCode()) && CollectionUtils.isNotEmpty(tReturnServiceList)) {
			for (JsonNode tGatewayNode : tGatewayArrayNode) {
				setCache(tGatewayNode, tStandardKey, tAgileKey, judgeType);
			}
		}
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 这边需要区分稳态和敏态 因为存贮的结构不同 稳态<key,<key,list<String>>> 租户+产品 + 服务
	 * 敏态<key,list<String>> 服务 + 产品
	 *
	 * @param tGatewayNode tGatewayNode
	 * @param tStandardKey 稳态键值
	 * @param tAgileKey    敏态键值
	 * @param judgeType    类型 false-稳 |true-敏
	 */
	private void setCache(JsonNode tGatewayNode, String tStandardKey, String tAgileKey, boolean judgeType) {

		ArrayNode tProductArrayNode = (ArrayNode) tGatewayNode.get(MdcConstant.PRODUCT);
		if (judgeType) {
			Map<String, String> tAgileProductMap = (Map<String, String>) cacheService.get(tAgileKey);
			for (JsonNode tProductNode : tProductArrayNode) {
				String tProductFullInfo = tProductNode.get(MdcConstant.NAME).asText() + MdcSymbolConstant.LEFT_DASH + tProductNode.get(MdcConstant.UID).asText() + MdcSymbolConstant.LEFT_DASH
						+ tProductNode.get(MdcConstant.EAI_UID).asText() + MdcSymbolConstant.LEFT_DASH + tProductNode.get(MdcConstant.VERSION).asText() + MdcSymbolConstant.LEFT_DASH
						+ tProductNode.get(MdcConstant.IP).asText() + MdcSymbolConstant.LEFT_DASH + tProductNode.get(MdcConstant.APID).asText() + MdcSymbolConstant.LEFT_DASH
						+ tGatewayNode.get(MdcConstant.GATEWAY_ID).asText();
				String tProductKey = tProductNode.get(MdcConstant.NAME).asText() + MdcSymbolConstant.COLON + tProductNode.get(MdcConstant.UID).asText();
				tProductFullInfo = tProductFullInfo + ":::" + tProductNode.get(MdcConstant.PROVIDER_API_VERSION).asText();

				if (null == tAgileProductMap) {
					tAgileProductMap = new HashMap<>();
					tAgileProductMap.put(tProductKey, tProductFullInfo);
				} else {
					tAgileProductMap.put(tProductKey, tProductFullInfo);
				}
			}
			cacheService.setSyncTimeOut(tAgileKey, tAgileProductMap, MdcApplicationParameter._TIME_OUT + (int) (Math.random() * MdcApplicationParameter._TIME_RANDOM));
		} else {
			Map<String, List<String>> tProductServiceMapFromCache = (Map<String, List<String>>) cacheService.get(tStandardKey);
			for (JsonNode tProductNode : tProductArrayNode) {
				String tProductFullInfo = tProductNode.get(MdcConstant.NAME).asText() + MdcSymbolConstant.LEFT_DASH + tProductNode.get(MdcConstant.UID).asText() + MdcSymbolConstant.LEFT_DASH
						+ tProductNode.get(MdcConstant.EAI_UID).asText() + MdcSymbolConstant.LEFT_DASH + tProductNode.get(MdcConstant.VERSION).asText() + MdcSymbolConstant.LEFT_DASH
						+ tProductNode.get(MdcConstant.IP).asText() + MdcSymbolConstant.LEFT_DASH + tProductNode.get(MdcConstant.APID).asText() + MdcSymbolConstant.LEFT_DASH
						+ tGatewayNode.get(MdcConstant.GATEWAY_ID).asText();
				String tProductKey = tProductNode.get(MdcConstant.NAME).asText() + MdcSymbolConstant.COLON + tProductNode.get(MdcConstant.UID).asText();

				List<String> tService = new ArrayList<>();
				tService.add(tProductNode.get(MdcConstant.SERVICE_NAME).asText() + MdcSymbolConstant.LEFT_DASH + tProductNode.get(MdcConstant.PROVIDER_API_VERSION).asText());
				tProductKey = tProductKey + MdcSymbolConstant.COLON + tProductNode.get(MdcConstant.EAI_UID).asText();

				if (null == tProductServiceMapFromCache) {
					tProductServiceMapFromCache = new HashMap<>();
					List<String> tNewList = new ArrayList<>();
					tNewList.add(tProductFullInfo);
					tNewList.addAll(tService);
					tProductServiceMapFromCache.put(tProductKey, tNewList);

				} else if (null != tProductServiceMapFromCache.get(tProductKey)) {
					tProductServiceMapFromCache.get(tProductKey)
							.add(tProductNode.get(MdcConstant.SERVICE_NAME).asText() + MdcSymbolConstant.LEFT_DASH + tProductNode.get(MdcConstant.PROVIDER_API_VERSION).asText());
					cacheService.setSync(tStandardKey, tProductServiceMapFromCache);
				} else if (null == tProductServiceMapFromCache.get(tProductKey)) {
					List<String> tNewList = new ArrayList<>();
					tNewList.add(tProductFullInfo);
					tNewList.addAll(tService);
					tProductServiceMapFromCache.put(tProductKey, tNewList);
					cacheService.setSync(tStandardKey, tProductServiceMapFromCache);
				}
			}
			cacheService.setSync(tStandardKey, tProductServiceMapFromCache);
		}

	}

	/**
	 * 將List轉成Map，並依照租戶ID、GatewayID及產品名稱分群 <br>
	 * 第一層key: 租戶ID <br>
	 * 第二層key: GatewayID <br>
	 * 第三層key: 產品名稱
	 *
	 * @param pList   pList
	 * @param pTenant pTenant
	 * @return Map
	 */
	private <T> Map<String, Map<String, Map<String, List<T>>>> transformListToGroupMap(List<T> pList, String pTenant) {
		Map<String, Map<String, Map<String, List<T>>>> tTenantMap = new HashMap<>();
		for (T tObject : pList) {
			Product tProduct = null;
			if (tObject instanceof Product) {
				tProduct = ((Product) tObject);
			} else if (tObject instanceof Service) {
				tProduct = ((Service) tObject).getProduct();
			}
			if (tProduct != null) {
				// 第一層key: 租戶ID 建立一個此租戶的gateway Map
				Map<String, Map<String, List<T>>> tGatewayMap = tTenantMap.computeIfAbsent(pTenant, k -> new HashMap<>());
				// 第二層key: GatewayID 建立一個此gateway的產品名稱 Map
				Map<String, List<T>> tProductNameMap = tGatewayMap.computeIfAbsent(null == tProduct.getGatewayId() ? "null" : tProduct.getGatewayId(), k -> new HashMap<>());
				// 第三層key: 產品名稱
				List<T> tServiceList = tProductNameMap.computeIfAbsent(tProduct.getName(), k -> new ArrayList<>());
				// 建立產品名稱的產品主機list
				tServiceList.add(tObject);
			}
		}
		return tTenantMap;
	}

	/**
	 * 类型转换-排序
	 *
	 * @param pArrayNode pArrayNode
	 * @return ArrayNode
	 */
	public ArrayNode sortProductArray(ArrayNode pArrayNode) {
		List<JsonNode> erpList = new ArrayList<>();
		List<JsonNode> erpIIList = new ArrayList<>();
		for (JsonNode node : pArrayNode) {
			if (node.get(MdcConstant.TYPE).asText().equals(MdcConstant.ERP)) {
				erpList.add(node);
			} else {
				erpIIList.add(node);
			}
		}
		Collections.sort(erpList, new Comparator() {
			@Override
			public int compare(Object o1, Object o2) {
				String name1;
				String name2;
				name1 = ((ObjectNode) o1).get(MdcConstant.NAME).asText();
				name2 = ((ObjectNode) o2).get(MdcConstant.NAME).asText();
				return name1.compareTo(name2);
			}
		});
		Collections.sort(erpIIList, new Comparator() {
			@Override
			public int compare(Object o1, Object o2) {
				String name1;
				String name2;
				name1 = ((ObjectNode) o1).get(MdcConstant.NAME).asText();
				name2 = ((ObjectNode) o2).get(MdcConstant.NAME).asText();
				return name1.compareTo(name2);
			}
		});
		// 排序後的json array
		ArrayNode sortedJsonArray = mapper.createArrayNode();
		for (JsonNode jsonNode : erpList) {
			sortedJsonArray.add(jsonNode);
		}
		for (JsonNode jsonNode : erpIIList) {
			sortedJsonArray.add(jsonNode);
		}
		return sortedJsonArray;
	}

	/**
	 * 获取敏态快取信息 回传 产品主键 + 产品信息
	 *
	 * @param tAgileKey tAgileKey
	 * @return map
	 */
	private Map<String, String> getAgileProductCache(String tAgileKey) {
		// 進元數據快取中找出該租戶的產品服務清單Map <产品主键,产品信息>
		Map<String, String> tProductServiceList = (Map<String, String>) cacheService.get(tAgileKey);
		// 檢查底下所有產品的服務，有符合的都回傳
		Map<String, String> tImplementProductMap = new HashMap<>();
		if (null != tProductServiceList) {
			for (String tProductInfoKey : tProductServiceList.keySet()) {
				String[] tApiInfoSplitAry = tProductServiceList.get(tProductInfoKey).split(":::");
				String tProductInfo = tApiInfoSplitAry[0];
				String tApiVersion = "";
				// 存在只存贮了服务名称的情况(云地同步接口更新)
				if (tApiInfoSplitAry.length == 2) {
					tApiVersion = tApiInfoSplitAry[1];
				}
				tImplementProductMap.put(tProductInfo, tApiVersion);
			}
		}
		return tImplementProductMap;
	}

	/**
	 * 获取稳态产品信息 回传 产品信息 + 服务信息
	 *
	 * @param key      key
	 * @param pApiName pApiName
	 * @return map
	 */
	private Map<String, String> getStandardProductCatch(String key, String pApiName) {
		// 進元數據快取中找出該租戶的產品服務清單Map List第一个对象是产品信息拼接,后面的是服务信息拼接
		Map<String, List<String>> tProductServiceMapFromCache = (Map<String, List<String>>) cacheService.get(key);
		Map<String, String> tImplementProductMap = new HashMap<>();
		if (null != tProductServiceMapFromCache) {
			for (String tProductInfoKey : tProductServiceMapFromCache.keySet()) {
				List<String> apiInfoList = tProductServiceMapFromCache.get(tProductInfoKey);
				if (CollectionUtils.isNotEmpty(apiInfoList)) {
					for (String tApiInfo : apiInfoList.subList(1, apiInfoList.size())) {
						// 服务信息截断
						String[] tApiInfoSplitAry = tApiInfo.split(MdcSymbolConstant.LEFT_DASH);
						String tApiName = tApiInfoSplitAry[0];
						String tApiVersion = "";
						// 存在只存贮了服务名称的情况(云地同步接口更新)
						if (tApiInfoSplitAry.length == 2) {
							tApiVersion = tApiInfoSplitAry[1];
						}
						if (pApiName.matches(tApiName)) {
							// 产品详情 + 服务版本
							tImplementProductMap.put(apiInfoList.get(0), tApiVersion);
						}
					}
				}
			}
		}
		return tImplementProductMap;
	}

	/**
	 * 通过 apiName 获取对应的版本信息
	 *
	 * @param pApiName  pApiName
	 * @param pTenantId pTenantId
	 * @return String
	 */
	private String getImplementServiceVersion(String pApiName, String pTenantId) {
		String tLatestVersion;
		String tKey = CacheMapTypeEnum.api_version_list.getCode() + MdcSymbolConstant.COLON + pApiName;
		// 進快取中找出該服務的最新版本號
		tLatestVersion = (String) cacheService.get(tKey);
		// 沒找到，進DB找
		if (null == tLatestVersion) {
			Api tApi;
			// 如果是客製服務 則用tenant_id, api_name找出對應API
			if (pApiName.startsWith("uc.")) {
				tApi = apiDao.getByNameAndTenantId(pApiName, pTenantId);
			} else {
				// 取得符合服務名稱的api
				tApi = apiDao.getByName(pApiName);
			}
			// 取得API最新版本
			if (null != tApi) {
				ApiVersion tApiLatestVersion = apiVersionDao.getLatestVerByApiId(tApi.getId());
				if (null != tApiLatestVersion) {
					tLatestVersion = tApiLatestVersion.getVersion();
				}
			}
			// 有在DB找到，回寫
			if (null != tLatestVersion) {
				cacheService.setSyncTimeOut(tKey, tLatestVersion, MdcApplicationParameter._TIME_OUT + (int) (Math.random() * MdcApplicationParameter._TIME_RANDOM));
			}
		}
		return (null == tLatestVersion) ? "" : tLatestVersion;
	}
}
