package com.digiwin.metadatacache.apiservice;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.util.EocInfoServiceUtil;
import com.digiwin.metadatacache.util.IamVerifyServiceUtil;
import com.digiwin.metadatacache.util.IntgInfoServiceUtil;
import org.springframework.beans.factory.annotation.Autowired;

import com.digiwin.metadatacache.dao.EocIntgMappingDao;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.enums.EocLevelEnum;
import com.digiwin.metadatacache.enums.ValidateStateEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.EocIntgMapping;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 取得租戶產品運營清單(含運營階層關係)<br>
 * Path:[/restful/standard/mdc/TenantProductOperationList/Get]
 */
@org.springframework.stereotype.Service
public class TenantProductOperationListGetService extends AbstractApiService {

    @Autowired
    private EocIntgMappingDao eocIntgMappingDao;

    @Autowired
    private ProductDao productDao;

    public TenantProductOperationListGetService() {
        super();
        jsonSchemaFileName = JsonSchemaFileConstant.TENANT_PRD_OPERATION_LIST_GET_SCHEMA;
    }

    /**
     * 查询产品运营单元清单
     *
     * @param validatorResult * @param validatorResult
     * @return Map
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        // 取出入參tenant, api_name
        String tTenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
        String tUid = null;
        String tId = null;
        if (tRequestJsonNode.get(MdcConstant.UID) != null
                && !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.UID).asText())) {
            tUid = tRequestJsonNode.get(MdcConstant.UID).asText();
        }
        if (tRequestJsonNode.get(MdcConstant.ID) != null
                && !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.ID).asText())) {
            tId = tRequestJsonNode.get(MdcConstant.ID).asText();
        }
        //获取当前租户的版本
        String version = IamVerifyServiceUtil.getCurrentTenantVersion(tTenantId);
        if (tRequestJsonNode.get(MdcConstant.PROD_NAME) != null
                && !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.PROD_NAME).asText())) {
            String tProdName = tRequestJsonNode.get(MdcConstant.PROD_NAME).asText();
            //老版本处理
            if(MdcConstant.TENANT_VERSION_V1.equalsIgnoreCase(version)){
                return processTenantProdOperationListGet(tTenantId, tProdName, tUid, tId);
            }else {
                return processTenantProdOperationListGetV2(tTenantId, tProdName, tUid, tId);
            }
        } else {
            // 如果有註明需要"敘述" "name_include": true 則先取EOC運營單元碼清單
            boolean tNameInclude = false;
            if (tRequestJsonNode.get(MdcConstant.NAME_INCLUDE) != null) {
                tNameInclude = tRequestJsonNode.get(MdcConstant.NAME_INCLUDE).asBoolean();
            }
            //老版本处理
            if(MdcConstant.TENANT_VERSION_V1.equalsIgnoreCase(version)){
                return processTenantOperationListGet(tTenantId, tNameInclude);
            }else {
                return processTenantOperationListGetV2(tTenantId, tNameInclude);
            }
        }
    }

    /**
     * 取得租戶所有產品的運營清單(蔣穎EOC配置畫面呼叫)
     */
    private Map<String, Object> processTenantOperationListGet(String pTenantId, Boolean pNameInclude) {
        /* 根據入參tenant_id取得eoc_intg_mapping資料 */
        Map<String, String> pCondition = new HashMap<>();
        pCondition.put(MdcConstant.TENANT_ID, pTenantId);
        List<EocIntgMapping> tMappings = eocIntgMappingDao.fetch(pCondition);
        ObjectNode tEocNode = null;
        // NameInclude=true，代表回傳訊息中要包含運營單元的描述(name)，須向EOC取得
        // 若該租戶還未配置運營，則不用取得token呼叫EOC
        if (pNameInclude && tMappings.size() != 0) {
            try {
                // 根據租戶ID取得EOC運營單元碼
                tEocNode = EocInfoServiceUtil.getEocInfoByTenantId(pTenantId);
                // 取得eoc清單時發生任何錯誤都繼續執行，因為取得name值(敘述)是額外的資訊
                // 不要因為取得name值(敘述)失敗，而影響取得運營大表
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
        // 組成org_type_enterprise物件
        ObjectNode tEnterpriseObjectNode = mapper.createObjectNode();
        List<EocIntgMapping> tEnterpriseIdMappings = new ArrayList<>();
        /* 取出enterprise_id層級的mapping */
        for (EocIntgMapping mapping : tMappings) {
            if (mapping.getEocLevel().equals(EocLevelEnum.enterprise_id.toString())) {
                tEnterpriseIdMappings.add(mapping);
            }
        }
        if (tEnterpriseIdMappings.size() != 0) {
            tEnterpriseObjectNode.put(MdcConstant.EOC_ENTERPRISE_ID, tEnterpriseIdMappings.get(0).getEocId());
            tEnterpriseObjectNode.put(MdcConstant.OM_ENTERPRISE_ID, tEnterpriseIdMappings.get(0).getIntgId());
        } else {
            tEnterpriseObjectNode.put(MdcConstant.EOC_ENTERPRISE_ID, "");
            tEnterpriseObjectNode.put(MdcConstant.OM_ENTERPRISE_ID, "");
        }
        // 將tMappings依照產品名稱分出來
        Map<String, List<EocIntgMapping>> tMappingsGroupByProductName = new HashMap<>();
        for (EocIntgMapping mapping : tMappings) {
            if (tMappingsGroupByProductName.get(mapping.getProducName()) == null) {
                List<EocIntgMapping> tProductNameMappingList = new ArrayList<>();
                tProductNameMappingList.add(mapping);
                tMappingsGroupByProductName.put(mapping.getProducName(), tProductNameMappingList);
            } else {
                tMappingsGroupByProductName.get(mapping.getProducName()).add(mapping);
            }
        }
        ArrayNode tProdEocMappingAryNode = mapper.createArrayNode();
        for (String tProductName : tMappingsGroupByProductName.keySet()) {
            JsonNode tIntgInfoNode = null;
            // 如果有註明需要"敘述" "name_include": true 則根據產品UID取得ESP單元對應碼
            try {
                if (pNameInclude) {
                    /* 取得ESP單元對應碼 */
                    // 根據tProdName, tUid呼叫ESP->ERP取得集成代碼
                    tIntgInfoNode = IntgInfoServiceUtil.getIntgInfo(null, pTenantId, tProductName,
                            tMappingsGroupByProductName.get(tProductName).get(0).getProductUid());
                }
                // 取得地端om時發生任何錯誤也略過 因為取得name_include是額外的功能
                // 不要因為取得name_include失敗而影響取得運營大表
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
            List<EocIntgMapping> tCompanyIdMappings = new ArrayList<>();
            List<EocIntgMapping> tSiteIdMappings = new ArrayList<>();
            List<EocIntgMapping> tRegionIdMappings = new ArrayList<>();
            /*將mappings依據eoc_level分出來 */
            for (EocIntgMapping mapping : tMappingsGroupByProductName.get(tProductName)) {
                if (mapping.getEocLevel().equals(EocLevelEnum.company_id.toString())) {
                    tCompanyIdMappings.add(mapping);
                } else if (mapping.getEocLevel().equals(EocLevelEnum.site_id.toString())) {
                    tSiteIdMappings.add(mapping);
                } else if (mapping.getEocLevel().equals(EocLevelEnum.region_id.toString())) {
                    tRegionIdMappings.add(mapping);
                }
            }
            // 個別產品名稱的運營物件
            ObjectNode tProdEocMappingObjectNode = mapper.createObjectNode();
            if (pNameInclude) {
                Map<String, String> tPrdCondition = new HashMap<>();
                tPrdCondition.put(MdcConstant.NAME, tProductName);
                tPrdCondition.put(MdcConstant.UID, tMappingsGroupByProductName.get(tProductName).get(0).getProductUid());
                List<Product> tProducts = productDao.fetch(tPrdCondition);
                String tEocSetting = "";
                String tEocSettingRegion = "";
                if (tProducts != null && tProducts.size() != 0 && tProducts.get(0).getEocSetting() != null) {
                    tEocSetting = tProducts.get(0).getEocSetting();
                    tEocSettingRegion = tProducts.get(0).getEocSettingRegion();
                }
                tProdEocMappingObjectNode.put(MdcConstant.EOC_SETTING, tEocSetting);
                tProdEocMappingObjectNode.put(MdcConstant.EOC_SETTING_REGION, tEocSettingRegion);
            }
            tProdEocMappingObjectNode.put(MdcConstant.PROD_NAME, tProductName);
            tProdEocMappingObjectNode.put(MdcConstant.UID, tMappingsGroupByProductName.get(tProductName).get(0).getProductUid());

            /* 組成回傳json */
            // 組成org_type_company陣列
            ArrayNode tOrgTypeCompanyArrayNode = mapper.createArrayNode();
            tProdEocMappingObjectNode.set(MdcConstant.ORG_TYPE_COMPANY, tOrgTypeCompanyArrayNode);
            // 組成eoc_level = company_id的物件
            // 從某產品的org_type_company mappings開始掃
            for (EocIntgMapping mapping : tCompanyIdMappings) {
                ObjectNode tCompanyLevelMappingNode = mapper.createObjectNode();
                tCompanyLevelMappingNode.put(MdcConstant.EOC_COMPANY_ID, mapping.getEocId());
                if (pNameInclude) {
                    ObjectNode tEocDataNode = (ObjectNode) tEocNode.get(MdcConstant.DATA);
                    ArrayNode tEocCompanyAryNode = (ArrayNode) tEocDataNode.get(MdcConstant.COMPANIES);
                    for (JsonNode tEocCompanyObject : tEocCompanyAryNode) {
                        if (tEocCompanyObject.get(MdcConstant.ID).asText().equals(mapping.getEocId())) {
                            tCompanyLevelMappingNode.set(MdcConstant.EOC_COMPANY_NAME, tEocCompanyObject.get(MdcConstant.NAME));
                        }
                    }
                    if (tIntgInfoNode != null) {
                        ArrayNode tIntgCompanyAryNode = (ArrayNode) tIntgInfoNode.get(MdcConstant.ORG_TYPE_COMPANY);
                        for (JsonNode tIntgCompanyObject : tIntgCompanyAryNode) {
                            if (mapping.getIntgId().equals(tIntgCompanyObject.get(MdcConstant.COMPANY_NO).asText())) {
                                tCompanyLevelMappingNode.set(MdcConstant.OM_COMPANY_NAME, tIntgCompanyObject.get(MdcConstant.COMPANY_NAME));
                            }
                        }
                    }
                }
                tCompanyLevelMappingNode.put(MdcConstant.OM_COMPANY_ID, mapping.getIntgId());
                if (mapping.getDatakey() != null && mapping.getDatakey().length() != 0) {
                    tCompanyLevelMappingNode.put(MdcConstant.DATAKEY, mapping.getDatakey());
                }
                tOrgTypeCompanyArrayNode.add(tCompanyLevelMappingNode);
            }
            // 組成eoc_level = site_id的物件，並放入parent物件
            for (EocIntgMapping mapping : tSiteIdMappings) {
                ObjectNode tSiteLevelMappingNode = mapper.createObjectNode();
                tSiteLevelMappingNode.put(MdcConstant.EOC_SITE_ID, mapping.getEocId());
                if (pNameInclude) {
                    ObjectNode tEocDataNode = (ObjectNode) tEocNode.get(MdcConstant.DATA);
                    ArrayNode tEocFactoryAryNode = (ArrayNode) tEocDataNode.get(MdcConstant.FACTORIES);
                    for (JsonNode tEocFactoryObject : tEocFactoryAryNode) {
                        if (tEocFactoryObject.get(MdcConstant.ID).asText().equals(mapping.getEocId())
                                && mapping.getEocIntgMapping() != null && tEocFactoryObject.get(MdcConstant.COMPANYID) != null
                                && mapping.getEocIntgMapping().getEocId()
                                .equals(tEocFactoryObject.get(MdcConstant.COMPANYID).asText())) {
                            tSiteLevelMappingNode.set(MdcConstant.EOC_SITE_NAME, tEocFactoryObject.get(MdcConstant.NAME));
                        }
                    }
                    if (tIntgInfoNode != null) {
                        ArrayNode tIntgCompanyAryNode = (ArrayNode) tIntgInfoNode.get(MdcConstant.ORG_TYPE_COMPANY);
                        for (JsonNode tIntgCompanyObject : tIntgCompanyAryNode) {
                            if (mapping.getEocIntgMapping() != null && mapping.getEocIntgMapping().getEocId()
                                    .equals(tIntgCompanyObject.get(MdcConstant.COMPANY_NO).asText())) {

                                if (tIntgCompanyObject.get(MdcConstant.ORG_TYPE_SITE) != null) {

                                    for (JsonNode tIntgSiteObject : tIntgCompanyObject.get(MdcConstant.ORG_TYPE_SITE)) {
                                        if (mapping.getIntgId().equals(tIntgSiteObject.get(MdcConstant.SITE_NO).asText())) {
                                            tSiteLevelMappingNode.set(MdcConstant.OM_SITE_NAME, tIntgSiteObject.get(MdcConstant.SITE_NAME));
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                tSiteLevelMappingNode.put(MdcConstant.OM_SITE_ID, mapping.getIntgId());
                if (mapping.getDatakey() != null && mapping.getDatakey().length() != 0) {
                    tSiteLevelMappingNode.put(MdcConstant.DATAKEY, mapping.getDatakey());
                }
                // 有parent mapping
                if (mapping.getEocIntgMapping() != null) {
                    // 找到parent節點
                    for (JsonNode tCompanyNode : tOrgTypeCompanyArrayNode) {
                        if (tCompanyNode.get(MdcConstant.EOC_COMPANY_ID).asText().equals(mapping.getEocIntgMapping().getEocId())) {
                            // parent節點已經有"org_type_site"陣列
                            if (tCompanyNode.get(MdcConstant.ORG_TYPE_SITE) != null) {
                                ((ArrayNode) tCompanyNode.get(MdcConstant.ORG_TYPE_SITE)).add(tSiteLevelMappingNode);
                                // parent節點沒有"org_type_site"陣列
                            } else {
                                ArrayNode tOrgTypeSiteArrayNode = mapper.createArrayNode();
                                ((ObjectNode) tCompanyNode).set(MdcConstant.ORG_TYPE_SITE, tOrgTypeSiteArrayNode);
                                tOrgTypeSiteArrayNode.add(tSiteLevelMappingNode);
                            }
                        }
                    }
                }
            }
            // 組成org_type_region陣列
            ArrayNode tOrgTypeRegionArrayNode = mapper.createArrayNode();
            tProdEocMappingObjectNode.set(MdcConstant.ORG_TYPE_REGION, tOrgTypeRegionArrayNode);
            // 組成eoc_level = region_id的物件
            // 從某產品的org_type_region mappings開始掃
            for (EocIntgMapping mapping : tRegionIdMappings) {
                ObjectNode tRegionLevelMappingNode = mapper.createObjectNode();
                tRegionLevelMappingNode.put(MdcConstant.EOC_REGION_ID, mapping.getEocId());
                if (pNameInclude) {
                    ObjectNode tEocDataNode = (ObjectNode) tEocNode.get(MdcConstant.DATA);
                    ArrayNode tEocRegionAryNode = (ArrayNode) tEocDataNode.get(MdcConstant.AREAS);
                    for (JsonNode tEocRegionObject : tEocRegionAryNode) {
                        if (tEocRegionObject.get(MdcConstant.ID).asText().equals(mapping.getEocId())
                                && tEocRegionObject.get(MdcConstant.TYPE).asText().equals(mapping.getRegionType())) {
                            tRegionLevelMappingNode.set(MdcConstant.EOC_REGION_NAME, tEocRegionObject.get(MdcConstant.NAME));
                        }
                    }
                    if (tIntgInfoNode != null) {
                        ArrayNode tIntgRegionAryNode = (ArrayNode) tIntgInfoNode.get(MdcConstant.ORG_TYPE_REGION);

                        if (tIntgRegionAryNode != null) {
                            for (JsonNode tIntgRegionObject : tIntgRegionAryNode) {
                                if (mapping.getIntgId().equals(tIntgRegionObject.get(MdcConstant.REGION_NO).asText()) && mapping
                                        .getRegionType().equals(tIntgRegionObject.get(MdcConstant.REGION_TYPE).asText())) {
                                    tRegionLevelMappingNode.set(MdcConstant.OM_REGION_NAME, tIntgRegionObject.get(MdcConstant.REGION_NAME));
                                }
                            }
                        }
                    }
                }
                tRegionLevelMappingNode.put(MdcConstant.OM_REGION_ID, mapping.getIntgId());
                if (mapping.getRegionType() == null) {
                    tRegionLevelMappingNode.put(MdcConstant.REGION_TYPE, "");
                } else {
                    tRegionLevelMappingNode.put(MdcConstant.REGION_TYPE, mapping.getRegionType());
                }
                if (mapping.getDatakey() != null && mapping.getDatakey().length() != 0) {
                    tRegionLevelMappingNode.put(MdcConstant.DATAKEY, mapping.getDatakey());
                }
                tOrgTypeRegionArrayNode.add(tRegionLevelMappingNode);
            }
            tProdEocMappingAryNode.add(tProdEocMappingObjectNode);
        }
        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
        tResponseJsonNode.set(MdcConstant.ORG_TYPE_ENTERPRISE, tEnterpriseObjectNode);
        tResponseJsonNode.set("prod_eoc_mapping", tProdEocMappingAryNode);
        tResponseJsonNode.put(MdcConstant.TENANT_ID, pTenantId);
        return converJsonNodeToMap(tResponseJsonNode);
    }

    /**
     * 取得租戶所有產品的運營清單(蔣穎EOC配置畫面呼叫) v2版本
     */
    private Map<String, Object> processTenantOperationListGetV2(String pTenantId, Boolean pNameInclude) {
        //根據入參tenant_id、eocLevel取得eoc_intg_mapping v2 資料
        Map<String, String> pCondition = new HashMap<>();
        pCondition.put(MdcConstant.TENANT_ID, pTenantId);
        pCondition.put(MdcConstant.EOC_LEVEL, MdcConstant.CLOUD_MAPPING_ID);
        List<EocIntgMapping> tMappings = eocIntgMappingDao.fetchGroupByEocId(pCondition);

        Map<String, String> tEocNode = new HashMap<>();
        // NameInclude=true，代表回傳訊息中要包含運營單元的描述(name)，須向EOC取得
        // 若該租戶還未配置運營，則不用取得token呼叫EOC
        if (pNameInclude && !tMappings.isEmpty()) {
            try {
                // 根據租戶ID取得EOC運營單元碼 v2版本接口
                tEocNode = EocInfoServiceUtil.getEocInfoV2MapByTenantId(pTenantId);
                // 取得eoc清單時發生任何錯誤都繼續執行，因為取得name值(敘述)是額外的資訊
                // 不要因為取得name值(敘述)失敗，而影響取得運營大表
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }

        // 將tMappings依照產品名稱分出來
        Map<String, List<EocIntgMapping>> tMappingsGroupByProductName = new HashMap<>();
        for (EocIntgMapping mapping : tMappings) {
            if (tMappingsGroupByProductName.get(mapping.getProducName()) == null) {
                List<EocIntgMapping> tProductNameMappingList = new ArrayList<>();
                tProductNameMappingList.add(mapping);
                tMappingsGroupByProductName.put(mapping.getProducName(), tProductNameMappingList);
            } else {
                tMappingsGroupByProductName.get(mapping.getProducName()).add(mapping);
            }
        }
        ArrayNode tProdEocMappingAryNode = mapper.createArrayNode();
        for (String tProductName : tMappingsGroupByProductName.keySet()) {

            // 個別產品名稱的運營物件
            ObjectNode tProdEocMappingObjectNode = mapper.createObjectNode();
            if (pNameInclude) {
                Map<String, String> tPrdCondition = new HashMap<>();
                tPrdCondition.put(MdcConstant.NAME, tProductName);
                tPrdCondition.put(MdcConstant.UID, tMappingsGroupByProductName.get(tProductName).get(0).getProductUid());
                List<Product> tProducts = productDao.fetch(tPrdCondition);
                String tEocSetting = "";
                String tEocSettingRegion = "";
                if (tProducts != null && tProducts.size() != 0 && tProducts.get(0).getEocSetting() != null) {
                    tEocSetting = tProducts.get(0).getEocSetting();
                    tEocSettingRegion = tProducts.get(0).getEocSettingRegion();
                }
                tProdEocMappingObjectNode.put(MdcConstant.EOC_SETTING, tEocSetting);
                tProdEocMappingObjectNode.put(MdcConstant.EOC_SETTING_REGION, tEocSettingRegion);
            }
            tProdEocMappingObjectNode.put(MdcConstant.PROD_NAME, tProductName);
            tProdEocMappingObjectNode.put(MdcConstant.UID, tMappingsGroupByProductName.get(tProductName).get(0).getProductUid());

            //組成回傳json
            ArrayNode orgTypeMapping = mapper.createArrayNode();
            tProdEocMappingObjectNode.set(MdcConstant.ORG_TYPE_MAPPING, orgTypeMapping);

            for (EocIntgMapping mapping : tMappingsGroupByProductName.get(tProductName)) {
                ObjectNode orgTypeMappingNode = mapper.createObjectNode();
                orgTypeMappingNode.put(MdcConstant.EOC_MAPPING_ID, mapping.getEocId());

                //赋值eoc companyName
                if (pNameInclude && tEocNode.get(mapping.getEocId()) !=  null) {
                    orgTypeMappingNode.put(MdcConstant.EOC_MAPPING_NAME, tEocNode.get(mapping.getEocId()));
                }
                orgTypeMapping.add(orgTypeMappingNode);
            }
            tProdEocMappingAryNode.add(tProdEocMappingObjectNode);
        }
        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
        tResponseJsonNode.set("prod_eoc_mapping", tProdEocMappingAryNode);
        tResponseJsonNode.put(MdcConstant.TENANT_ID, pTenantId);
        return converJsonNodeToMap(tResponseJsonNode);
    }

    /**
     * 取得租戶特定產品的運營清單
     */
    public Map<String, Object> processTenantProdOperationListGet(String pTenantId, String pProdName, String pUid,
                                                                  String pId) throws Exception {
        // 組CacheKey 取tenant_id+'/'+api_name做HashKey
        String tCacheMapKey = CacheMapTypeEnum.tenant_product_operation_cachemap.getCode() + MdcSymbolConstant.COLON + pTenantId;
        String tKey = pProdName;
        if (pId != null) {
            tKey = tKey.concat(MdcSymbolConstant.COLON).concat("id=").concat(pId);
        }
        if (pUid != null) {
            tKey = tKey.concat(MdcSymbolConstant.COLON).concat("uid=").concat(pUid);
        }
        // 進元數據快取中找出對應資料
        String tDataFromCache = cacheService.getCacheMapValue(tCacheMapKey, tKey);
        // 有對應資料直接回傳
        if (null != tDataFromCache) {
            log.info("命中快取: " + tCacheMapKey + "// " + tKey);
            return converJsonStringToMap(tDataFromCache);
        }
        /*根據入參tenant_id, prod_name取得eoc_intg_mapping資料 */
        Map<String, String> pCondition = new HashMap<>();
        pCondition.put(MdcConstant.TENANT_ID, pTenantId);
        pCondition.put(MdcConstant.PROD_NAME, pProdName);
        if (pUid != null) {
            pCondition.put(MdcConstant.UID, pUid);
        }
        List<EocIntgMapping> tMappings = eocIntgMappingDao.fetch(pCondition);
        /* 將mappings依據eoc_level分出來 */
        List<EocIntgMapping> tEnterpriseIdMappings = new ArrayList<>();
        List<EocIntgMapping> tCompanyIdMappings = new ArrayList<>();
        List<EocIntgMapping> tSiteIdMappings = new ArrayList<>();
        List<EocIntgMapping> tRegionIdMappings = new ArrayList<>();
        for (EocIntgMapping mapping : tMappings) {
            if (mapping.getEocLevel().equals(EocLevelEnum.company_id.toString())) {
                tCompanyIdMappings.add(mapping);
            } else if (mapping.getEocLevel().equals(EocLevelEnum.site_id.toString())) {
                tSiteIdMappings.add(mapping);
            } else if (mapping.getEocLevel().equals(EocLevelEnum.region_id.toString())) {
                tRegionIdMappings.add(mapping);
            } else if (mapping.getEocLevel().equals(EocLevelEnum.enterprise_id.toString())) {
                tEnterpriseIdMappings.add(mapping);
            }
        }
        /*組成回傳json */
        // 組成org_type_enterprise物件
        ObjectNode tEnterpriseLevelMappingNode = mapper.createObjectNode();
        if (!tEnterpriseIdMappings.isEmpty()) {
            EocIntgMapping mapping = tEnterpriseIdMappings.get(0);
            tEnterpriseLevelMappingNode.put(MdcConstant.EOC_ENTERPRISE_ID, mapping.getEocId());
            tEnterpriseLevelMappingNode.put(MdcConstant.OM_ENTERPRISE_ID, mapping.getIntgId());
            if (mapping.getDatakey() != null && mapping.getDatakey().length() != 0) {
                tEnterpriseLevelMappingNode.put(MdcConstant.DATAKEY, mapping.getDatakey());
            }
        }
        // 組成org_type_company陣列
        ArrayNode tOrgTypeCompanyArrayNode = mapper.createArrayNode();
        for (EocIntgMapping mapping : tCompanyIdMappings) {
            ObjectNode tCompanyLevelMappingNode = mapper.createObjectNode();
            tCompanyLevelMappingNode.put(MdcConstant.EOC_COMPANY_ID, mapping.getEocId());
            tCompanyLevelMappingNode.put(MdcConstant.OM_COMPANY_ID, mapping.getIntgId());
            if (mapping.getDatakey() != null && mapping.getDatakey().length() != 0) {
                tCompanyLevelMappingNode.put(MdcConstant.DATAKEY, mapping.getDatakey());
            }
            tOrgTypeCompanyArrayNode.add(tCompanyLevelMappingNode);
        }
        // 組成eoc_level = site_id的物件，並放入parent物件
        for (EocIntgMapping mapping : tSiteIdMappings) {
            ObjectNode tSiteLevelMappingNode = mapper.createObjectNode();
            tSiteLevelMappingNode.put(MdcConstant.EOC_SITE_ID, mapping.getEocId());
            tSiteLevelMappingNode.put(MdcConstant.OM_SITE_ID, mapping.getIntgId());
            if (mapping.getDatakey() != null && mapping.getDatakey().length() != 0) {
                tSiteLevelMappingNode.put(MdcConstant.DATAKEY, mapping.getDatakey());
            }
            // 有parent mapping
            if (mapping.getEocIntgMapping() != null) {
                // 找到parent節點
                for (JsonNode tCompanyNode : tOrgTypeCompanyArrayNode) {
                    if (tCompanyNode.get(MdcConstant.EOC_COMPANY_ID).asText().equals(mapping.getEocIntgMapping().getEocId())) {
                        // parent節點已經有"org_type_site"陣列
                        if (tCompanyNode.get(MdcConstant.ORG_TYPE_SITE) != null) {
                            ((ArrayNode) tCompanyNode.get(MdcConstant.ORG_TYPE_SITE)).add(tSiteLevelMappingNode);
                            // parent節點沒有"org_type_site"陣列
                        } else {
                            ArrayNode tOrgTypeSiteArrayNode = mapper.createArrayNode();
                            ((ObjectNode) tCompanyNode).set(MdcConstant.ORG_TYPE_SITE, tOrgTypeSiteArrayNode);
                            tOrgTypeSiteArrayNode.add(tSiteLevelMappingNode);
                        }
                    }
                }
            }
        }
        // 組成org_type_region陣列
        ArrayNode tOrgTypeRegionArrayNode = mapper.createArrayNode();
        for (EocIntgMapping mapping : tRegionIdMappings) {
            ObjectNode tRegionLevelMappingNode = mapper.createObjectNode();
            tRegionLevelMappingNode.put(MdcConstant.EOC_REGION_ID, mapping.getEocId());
            tRegionLevelMappingNode.put(MdcConstant.OM_REGION_ID, mapping.getIntgId());
            if (mapping.getRegionType() == null) {
                tRegionLevelMappingNode.put(MdcConstant.REGION_TYPE, "");
            } else {
                tRegionLevelMappingNode.put(MdcConstant.REGION_TYPE, mapping.getRegionType());
            }
            if (mapping.getDatakey() != null && mapping.getDatakey().length() != 0) {
                tRegionLevelMappingNode.put(MdcConstant.DATAKEY, mapping.getDatakey());
            }
            tOrgTypeRegionArrayNode.add(tRegionLevelMappingNode);
        }
        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
        tResponseJsonNode.set(MdcConstant.ORG_TYPE_ENTERPRISE, tEnterpriseLevelMappingNode);
        tResponseJsonNode.set(MdcConstant.ORG_TYPE_COMPANY, tOrgTypeCompanyArrayNode);
        tResponseJsonNode.set(MdcConstant.ORG_TYPE_REGION, tOrgTypeRegionArrayNode);
        tResponseJsonNode.put(MdcConstant.TENANT_ID, pTenantId);
        tResponseJsonNode.put(MdcConstant.PROD_NAME, pProdName);
        // 穩態產品取eoc_intg_mapping表中product_uid
        if (!tMappings.isEmpty()) {
            tResponseJsonNode.put(MdcConstant.UID, tMappings.get(0).getProductUid());
            tResponseJsonNode.put(MdcConstant.PROD_TYPE, ApiTypeEnum.standard.toString());
            tResponseJsonNode.put(MdcConstant.URL, "");
        } else {
            // 敏態產品取product表中的產品資訊(新規則)
            Map<String, String> tAgileAppCondition = new HashMap<>();
            tAgileAppCondition.put(MdcConstant.PRODUCT_TYPE, ApiTypeEnum.agile.toString());
            tAgileAppCondition.put(MdcConstant.NAME, pProdName);
            if (pId != null) {
                tAgileAppCondition.put(MdcConstant.ID, pId);
            } else if (pUid != null) {
                tAgileAppCondition.put(MdcConstant.UID, pUid);
            }
            List<Product> tAgileAppList = productDao.fetch(tAgileAppCondition);
            // 有找到對應的敏態產品
            if (tAgileAppList != null && !tAgileAppList.isEmpty()) {
                // // 產品名稱是"Athena"
                // if (("Athena").equals(pProdName)) {
                // tResponseJsonNode.put(UID, "MonitorEngine");
                // // 其他敏態產品
                // } else {
                tResponseJsonNode.put(MdcConstant.UID, tAgileAppList.get(0).getUid());
                tResponseJsonNode.put(MdcConstant.PROD_TYPE, ApiTypeEnum.agile.toString());
                tResponseJsonNode.put(MdcConstant.URL, tAgileAppList.get(0).getUrl());
                // }
            }
        }
        // 將回傳訊息加入快取中
        if (!tMappings.isEmpty()) {
            cacheService.setCacheMapValue(tCacheMapKey, tKey, tResponseJsonNode.toString());
        }

        return converJsonNodeToMap(tResponseJsonNode);
    }

    /**
     * 取得租戶特定產品的運營清單 v2 数据去重返回云端id
     */
    public Map<String, Object> processTenantProdOperationListGetV2(String pTenantId, String pProdName, String pUid,
                                                                 String pId) throws Exception {

        //根據入參tenant_id, prod_name、云地映射标识取得eoc_intg_mapping資料
        Map<String, String> pCondition = new HashMap<>();
        pCondition.put(MdcConstant.TENANT_ID, pTenantId);
        pCondition.put(MdcConstant.PROD_NAME, pProdName);
        pCondition.put(MdcConstant.EOC_LEVEL, MdcConstant.CLOUD_MAPPING_ID);
        if (pUid != null) {
            pCondition.put(MdcConstant.UID, pUid);
        }
        List<EocIntgMapping> tMappings = eocIntgMappingDao.fetchGroupByEocId(pCondition);

        //組成回傳json
        ArrayNode orgTypeMapping = mapper.createArrayNode();

        for (EocIntgMapping mapping : tMappings) {

            ObjectNode orgTypeMappingNode = mapper.createObjectNode();
            orgTypeMappingNode.put(MdcConstant.EOC_MAPPING_ID, mapping.getEocId());
            orgTypeMapping.add(orgTypeMappingNode);
        }

        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
        tResponseJsonNode.set(MdcConstant.ORG_TYPE_MAPPING, orgTypeMapping);
        tResponseJsonNode.put(MdcConstant.TENANT_ID, pTenantId);
        tResponseJsonNode.put(MdcConstant.PROD_NAME, pProdName);
        // 穩態產品取eoc_intg_mapping表中product_uid
        if (!tMappings.isEmpty()) {
            tResponseJsonNode.put(MdcConstant.UID, tMappings.get(0).getProductUid());
            tResponseJsonNode.put(MdcConstant.PROD_TYPE, ApiTypeEnum.standard.toString());
            tResponseJsonNode.put(MdcConstant.URL, "");
        } else {
            // 敏態產品取product表中的產品資訊(新規則)
            Map<String, String> tAgileAppCondition = new HashMap<>();
            tAgileAppCondition.put(MdcConstant.PRODUCT_TYPE, ApiTypeEnum.agile.toString());
            tAgileAppCondition.put(MdcConstant.NAME, pProdName);
            if (pId != null) {
                tAgileAppCondition.put(MdcConstant.ID, pId);
            } else if (pUid != null) {
                tAgileAppCondition.put(MdcConstant.UID, pUid);
            }
            List<Product> tAgileAppList = productDao.fetch(tAgileAppCondition);
            // 有找到對應的敏態產品
            if (tAgileAppList != null && !tAgileAppList.isEmpty()) {
                tResponseJsonNode.put(MdcConstant.UID, tAgileAppList.get(0).getUid());
                tResponseJsonNode.put(MdcConstant.PROD_TYPE, ApiTypeEnum.agile.toString());
                tResponseJsonNode.put(MdcConstant.URL, tAgileAppList.get(0).getUrl());
            }
        }

        return converJsonNodeToMap(tResponseJsonNode);
    }

    /**
     * 取得租戶特定產品的運營清單 v2版本,数据不去重返回云端id和地端id
     */
    public Map<String, Object> tenantProdOperationListGetV2(String pTenantId, String pProdName, String pUid,
                                                                   String pId) {

        //根據入參tenant_id, prod_name、云地映射标识取得eoc_intg_mapping資料
        Map<String, String> pCondition = new HashMap<>();
        pCondition.put(MdcConstant.TENANT_ID, pTenantId);
        pCondition.put(MdcConstant.PROD_NAME, pProdName);
        pCondition.put(MdcConstant.EOC_LEVEL, MdcConstant.CLOUD_MAPPING_ID);
        if (pUid != null) {
            pCondition.put(MdcConstant.UID, pUid);
        }
        List<EocIntgMapping> tMappings = eocIntgMappingDao.fetch(pCondition);

        //組成回傳json
        ArrayNode orgTypeMapping = mapper.createArrayNode();

        for (EocIntgMapping mapping : tMappings) {

            ObjectNode orgTypeMappingNode = mapper.createObjectNode();
            orgTypeMappingNode.put(MdcConstant.EOC_MAPPING_ID, mapping.getEocId());
            orgTypeMappingNode.put(MdcConstant.OM_MAPPING_ID, mapping.getIntgId());

            //type
            if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.enterprise_id.toString())) {
                orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.ENTERPRISE_ID);
            }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.company_id.toString())){
                orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.COMPANY_ID);
            }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.site_id.toString())){
                orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.SITE_ID);
            }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.region_id.toString())){
                orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.REGION_ID);
            }

            if (mapping.getDatakey() != null && !mapping.getDatakey().isEmpty()) {
                orgTypeMappingNode.put(MdcConstant.DATAKEY, mapping.getDatakey());
            }

            orgTypeMapping.add(orgTypeMappingNode);
        }

        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);

        tResponseJsonNode.set(MdcConstant.ORG_TYPE_MAPPING, orgTypeMapping);
        tResponseJsonNode.put(MdcConstant.TENANT_ID, pTenantId);
        tResponseJsonNode.put(MdcConstant.PROD_NAME, pProdName);
        // 穩態產品取eoc_intg_mapping表中product_uid
        if (!tMappings.isEmpty()) {
            tResponseJsonNode.put(MdcConstant.UID, tMappings.get(0).getProductUid());
            tResponseJsonNode.put(MdcConstant.PROD_TYPE, ApiTypeEnum.standard.toString());
            tResponseJsonNode.put(MdcConstant.URL, "");
        } else {
            // 敏態產品取product表中的產品資訊(新規則)
            Map<String, String> tAgileAppCondition = new HashMap<>();
            tAgileAppCondition.put(MdcConstant.PRODUCT_TYPE, ApiTypeEnum.agile.toString());
            tAgileAppCondition.put(MdcConstant.NAME, pProdName);
            if (pId != null) {
                tAgileAppCondition.put(MdcConstant.ID, pId);
            } else if (pUid != null) {
                tAgileAppCondition.put(MdcConstant.UID, pUid);
            }
            List<Product> tAgileAppList = productDao.fetch(tAgileAppCondition);
            // 有找到對應的敏態產品
            if (tAgileAppList != null && !tAgileAppList.isEmpty()) {

                tResponseJsonNode.put(MdcConstant.UID, tAgileAppList.get(0).getUid());
                tResponseJsonNode.put(MdcConstant.PROD_TYPE, ApiTypeEnum.agile.toString());
                tResponseJsonNode.put(MdcConstant.URL, tAgileAppList.get(0).getUrl());

            }
        }
        return converJsonNodeToMap(tResponseJsonNode);
    }
}