package com.digiwin.metadatacache.apiservice;

import com.digiwin.metadatacache.MdcApplicationParameter;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.dao.EocIntgMappingDao;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.dao.ServiceDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.services.MdcCacheService;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

@Service
@CommonsLog
public class TenantProductUpdateProcessService {

    @Autowired
    private ServiceDao serviceDao;

    @Autowired
    private ProductDao productDao;

    @Autowired
    private MdcCacheService cacheService;

    @Autowired
    private EocIntgMappingDao eocIntgMappingDao;

    @Transactional(rollbackFor = Exception.class,value = "mdcTransactionManager")
    @Async
    public void updateProductService(JsonNode tProductNode, List<Product> pListToUpdate) {
        for (Product tProduct : pListToUpdate) {
            if (tProductNode.get(MdcConstant.NAME).asText().equals(tProduct.getName())
                    && tProductNode.get(MdcConstant.UID).asText().equals(tProduct.getUid())) {
                ArrayNode tSrvInfoAryNode = (ArrayNode) tProductNode.get(MdcConstant.SERVICES);
                log.info("产品Id: " + tProduct.getId());
                // 获取产品所有服务信息
                List<com.digiwin.metadatacache.model.Service> tCurrentServiceList = serviceDao.getByPrdId(tProduct.getId());
                List<String> tCurrentServiceNameList = new ArrayList<>();
                for (com.digiwin.metadatacache.model.Service service : tCurrentServiceList) {
                    tCurrentServiceNameList.add(service.getName());
                }
                // 刪除此產品的service
                serviceDao.removeByPrdId(tProduct.getId());
                if (tSrvInfoAryNode.isArray()) {
                    // 要新增的產品
                    if (tProduct.getId() == null) {
                        Map<String, String> tCondition = new HashMap<>();
                        tCondition.put(MdcConstant.NAME, tProduct.getName());
                        tCondition.put(MdcConstant.UID, tProduct.getUid());
                        tCondition.put(MdcConstant.EAI_UID, tProduct.getEaiUid());
                        List<Product> tProductList = productDao.fetch(tCondition);
                        if (tProductList != null && tProductList.size() != 0) {
                            tProduct.setId(tProductList.get(0).getId());
                        }
                    }
                    List<com.digiwin.metadatacache.model.Service> tUpdateServiceList = new ArrayList<>();
                    List<String> tUpdateServiceNameList = new ArrayList<>();
                    for (JsonNode tSrvNode : tSrvInfoAryNode) {
                        // 新增service
                        com.digiwin.metadatacache.model.Service tService = new com.digiwin.metadatacache.model.Service();
                        tService.setBuildTime(Calendar.getInstance());
                        tService.setProduct(tProduct);
                        if (tSrvNode.get(MdcConstant.VERSION) != null && tSrvNode.get(MdcConstant.VERSION).asText().length() != 0) {
                            tService.setVersion(tSrvNode.get(MdcConstant.VERSION).asText());
                        }
                        tService.setName(tSrvNode.get(MdcConstant.NAME).asText());
                        tUpdateServiceList.add(tService);
                        tUpdateServiceNameList.add(tSrvNode.get(MdcConstant.NAME).asText());
                    }
                    serviceDao.saveServices(tUpdateServiceList);
                    // 取得資料庫與同步清單中的相異的服務
                    List<String> tServiceNameDifferenceList
                            = (List<String>) CollectionUtils.disjunction(tCurrentServiceNameList, tUpdateServiceNameList);
                    // 穩態
                    if (tProduct.getProductType().equals(ApiTypeEnum.standard.toString())) {
                        for (String serviceName : tServiceNameDifferenceList) {
                            if (tCurrentServiceNameList.contains(serviceName)) {
                                removeServiceFromProductInCache(tProduct, serviceName, ApiTypeEnum.standard);
                            } else {
                                if (StringUtils.isNotBlank(tProduct.getTenantId())) {
                                    addServiceFromProductInCache(tProduct, serviceName, ApiTypeEnum.standard);
                                }
                            }
                        }
                    } else {
                        // 敏态
                        for (String serviceName : tServiceNameDifferenceList) {
                            if (tCurrentServiceNameList.contains(serviceName)) {
                                removeServiceFromProductInCache(tProduct, serviceName,
                                        ApiTypeEnum.agile);
                            } else {
                                addServiceFromProductInCache(tProduct, serviceName, ApiTypeEnum.agile);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * 删除服务与产品
     * @param pProductList pProductList
     */
    public void removeProductList(List<Product> pProductList) {
        // 清除要刪除的產品的service，再清除要刪除的產品
        for (Product tProduct : pProductList) {
            List<com.digiwin.metadatacache.model.Service> tServiceRemoveList = serviceDao.getByPrdId(tProduct.getId());
            serviceDao.removeByPrdId(tProduct.getId());
            productDao.remove(tProduct.getId());
            // 刪除產品同時要刪除該產品主機配置的運營
            eocIntgMappingDao.clearListWithLevel(tProduct.getTenantId(), tProduct.getUid(), "All");
            // 清除此產品實作服務的租戶產品清單快取
            if (CollectionUtils.isNotEmpty(tServiceRemoveList)) {
                // 稳态
                if (tProduct.getProductType().equals(ApiTypeEnum.standard.toString())) {
                    for (com.digiwin.metadatacache.model.Service tService : tServiceRemoveList) {
                        removeServiceFromProductInCache(tProduct, tService.getName(), ApiTypeEnum.standard);
                    }
                } else {
                    // 敏态
                    for (com.digiwin.metadatacache.model.Service tService : tServiceRemoveList) {
                        removeServiceFromProductInCache(tProduct, tService.getName(), ApiTypeEnum.agile);
                    }
                }
            }
        }
    }

    /**
     * 删除对应快取信息
     * @param pProduct 产品信息
     * @param pServiceName 服务名称
     * @param pSearchProdTypeEnum 类型
     */
    private void removeServiceFromProductInCache(Product pProduct,
                                                 String pServiceName, ApiTypeEnum pSearchProdTypeEnum) {
        String tKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode()
                + MdcSymbolConstant.COLON + MdcConstant.AGILE + MdcSymbolConstant.COLON + pServiceName;
        if (pSearchProdTypeEnum.equals(ApiTypeEnum.standard)) {
            tKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + pProduct.getTenantId();
            Map<String, List<String>> tProductServiceMapFromCache = (Map<String, List<String>>) cacheService.get(tKey);
            String tProductKey = pProduct.getName()
                    + MdcSymbolConstant.COLON + pProduct.getUid() + MdcSymbolConstant.COLON + pProduct.getEaiUid();
            if (tProductServiceMapFromCache != null && tProductServiceMapFromCache.get(tProductKey) != null) {
                List<String> tServiceToRemove = new ArrayList<>();
                for (String tServiceString : tProductServiceMapFromCache.get(tProductKey)
                        .subList(1, tProductServiceMapFromCache.get(tProductKey).size())) {
                    if (tServiceString.contains(pServiceName)) {
                        tServiceToRemove.add(tServiceString);
                        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "清除快取(產品中的服務)：" + "key = " + tKey);
                    }
                }
                if (!tServiceToRemove.isEmpty()) {
                    tProductServiceMapFromCache.get(tProductKey).removeAll(tServiceToRemove);
                    cacheService.setSync(tKey, tProductServiceMapFromCache);
                }
            }
        } else {
            Map<String, String> tProductServiceMapFromCache = (Map<String, String>) cacheService.get(tKey);
            if (null != tProductServiceMapFromCache) {
                String tProductKey = pProduct.getName() + MdcSymbolConstant.COLON + pProduct.getUid();
                tProductServiceMapFromCache.remove(tProductKey);
                cacheService.setSyncTimeOut(tKey, tProductServiceMapFromCache, MdcApplicationParameter._TIME_OUT + (int) (Math.random() * MdcApplicationParameter._TIME_RANDOM));
            }
        }
    }

    private void addServiceFromProductInCache(Product pProduct, String pServiceString,
                                              ApiTypeEnum pSearchProdTypeEnum) {
        String tKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode()
                + MdcSymbolConstant.COLON + MdcConstant.AGILE + MdcSymbolConstant.COLON + pServiceString;
        if (pSearchProdTypeEnum.equals(ApiTypeEnum.standard)) {
            tKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + pProduct.getTenantId();
            Map<String, List<String>> tProductServiceMapFromCache = (Map<String, List<String>>) cacheService.get(tKey);
            String tProductKey = pProduct.getName() + MdcSymbolConstant.COLON + pProduct.getUid() + MdcSymbolConstant.COLON + pProduct.getEaiUid();
            if (tProductServiceMapFromCache != null && tProductServiceMapFromCache.get(tProductKey) != null) {
                log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "更新快取(產品中的服務)：" + "key = " + tKey);
                tProductServiceMapFromCache.get(tProductKey).add(pServiceString);
                cacheService.setSync(tKey, tProductServiceMapFromCache);
            }
        }else {
            Map<String, String> tProductServiceMapFromCache = (Map<String, String>) cacheService.get(tKey);
            if (null != tProductServiceMapFromCache) {
                String tProductKey = pProduct.getName() + MdcSymbolConstant.COLON + pProduct.getUid();
                String tProductFullInfo = pProduct.getName()
                        + MdcSymbolConstant.LEFT_DASH + pProduct.getUid()
                        + MdcSymbolConstant.LEFT_DASH + pProduct.getEaiUid()
                        + MdcSymbolConstant.LEFT_DASH + pProduct.getVersion()
                        + MdcSymbolConstant.LEFT_DASH + pProduct.getIp()
                        + MdcSymbolConstant.LEFT_DASH + pProduct.getApid()
                        + MdcSymbolConstant.LEFT_DASH + pProduct.getGatewayId();
                tProductServiceMapFromCache.put(tProductKey, tProductFullInfo);
                cacheService.setSyncTimeOut(tKey, tProductServiceMapFromCache, MdcApplicationParameter._TIME_OUT + (int) (Math.random() * MdcApplicationParameter._TIME_RANDOM));
            }
        }
    }
}
