package com.digiwin.metadatacache.services;

import com.digiwin.app.service.DWServiceContext;
import com.digiwin.http.client.DWHttpClient;
import com.digiwin.http.client.DWRequestOption;
import com.digiwin.http.client.entity.DWJsonEntity;
import com.digiwin.http.client.exception.DWHttpFailedException;
import com.digiwin.http.client.utils.DWURIBuilder;
import com.digiwin.metadatacache.MdcApplicationParameter;
import com.digiwin.metadatacache.constant.InterfaceConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.exception.TenantTypeRetrievedException;
import com.digiwin.metadatacache.exception.TokenNotRetrievedException;
import com.digiwin.metadatacache.exception.UserInfoRetrievedException;
import com.digiwin.metadatacache.exception.UserRoleRetrievedException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.net.URI;
import java.util.HashMap;
import java.util.Map;

@Service
public class IamVerifyService {

	private final Log log = LogFactory.getLog(getClass());

	@Autowired
	private DWHttpClient dwHttpClient;

	@Value("${iamApToken}")
	private String iamApToken;

	/**
	 * true=支持v2租户，false=不支持v2租户
	 */
	@Value("${supportTenantV2:false}")
	private String supportTenantV2;

	ObjectMapper mapper = new ObjectMapper();

	/**
	 * 取得user name
	 */
	public String getUserName(String pUserToken) throws Exception {
		Map<String, String> tResponse = getUserInfo(pUserToken);
		String tUserName = null;
		if (tResponse.get(MdcConstant.NAME) != null) {
			tUserName = tResponse.get(MdcConstant.NAME);
		}
		return tUserName;
	}

	/**
	 * 取得user info
	 */
	public Map<String, String> getUserInfo(String pUserToken) throws Exception {
		Map<String, String> tResult = new HashMap<>();
		try {
			JsonNode tResponseNode = analyzeToken(pUserToken);
			if (tResponseNode.get(MdcConstant.TENANTID) != null) {
				tResult.put(MdcConstant.TENANTID, tResponseNode.get(MdcConstant.TENANTID).asText());
			}
			if (tResponseNode.get(MdcConstant.ID) != null) {
				tResult.put(MdcConstant.USERID, tResponseNode.get(MdcConstant.ID).asText());
			}
			if (tResponseNode.get(MdcConstant.NAME) != null) {
				tResult.put(MdcConstant.NAME, tResponseNode.get(MdcConstant.NAME).asText());
			}
		}catch (Exception e) {
			throw new UserInfoRetrievedException(getClass());
		}
		return tResult;
	}

	private JsonNode analyzeToken(String pToken) throws Exception {
		URI tUri = DWURIBuilder.create(MdcApplicationParameter._IAM_HOST + InterfaceConstant.LOGIN_PATH).build();
		HttpPost tPost = new HttpPost(tUri);
		tPost.setHeader(MdcConstant.CONTENT_TYPE, "application/json;charset=UTF-8");
		tPost.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_USER, pToken);
		tPost.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_APP, iamApToken);
		tPost.setHeader(MdcConstant.TOKEN, pToken);
		if (null != DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY)) {
			tPost.setHeader(MdcConstant.ROUTER_KEY, DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY).toString());
		}
		String tResponse = dwHttpClient.execute(tPost, String.class, new DWRequestOption(false));
		return mapper.readTree(tResponse);
	}

	/** 檢查user_id對應的角色是否為superadmin **/
	public boolean isSuperAdmin(String userId, String token) throws Exception {
		try {
			URI tIamTargetUri = DWURIBuilder.create(MdcApplicationParameter._IAM_HOST  + InterfaceConstant.USER_ROLE).build();
			HttpPost tPost = new HttpPost(tIamTargetUri);
			ObjectNode objectNode = mapper.createObjectNode();
			objectNode.put(MdcConstant.ID, userId);
			StringEntity entity = new StringEntity(objectNode.toString(), "UTF-8");
			entity.setContentEncoding("UTF-8");
			tPost.setEntity(entity);
			tPost.setHeader(MdcConstant.TOKEN, token);
			tPost.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_USER, token);
			tPost.setHeader(MdcConstant.CONTENT_TYPE, MdcConstant.APPLICATION_JSON);
			tPost.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_APP, iamApToken);
			if (null != DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY)) {
				tPost.setHeader(MdcConstant.ROUTER_KEY, DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY).toString());
			}
			String tResponse = dwHttpClient.execute(tPost, String.class, new DWRequestOption(false));
			JsonNode tResultNode = mapper.readTree(tResponse);
			ArrayNode tRolesNode = (ArrayNode) tResultNode.get(MdcConstant.ROLES);
			for (JsonNode node : tRolesNode) {
				if (node.get(MdcConstant.ID).asText().equals(MdcConstant.SUPERADMIN)) {
					return true;
				}
			}
			return false;
		} catch (Exception e) {
			throw new UserRoleRetrievedException(e, getClass());
		}
	}

	/** 檢查token對應的租戶是否為個人租戶 **/
	public boolean isPersonalTenant(String token) throws Exception {
		try {
			URI tIamTargetUri = DWURIBuilder.create(MdcApplicationParameter._IAM_HOST  + InterfaceConstant.GET_TENANT_CURRENT).build();
			HttpPost tPost = new HttpPost(tIamTargetUri);
			tPost.setHeader(MdcConstant.TOKEN, token);
			tPost.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_USER, token);
			tPost.setHeader(MdcConstant.CONTENT_TYPE, MdcConstant.APPLICATION_JSON);
			tPost.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_APP, iamApToken);
			if (null != DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY)) {
				tPost.setHeader(MdcConstant.ROUTER_KEY, DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY).toString());
			}
			String tResponse = dwHttpClient.execute(tPost, String.class, new DWRequestOption(false));
			JsonNode tResultNode = mapper.readTree(tResponse);
			int tEnterpriseType = tResultNode.get(MdcConstant.TENANT).get(MdcConstant.ENTERPRISETYPE).asInt();
			return tEnterpriseType == 0;
		} catch (Exception e) {
			throw new TenantTypeRetrievedException(e, getClass());
		}
	}

	/**
	 * 查询租户版本
	 */
	public String getCurrentTenantVersion(String pTenantId) throws Exception {
        //增加是否开启v2租户开关，为false表示不支持v2租户
        if(StringUtils.isNotEmpty(supportTenantV2) && "false".equals(supportTenantV2)){
            return MdcConstant.TENANT_VERSION_V1;
        }
        if (DWServiceContext.getContext().getProfile() == null || MapUtils.isEmpty(DWServiceContext.getContext().getProfile())) {
			// 匿名方法没有profile，或者返回的是不可变map，重新初始化一下，
			DWServiceContext.getContext().setProfile(new HashMap<>());
		}
		//请求头找不到租户版本则主动查询租户版本
		if (!DWServiceContext.getContext().getProfile().containsKey(MdcConstant.VERSION)) {
			Map<String, Object> headers = DWServiceContext.getContext().getRequestHeader();
			//获取token
			String tToken =MapUtils.getString(headers, MdcConstant.TOKEN);
			if(StringUtils.isEmpty(tToken)){
				// 根據license key呼叫IAM取得userToken
				tToken = getTokenByTenant(pTenantId);
			}
			DWServiceContext.getContext().getProfile().put(MdcConstant.VERSION, getTenantOperationUnitVersion(tToken));
		}

		return MapUtils.getString(DWServiceContext.getContext().getProfile(), MdcConstant.VERSION);
	}
	/**
	 * 查询租户版本
	 */
	public String getCurrentTenantVersionByToken(String token) throws Exception {
		// 查询租户版本
		JsonNode simpleByToken = getCurrentTenantSimpleByToken(token);
		JsonNode jsonNode = simpleByToken.get(MdcConstant.DATA);
		//等于空就是v1
		if(jsonNode.get(MdcConstant.VERSION)==null){
			return MdcConstant.TENANT_VERSION_V1;
		}else {
			return jsonNode.get(MdcConstant.VERSION).asText();
		}
	}

	public String getTenantOperationUnitVersion(String token) throws Exception {
		//token 为空默认返回v1
		if(StringUtils.isEmpty(token)){
			return MdcConstant.TENANT_VERSION_V1;
		}

		// 查询租户版本
		JsonNode simpleByToken = getCurrentTenantSimpleByToken(token);
		JsonNode jsonNode = simpleByToken.get(MdcConstant.DATA);
		//等于空就是v1
		if(jsonNode.get(MdcConstant.VERSION)==null){
			return MdcConstant.TENANT_VERSION_V1;
		}else {
			return jsonNode.get(MdcConstant.VERSION).asText();
		}
	}

	/**
	 * 查询租户版本
	 */
	public JsonNode getCurrentTenantSimpleByToken(String pToken) throws Exception {
		try {
		URI tUri = DWURIBuilder.create(MdcApplicationParameter._IAM_HOST + InterfaceConstant.GET_CURRENT_SIMPLE).build();
		HttpGet tGet = new HttpGet(tUri);
		tGet.setHeader(MdcConstant.CONTENT_TYPE, "application/json;charset=UTF-8");
		tGet.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_USER, pToken);
		tGet.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_APP, iamApToken);
/*		tGet.setHeader(MdcConstant.TOKEN, pToken);
		if (null != DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY)) {
			tGet.setHeader(MdcConstant.ROUTER_KEY, DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY).toString());
		}*/
		String tResponse = dwHttpClient.execute(tGet, String.class, new DWRequestOption(false));
		return mapper.readTree(tResponse);
		} catch (Exception e) {
			log.error("Route=>IamVerifyService method=getCurrentTenantSimpleByToken error:",e);
			throw new TenantTypeRetrievedException(e, getClass());
		}
	}

	/**
	 * 根據license key呼叫IAM取得userToken
	 */
	@SuppressWarnings("unchecked")
	public String getTokenByTenant(String tenantId) throws Exception {
		try {
			URI tIamTargetUri = DWURIBuilder.create("iam", InterfaceConstant.IAM_LOGIN_INTERNAL).build();
			HttpPost tPost = new HttpPost(tIamTargetUri);
			Map<String, Object> tRequestEntity = new HashMap<>();
			tRequestEntity.put("identityType", "internal");
			tRequestEntity.put("userId", "integration");
			tRequestEntity.put("passwordHash", "6826CC688C4AF1BD0A8DDA2DBDF8897B");
			tRequestEntity.put("tenantId", tenantId);
			tPost.setEntity(new DWJsonEntity(tRequestEntity));
			tPost.setHeader(MdcConstant.DIGI_MIDDLEWARE_AUTH_APP, iamApToken);
			if (null != DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY)) {
				tPost.setHeader(MdcConstant.ROUTER_KEY, DWServiceContext.getContext().getRequestHeader().get(MdcConstant.ROUTER_KEY).toString());
			}
			Map<String, Object> tResponseEntity = dwHttpClient.execute(tPost, Map.class, new DWRequestOption(false));
			return (String) tResponseEntity.get(MdcConstant.TOKEN);
		} catch (DWHttpFailedException e) {
			Map<String, Object> tFailedEntity = e.getEntity(Map.class);
			String tMsg = String.format("IAM response: %s", tFailedEntity.get("message"));
			throw new TokenNotRetrievedException(tMsg, getClass());
		} catch (Exception e) {
			throw new TokenNotRetrievedException(e, getClass());
		}
	}

}
