package com.digiwin.metadatacache.services;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * 本地快取
 */
@Service
public class LoadCacheService {

    private final Log log = LogFactory.getLog(getClass());

    private static final RemovalListener<String, Map<String, ObjectNode>> listener = notification
            -> System.err.println("remove key:" + notification.getKey());

    /**
     * 快取
     * 格式：<api_name,map<租户Id,对应的值>>
     */
    private static final LoadingCache<String, Map<String, ObjectNode>> loadingCache = CacheBuilder
            .newBuilder()
            .maximumSize(300) // 缓存的个数
            .removalListener(listener) // 移除缓存时的监听事件
            .expireAfterWrite(10, TimeUnit.MINUTES)
            .build(new CacheLoader<String, Map<String, ObjectNode>>() {
                @Override
                public Map<String, ObjectNode> load(String key) {
                    return new HashMap<>();
                }
            });

    /**
     * 保存数据
     *
     * @param key   key
     * @param value value
     */
    public void saveData(String key, String tenantId, ObjectNode value) {
        Map<String, ObjectNode> map;
        try {
            map = loadingCache.get(key);
        } catch (ExecutionException e) {
            log.error(e.getMessage(), e);
            return;
        }
        if (map.size() > 30) {
            map.clear();
        }
        map.put(tenantId, value);
        loadingCache.put(key, map);
    }

    /**
     * 查询数据
     *
     * @param key key
     * @return value
     */
    public ObjectNode getData(String key, String tenantId) {
        try {
            Map<String, ObjectNode> map = loadingCache.getIfPresent(key);
            if (null != map) {
                return map.get(tenantId);
            }
        } catch (Exception e) {
            log.error("本地快取插叙异常: key " + key + "tenantId " + tenantId);
        }
        return null;
    }

    /**
     * 删除数据
     *
     * @param key key
     */
    public void removeData(String key) {
        loadingCache.invalidate(key);
    }

    /**
     * 删除数据
     */
    public void removeAllData() {
        loadingCache.invalidateAll();
    }

    /**
     * 删除数据
     */
    public Map<String, Object> showData() {
        Map<String, Object> map = new HashMap<>();
        //利用迭代器
        for (Map.Entry<String, Map<String, ObjectNode>> stringMapEntry : loadingCache.asMap().entrySet()) {
            map.put(stringMapEntry.getKey(), stringMapEntry.getValue());
        }
        return map;
    }
}
