package com.digiwin.metadatacache.services;

import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RTopic;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * @description:
 * @author: liunansheng
 * @date: 2024/5/23 17:31
 */
@CommonsLog
@Service
public class LocalCacheService {

    private static final Object lock = new Object();

    private static final String CLEAR_ALL_MSG= "[clear_all]";

    private static final String CURRENT_CLIENT_ID = UUID.randomUUID().toString().replaceAll("-", "");
    private static final String CHANGE_MSG_SLIPTOR = "|||";

    private Map<String, Cache> CACHE_CONTEXT = new HashMap<>();
    private Map<String, RTopic> TOPIC_CONTEXT = new HashMap<>();

    private final RedissonService redissonService;

    public LocalCacheService(RedissonService redissonService) {
        this.redissonService = redissonService;
    }

    public <T extends Serializable> void putToL2Cache(CacheMapTypeEnum pCacheKeyEnum, String pKey, T val) {
        try {
            Cache<String, T> cache = getCache(pCacheKeyEnum);
            cache.put(pKey, val);
            publishChange(pCacheKeyEnum, pKey);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    public <T extends Serializable> void putToL2Cache(CacheMapTypeEnum pCacheKeyEnum, String firstKey, String secondKey, T val) {
        try {
            Cache<String, Map<String, T>> cache = getCache(pCacheKeyEnum);
            Map<String, T> cacheMap = cache.get(firstKey, one -> new ConcurrentHashMap<>());
            cacheMap.put(secondKey, val);
            cache.put(firstKey, cacheMap);
            publishChange(pCacheKeyEnum, firstKey, secondKey);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    public <T extends Serializable> T getFromL2Cache(CacheMapTypeEnum pCacheKeyEnum, String pKey) {
        try {
            Cache<String, T> cache = getCache(pCacheKeyEnum);
            return cache.get(pKey, key -> null);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    public <T extends Serializable> T getFromL2Cache(CacheMapTypeEnum pCacheKeyEnum, String firstKey, String secondKey) {
        try {
            Cache<String, Map<String, T>> cache = getCache(pCacheKeyEnum);
            Map<String, T> cacheMap = cache.get(firstKey, one -> null);
            return null == cacheMap ? null : cacheMap.get(secondKey);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    public void remove(CacheMapTypeEnum pCacheKeyEnum, String pKey) {
        getCache(pCacheKeyEnum).invalidate(pKey);
        publishChange(pCacheKeyEnum, pKey);
    }

    private void publishChange(CacheMapTypeEnum pCacheKeyEnum, String key) {
        try {
            TOPIC_CONTEXT.get(pCacheKeyEnum.getCode()).publish(CURRENT_CLIENT_ID + CHANGE_MSG_SLIPTOR + key);
        } catch (Exception e) {
            log.error("publish modify err", e);
        }
    }

    private void publishChange(CacheMapTypeEnum pCacheKeyEnum, String firstKey, String secondKey) {
        try {
            TOPIC_CONTEXT.get(pCacheKeyEnum.getCode()).publish(CURRENT_CLIENT_ID + CHANGE_MSG_SLIPTOR + firstKey + CHANGE_MSG_SLIPTOR + secondKey);
        } catch (Exception e) {
            log.error("publish modify err", e);
        }
    }

    private <T> Cache<String, T> getCache(CacheMapTypeEnum cacheKeyEnum) {
        String namespace = cacheKeyEnum.getCode();
        Cache<String, T> cache = CACHE_CONTEXT.get(namespace);
        if (null == cache) {
            synchronized (lock) {
                cache = CACHE_CONTEXT.get(namespace);
                if (cache == null) {
                    cache = Caffeine.newBuilder()
                            .maximumSize(2000)              //最大数量
                            .expireAfterWrite(12, TimeUnit.HOURS) //过期时间
                            .build();
                    CACHE_CONTEXT.put(namespace, cache);
                    RTopic topic = redissonService.getClient().getTopic(namespace + ":change");
                    TOPIC_CONTEXT.put(namespace, topic);
                    //监听缓存变化事件
                    topic.addListener(String.class, (channel, msg) -> {
                        String[] msgInfo = StringUtils.split(msg, CHANGE_MSG_SLIPTOR);
                        if (!CURRENT_CLIENT_ID.equals(msgInfo[0])) {
                            //消息发送端是当前机器，则不做处理
                            if (CLEAR_ALL_MSG.equals(msgInfo[1])) {
                                CACHE_CONTEXT.get(namespace).invalidateAll();
                            } else if (msgInfo.length == 2) {
                                CACHE_CONTEXT.get(namespace).invalidate(msgInfo[1]);
                            } else if (msgInfo.length > 2) {
                                Cache<String, Map<String, T>> firstCache = CACHE_CONTEXT.get(namespace);
                                Map<String, T> cacheMap = firstCache.get(msgInfo[1], k -> null);
                                if (null != cacheMap) {
                                    cacheMap.remove(msgInfo[2]);
                                }
                            }
                        }
                    });
                }
            }
        }
        return cache;
    }
}
