package com.digiwin.metadatacache.services;

import com.digiwin.app.redis.service.DWRedisService;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Service
public class MdcCacheService {

    private final Log log = LogFactory.getLog(getClass());

    @Autowired
    private DWRedisService redisService;

    @Autowired
    private RedissonService redissonService;

    private ThreadPoolExecutor poolExecutor = new ThreadPoolExecutor(10, 50,
            60,
            TimeUnit.SECONDS,
            new ArrayBlockingQueue<>(50),
            new ThreadPoolExecutor.CallerRunsPolicy());

    /**
     * 從快取取得資料
     *
     * @param pCacheMapKey 主key
     * @param pKey         副key
     * @return value
     */
    public String getCacheMapValue(String pCacheMapKey, String pKey) {
        try {
            return (String) redisService.get(pCacheMapKey + MdcSymbolConstant.COLON + pKey);
        } catch (Exception e) {
            log.warn("请求redis异常方法:getCacheMapValue() " + pCacheMapKey + "--" + pKey, e);
            return null;
        }
    }

    public Object get(String pKey) {
        try {
            return redisService.get(pKey);
        } catch (Exception e) {
            log.warn("请求redis异常:get() " + pKey, e);
            return null;
        }
    }

    public void remove(String pKey) {
        executionAndRetry(pKey, () -> redisService.remove(pKey));
    }

    public void set(String pKey, Object pValue) {
        executionAndRetry(pKey, () -> redisService.set(pKey, pValue, getOneDayWithRandomLiveSeconds()));
    }

    public void setSync(String pKey, Object pValue) {
        executionAndRetry(pKey, () -> redisService.set(pKey, pValue, getOneDayWithRandomLiveSeconds()));
    }

    /**
     * 设置对应的过期时间 + 随机数
     *
     * @param pKey    pKey
     * @param pValue  pValue
     * @param timeout timeout
     */
    public void setSyncTimeOut(String pKey, Object pValue, long timeout) {
        executionAndRetry(pKey, () -> redisService.set(pKey, pValue, timeout));
    }

    public void setCacheMapValue(String pCacheMapKey, String pKey, String pValue) {
        String tErrorMsgKey = pCacheMapKey + ' ' + pKey;
        executionAndRetry(tErrorMsgKey, () -> {
            redisService.set(pCacheMapKey + MdcSymbolConstant.COLON + pKey, pValue, getOneDayWithRandomLiveSeconds());
        });
    }

    /**
     * 快取移除資料 鍵值包含某字段
     **/
    public void removeCacheMapValueContainsWord(String pCacheMapKey, String pWord) {
        String tErrorMsgKey = pCacheMapKey + ' ' + pWord;
        executionAndRetry(tErrorMsgKey, () -> {
            String pattern = pCacheMapKey + MdcSymbolConstant.ASTERISK_SYMBOL;
            Set<String> keys = redisService.keys(pattern);
            if (CollectionUtils.isNotEmpty(keys)) {
                keys.forEach(one -> {
                    if (StringUtils.substringAfter(one, pattern).contains(pWord)) {
                        redisService.remove(one);
                    }
                });
            }
        });
    }

    /**
     * 清除redis之中所有符合KeyPattern的key
     *
     * @param pKeyPattern pKeyPattern
     */
    public void removeByPattern(String pKeyPattern) {
        executionAndRetry(pKeyPattern, () -> {
            Set<String> keys = redisService.keys(pKeyPattern);
            for (String key : keys) {
                redisService.remove(key);
            }
        });
    }

    public Set<String> getKeysByPattern(String pKeyPattern) {
        return redisService.keys(pKeyPattern);
    }

    private void executionAndRetry(String pErrorMsgKey, CacheExecutionHandler pHandler) {
        poolExecutor.execute(() -> {
            int tRetryTimes = 2;
            while (true) {
                try {
                    pHandler.execute();
                    break;
                } catch (Exception e) {
                    log.error(e);
                    tRetryTimes--;
                    if (tRetryTimes <= 0) {
                        break;
                    }
                    // 重試間隔
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException e1) {
                        log.error(e1);
                        Thread.currentThread().interrupt();
                    }
                }
            }
        });
    }

    private interface CacheExecutionHandler {
        void execute();
    }

    public boolean lock(String pLockKey) throws InterruptedException {
        return redissonService.lock(pLockKey);
    }

    public void unLock(String pLockKey) {
        redissonService.unLock(pLockKey);
    }

    private long getOneDayWithRandomLiveSeconds() {
        int liveTime = 24 * 60 + RandomUtils.nextInt(0, 6 * 60);
        return TimeUnit.MINUTES.toSeconds(liveTime);
    }

}
