package com.digiwin.metadatacache.util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.type.CollectionType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import lombok.Getter;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/6/2 15:44
 */
public class JsonUtil {

    @Getter
    private final static ObjectMapper MAPPER = new ObjectMapper();

    static {
        //true - 遇到没有的属性就报错 false - 没有的属性不会管，不会报错
        MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        //如果是空对象的时候,不抛异常
        MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        //该特性决定parser是否允许单引号来包住属性名称和字符串值
        MAPPER.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
        // 忽略null欄位
        MAPPER.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    public static String toJson(Object obj) {
        if (null == obj) {
            return null;
        }
        try {
            return MAPPER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T toJavaObj(String json, Class<T> tClass) {
        if (StringUtils.isBlank(json)) {
            return null;
        }
        try {
            return MAPPER.readValue(json, tClass);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T toJavaObj(JsonNode json, Class<T> tClass) {
        if (null == json) {
            return null;
        }
        try {
            return MAPPER.treeToValue(json, tClass);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> List<T> toJavaList(String jsonArray, Class<T> elementType) {
        if (StringUtils.isBlank(jsonArray)) {
            return null;
        }

        try {
            CollectionType listType = TypeFactory.defaultInstance().constructCollectionType(List.class, elementType);
            return MAPPER.readValue(jsonArray, listType);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static JsonNode readJsonNode(InputStream in) {
        if (null == in) {
            return null;
        }
        try {
            return MAPPER.readTree(in);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static JsonNode readJsonNode(String json) {
        if (StringUtils.isBlank(json)) {
            return null;
        }
        try {
            return MAPPER.readTree(json);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static ObjectNode createJsonObject() {
        return MAPPER.createObjectNode();
    }

    /**
     * 如果是json则转换成Json对象，否则返回原字符串信息
     *
     * @param message
     * @return
     */
    public static Object toObjectWhenIsJson(String message) {
        if (StringUtils.isBlank(message)) {
            return message;
        }
        String tmp = StringUtils.trimToEmpty(message);
        boolean isObjectFormat = tmp.startsWith("{") && tmp.startsWith("}");
        boolean isArrayFormat = tmp.startsWith("[") && tmp.startsWith("]");
        if (!isObjectFormat && !isArrayFormat) {
            return message;
        }
        try {
            return MAPPER.readTree(message);
        } catch (Exception e) {
            return message;
        }
    }
}