package com.digiwin.metadatacache.apiservice;

import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.dao.ApiVersionDao;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.ApiVersion;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.services.LoadCacheService;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 取得API版本基础信息(用于ESP-SDK)
 * 做本地缓存
 * Path:[/restful/standard/mdc/ApiBaseInfo/Get]
 */
@Service
public class ApiBaseInfoGetService extends AbstractApiService {

    @Autowired
    private ApiVersionDao apiVersionDao;

    @Autowired
    protected LoadCacheService loadCacheService;

    public ApiBaseInfoGetService() {
        super();
        jsonSchemaFileName = JsonSchemaFileConstant.API_BASE_INFO_GET_SCHEMA;
    }

    /**
     * 查询API信息
     *
     * @param validatorResult * @param validatorResult
     * @return map
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
        // 組成回傳訊息
        ObjectNode tDataNode = mapper.createObjectNode();
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
                + ", 原始訊息: " + tRequestJsonNode);
        String apiName = tRequestJsonNode.get(MdcConstant.API_NAME).asText();
        String tenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
        String key = CacheMapTypeEnum.api_base_info_cachemap.getCode() + MdcSymbolConstant.COLON + apiName;
        ObjectNode data = loadCacheService.getData(key, tenantId);
        if (null == data) {
            ApiVersion apiVersion = getBaseInfo(apiName, tenantId);
            if (null != apiVersion) {
                tDataNode.put(MdcConstant.API_NAME, apiName);
                tDataNode.put(MdcConstant.TENANT_ID, apiVersion.getApi().getTenantId());
                tDataNode.put(MdcConstant.IDEMPOTENCY, apiVersion.isIdempotency());
                tDataNode.put(MdcConstant.VERSION, apiVersion.getVersion());
                tDataNode.put(MdcConstant.APPROVED_STATUS, apiVersion.getApprovedStatus());
                log.info(key + ": " + tDataNode.toString());
                loadCacheService.saveData(key, tenantId, tDataNode);
                log.info("缓存存入: " + key + MdcSymbolConstant.COLON + tenantId);
            }
        } else {
            log.info("缓存检索成功: " + key + MdcSymbolConstant.COLON + tenantId);
            tDataNode = data;
        }
        StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
        return converJsonNodeToMap(tResponseJsonNode);
    }

    /**
     * 通过租户+api查询最新API版本信息
     *
     * @param apiName  Api名称
     * @param tenantId 租户Id
     * @return apiVersion
     */
    protected ApiVersion getBaseInfo(String apiName, String tenantId) {
        // 通过租户Id与api名称查询version
        List<ApiVersion> apiVersionList = apiVersionDao.getBaseInfoByApiNameAndTenantId(apiName, tenantId);
        if (CollectionUtils.isNotEmpty(apiVersionList)) {
            List<ApiVersion> apiVersions
                    = apiVersionList.stream().filter(x -> StringUtils.isNotBlank(x.getApi().getTenantId())
                    && tenantId.equals(x.getApi().getTenantId())).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(apiVersions)) {
                return getLastedVersion(apiVersions);
            } else {
                return getLastedVersion(apiVersionList);
            }
        }
        return null;
    }
}
