package com.digiwin.metadatacache.apiservice;

import com.digiwin.app.service.DWServiceContext;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.dao.ApiDao;
import com.digiwin.metadatacache.dao.ApiRelationDao;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.exception.ApiNameNotExistException;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.exception.ParentApiRelationExistException;
import com.digiwin.metadatacache.exception.TokenNotRetrievedException;
import com.digiwin.metadatacache.model.ApiRelation;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.services.LocalCacheService;
import com.digiwin.metadatacache.util.IamVerifyServiceUtil;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Map;

/**
 * 取得API映射关系
 * Path:[/restful/standard/mdc/ApiRelation/Save]
 */
@Service
public class ApiRelationSaveService extends AbstractApiService {

	@Autowired
	private ApiRelationDao apiRelationDao;

	@Autowired
	private LocalCacheService localCacheService;

	@Autowired
	private ApiDao apiDao;

	public ApiRelationSaveService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.API_RELATION_SAVE_SCHEMA;
	}

	/**
	 * 保存映射关系信息
	 * @param validatorResult      * @param validatorResult
	 * @return map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		// 1、获取token信息
		String tToken = (String) DWServiceContext.getContext().getRequestHeader().get(MdcConstant.TOKEN);
		if (null == tToken) {
			throw new TokenNotRetrievedException(getClass());
		}
		String userName = IamVerifyServiceUtil.getUserName(tToken);
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
				+ ", 原始訊息: " + tRequestJsonNode);
		ArrayNode tApiArrayNode = (ArrayNode) tRequestJsonNode.get(MdcConstant.API_INFO);
		if (tApiArrayNode.isArray()) {
			for (JsonNode tApi : tApiArrayNode) {
				String parentApiName = tApi.get(MdcConstant.PARENT_API_NAME).asText();
				String subApiName = tApi.get(MdcConstant.SUB_API_NAME).asText();
				String tenantId = null;
				if (null != tApi.get(MdcConstant.SUB_TENANT_ID)
						&& !StringUtil.isEmptyOrSpace(tApi.get(MdcConstant.SUB_TENANT_ID).asText())) {
					tenantId = tApi.get(MdcConstant.SUB_TENANT_ID).asText();
				}
				// 2、确认对应的父类API是否存在
				boolean parentApiJudge = apiDao.judgeApi(parentApiName, tenantId, true);
				if (!parentApiJudge) {
					throw new ApiNameNotExistException("父类API" + parentApiName + "不存在" , getClass());
				}
				// 3、确认子类API是否存在
				boolean subApiJudge = apiDao.judgeApi(subApiName, tenantId, false);
				if (!subApiJudge) {
					throw new ApiNameNotExistException("子类API" + subApiName + "不存在", getClass());
				}
				// 4、判定父类api是否为子类
				ApiRelation parentApiRelation = apiRelationDao.judgeIsSub(parentApiName, tenantId);
				if (null != parentApiRelation) {
					throw new ParentApiRelationExistException("父类API" + parentApiName + "作为子类关联关系已存在", getClass());
				}
				// 5、判定子类api是否为父类
				ApiRelation subApiRelation = apiRelationDao.judgeIsParent(subApiName, tenantId);
				if (null != subApiRelation) {
					throw new ParentApiRelationExistException("子类API" + subApiName + "作为父类关联关系已存在", getClass());
				}
				saveApiRelation(parentApiName, subApiName, tenantId, userName);
			}
		}
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription,null);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 保存关联关系
	 * @param parentApiName 父类API
	 * @param subApiName    子类API
	 * @param tenantId      租户ID
	 * @param userName      用户名称
	 */
	public void saveApiRelation(String parentApiName, String subApiName, String tenantId, String userName) {
        // 6、确定绑定关系是否已经存在,存在则更新 不存在则新增
		ApiRelation apiRelation = apiRelationDao.getRelationBySubInfo(subApiName, tenantId);
		localCacheService.remove(CacheMapTypeEnum.api_relation_idempotency_cachemap, subApiName);
		if (null == apiRelation) {
			apiRelation = new ApiRelation();
			apiRelation.setParentApiName(parentApiName);
			apiRelation.setSubApiName(subApiName);
			apiRelation.setSubTenantId(tenantId);
			apiRelation.setBuildTime(LocalDateTime.now());
			apiRelation.setCreateId(userName);
			apiRelationDao.save(apiRelation);
		} else {
			apiRelation.setUpdateTime(LocalDateTime.now());
			apiRelation.setUpdateId(userName);
			apiRelation.setParentApiName(parentApiName);
			apiRelationDao.saveOrupdate(apiRelation);
		}
	}
}
