package com.digiwin.metadatacache.apiservice;

import com.digiwin.metadatacache.constant.AssetConstant;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.dao.AssetDataModelDao;
import com.digiwin.metadatacache.dao.AssetDataQueryDao;
import com.digiwin.metadatacache.dao.AssetPropertyDao;
import com.digiwin.metadatacache.enums.PropertySourceEnum;
import com.digiwin.metadatacache.enums.PropertyTypeEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.*;
import com.digiwin.metadatacache.services.ElasticsearchService;
import com.digiwin.metadatacache.util.IdUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;

/**
 * 资产中心 - 资产注册
 * Path:[/restful/standard/mdc/Asset/Register]
 */
@Service
public class AssetRegisterService extends AbstractApiService {

    @Autowired
    private AssetDataModelDao assetDataModelDao;

    @Autowired
    private AssetDataQueryDao assetDataQueryDao;

    @Autowired
    private AssetPropertyDao assetPropertyDao;

    @Autowired
    private ElasticsearchService elasticsearchService;

    @Autowired
    AssetRegisterService assetRegisterService;

    public AssetRegisterService() {
        super();
        jsonSchemaFileName = JsonSchemaFileConstant.ASSET_REGISTER_SCHEMA;
    }

    /**
     * 注册资产(业务数据模型、查询方案)
     * API 另一个接口内部实现
     *
     * @param validatorResult validatorResult
     * @return Map
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
        // 1、获取token信息
//        String tToken = (String) DWServiceContext.getContext().getRequestHeader().get(MdcConstant.TOKEN);
//        if (null == tToken) {
//            throw new TokenNotRetrievedException(getClass());
//        }
//        String userName = IamVerifyServiceUtil.getUserName(tToken);
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        assetRegisterService.saveAsset(tRequestJsonNode);
        StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
        return converJsonNodeToMap(tResponseJsonNode);
    }

    @Transactional(rollbackFor = Exception.class,value = "mdcTransactionManager")
    public void saveAsset(JsonNode tRequestJsonNode) throws Exception {
        ArrayNode tApiArrayNode = (ArrayNode) tRequestJsonNode.get(MdcConstant.ASSET_DATA);
        LocalDateTime localDateTime = LocalDateTime.now();
        Gson gson = new Gson();
        for (JsonNode jsonNode : tApiArrayNode) {
            AssetProperty assetProperty = new AssetProperty();
            String propertySource = jsonNode.get(AssetConstant.PROPERTY_SOURCE).asText();
            String ownerProduct = jsonNode.get(AssetConstant.OWNER_PRODUCT).asText();
            String propertyType = jsonNode.get(AssetConstant.PROPERTY_TYPE).asText();
            assetProperty.setId(IdUtil.nextId());
            assetProperty.setPropertySource(propertySource);
            assetProperty.setOwnerProduct(ownerProduct);
            assetProperty.setPropertyType(propertyType);
            //assetProperty.setCreateBy(userName);
            assetProperty.setCreateDate(localDateTime);
            JsonNode jsonNodeData = jsonNode.get(AssetConstant.DATA);
            Map<String, Object> message = new HashMap<>();
            Long id = jsonNodeData.get(AssetConstant.ID).asLong();
            assetProperty.setPropertyKey(id.toString());
            String code = jsonNodeData.get(AssetConstant.CODE).asText();
            String name = jsonNodeData.get(AssetConstant.NAME).asText();
            assetProperty.setCode(code);
            assetProperty.setName(name);
            if (PropertyTypeEnum.BusinessDataModel.toString().equals(propertyType)) {
                AssetDataModel assetDataModel = new AssetDataModel();
                assetDataModel.setId(id);
                assetDataModel.setCode(code);
                assetDataModel.setName(name);
                assetDataModel.setTables(jsonNodeData.get(AssetConstant.TABLES).asText());
                assetDataModel.setLinks(jsonNodeData.get(AssetConstant.LINKS).asText());
                assetDataModel.setType(jsonNodeData.get(AssetConstant.TYPE).asText());
                message.put(AssetConstant.ID, id);
                message.put(AssetConstant.CODE, code);
                message.put(AssetConstant.NAME, name);
                message.put(AssetConstant.TABLES, gson.fromJson(jsonNodeData.get(AssetConstant.TABLES).asText(), new TypeToken<ArrayList<Map<String, Object>>>() {}.getType()));
                message.put(AssetConstant.LINKS, gson.fromJson(jsonNodeData.get(AssetConstant.LINKS).asText(), new TypeToken<ArrayList<Map<String, Object>>>() {}.getType()));
                message.put(AssetConstant.TYPE, jsonNodeData.get(AssetConstant.TYPE).asText());
                boolean assetModelDataExists = assetDataModelDao.exists(assetDataModel.getId());
                if (assetModelDataExists) {
                    //assetDataModel.setModifiedBy(userName);
                    assetDataModel.setModifiedDate(localDateTime);
                    assetDataModelDao.saveOrupdate(assetDataModel);
                } else {
                    //assetDataModel.setCreateBy(userName);
                    assetDataModel.setCreateDate(localDateTime);
                    assetDataModelDao.save(assetDataModel);
                }
            } else if (PropertyTypeEnum.QueryProject.toString().equals(propertyType)) {
                AssetDataQuery assetDataQuery = new AssetDataQuery();
                assetDataQuery.setId(id);
                assetDataQuery.setCode(code);
                assetDataQuery.setName(name);
                assetDataQuery.setModelId(jsonNodeData.get(AssetConstant.MODEL_ID).asLong());
                assetDataQuery.setPlayground(jsonNodeData.get(AssetConstant.PLAY_GROUND).asText());
                message.put(AssetConstant.ID, id);
                message.put(AssetConstant.CODE, code);
                message.put(AssetConstant.NAME, name);
                message.put(AssetConstant.MODEL_ID, jsonNodeData.get(AssetConstant.MODEL_ID).asText());
                message.put(AssetConstant.PLAY_GROUND, gson.fromJson(jsonNodeData.get(AssetConstant.PLAY_GROUND).asText(), new TypeToken<Map<String, Object>>(){}.getType()));
                boolean assetDataQueryExists = assetDataQueryDao.exists(assetDataQuery.getId());
                if (assetDataQueryExists) {
                    //assetDataQuery.setModifiedBy(userName);
                    assetDataQuery.setModifiedDate(localDateTime);
                    assetDataQueryDao.saveOrupdate(assetDataQuery);
                } else {
                    //assetDataQuery.setCreateBy(userName);
                    assetDataQuery.setCreateDate(localDateTime);
                    assetDataQueryDao.save(assetDataQuery);
                }
            }
            //1、删除资产表;删除ES
            deletePropertyAndES(assetProperty.getCode(), assetProperty.getPropertyType(), assetProperty.getTenantId());
            //2、新增资产表;新增ES
            insertPropertyAndES(assetProperty, message);
        }
    }

    public void saveAssetProperty(String tApiName, String description, String provider,
                                  ApiMetadata apiMetadata, String tTenantId, String tUserId) throws Exception {
        AssetProperty assetProperty = new AssetProperty();
        assetProperty.setId(IdUtil.nextId());
        assetProperty.setCode(tApiName);
        assetProperty.setName(description);
        assetProperty.setTenantId(tTenantId);
        assetProperty.setPropertyKey(apiMetadata.getId().toString());
        assetProperty.setPropertySource(tApiName.startsWith(PropertySourceEnum.dmp.toString())
                ? PropertySourceEnum.dmp.toString() : tApiName.startsWith(PropertySourceEnum.bmd.toString())
                ? PropertySourceEnum.bmd.toString() : tApiName.startsWith(PropertySourceEnum.bm.toString())
                ? PropertySourceEnum.bm.toString() : null);
        if (null != provider) {
            assetProperty.setOwnerProduct(provider);
        }
        assetProperty.setPropertyType(PropertyTypeEnum.API.toString());
        assetProperty.setCreateBy(tUserId);
        assetProperty.setCreateDate(LocalDateTime.now());
        assetProperty.setModifiedDate(LocalDateTime.now());
        assetProperty.setModifiedBy(tUserId);
        Map<String, Object> message = new HashMap<>();
        message.put(AssetConstant.ID, apiMetadata.getId());
        message.put(AssetConstant.CODE, tApiName);
        message.put(AssetConstant.NAME, description);
        message.put(AssetConstant.API_VERSION_ID, apiMetadata.getApiVersionId());
        message.put(AssetConstant.METADATA, new Gson().fromJson(apiMetadata.getMetadata(), new TypeToken<Map<String, Object>>() {}.getType()));
        insertPropertyAndES(assetProperty, message);
    }

    /**
     * 删除资产表相关API数据 (通过API名称与租户与类别)
     *
     * @param code     api名称
     * @param tenantId 租户Id
     */
    public void deletePropertyAndES(String code, String propertyType, String tenantId) throws Exception {
        Map<String, String> pCondition = new HashMap<>();
        pCondition.put(AssetConstant.CODE, code);
        pCondition.put(MdcConstant.TENANT_ID, tenantId);
        pCondition.put(AssetConstant.PROPERTY_TYPE, propertyType);
        List<AssetProperty> assetProperties = assetPropertyDao.getAssetPropertyByInfo(pCondition);
        if (CollectionUtils.isNotEmpty(assetProperties)) {
            for (AssetProperty assetProperty : assetProperties) {
                assetPropertyDao.remove(assetProperty.getId());
                //删除es指定docId
                elasticsearchService.deleteAssetDataDoc(assetProperty.getId().toString());
            }
        }
    }

    /**
     * 保存资产表相关API数据 (通过API名称与租户与类别)
     *
     * @param assetProperty 资产信息
     */
    public void insertPropertyAndES(AssetProperty assetProperty, Map<String, Object> message) throws Exception {
        assetPropertyDao.save(assetProperty);
        //保存文档,指定docId
        Map<String ,Object> objectNode = new HashMap<>();
        objectNode.put(AssetConstant.ID, assetProperty.getId());
        objectNode.put(AssetConstant.CODE, assetProperty.getCode());
        objectNode.put(AssetConstant.NAME, assetProperty.getName());
        objectNode.put(AssetConstant.CONTENT, message);
        objectNode.put(AssetConstant.PROPERTY_SOURCE, assetProperty.getPropertySource());
        objectNode.put(AssetConstant.OWNER_PRODUCT, assetProperty.getOwnerProduct());
        objectNode.put(AssetConstant.PROPERTY_TYPE, assetProperty.getPropertyType());
        elasticsearchService.insertAssetDataDoc(assetProperty.getId().toString(), objectNode);
    }
}
