package com.digiwin.metadatacache.apiservice;

import com.digiwin.app.service.DWServiceChainContext;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.http.client.exception.DWHttpFailedException;
import com.digiwin.metadatacache.apiservice.request.SeviceBaseInfoGetRequest;
import com.digiwin.metadatacache.apiservice.response.SeviceBaseInfoGetResponse;
import com.digiwin.metadatacache.bo.ApiRelationCacheVal;
import com.digiwin.metadatacache.bo.KgServiceProductConfigDto;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.enums.ErrorCodeEnum;
import com.digiwin.metadatacache.enums.ValidateStateEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.exception.MultipleProdException;
import com.digiwin.metadatacache.model.ApiVersion;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.services.DispatchService;
import com.digiwin.metadatacache.services.MdcKgService;
import com.digiwin.metadatacache.services.LocalCacheService;
import com.digiwin.metadatacache.services.MdcProductService;
import com.digiwin.metadatacache.util.IamVerifyServiceUtil;
import com.digiwin.metadatacache.util.JsonUtil;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2024/7/18 11:16
 *  Path:[/restful/standard/mdc/ServiceBaseInfo/get]
 */
@org.springframework.stereotype.Service
public class ServiceBaseInfoGetService extends ApiBaseInfoGetService {

    @Autowired
    private MdcProductService productService;

    @Autowired
    private TenantProductOperationListGetService tenantProductOperationListGetService;

    @Autowired
    private DispatchService dispatchService;

    @Autowired
    private LocalCacheService localCacheService;

    @Autowired
    private MdcKgService kgService;

    public ServiceBaseInfoGetService() {
        super();
        jsonSchemaFileName = JsonSchemaFileConstant.SERVICE_BAE_INFO_GET_SCHEMA;
    }

    @Override
    public Map<String, Object> execute(String pRequestMsg) throws Exception {
        StateCode tStateCode;
        StringBuilder tDescription = new StringBuilder();
        if (DWServiceContext.getContext().getRequestHeader().get("locale") != null) {
            locale = DWServiceContext.getContext().getRequestHeader().get("locale").toString();
        }
        try {
            ValidatorResult tValidatorResult = validate(pRequestMsg);
            return processData(tValidatorResult);
        } catch (DWHttpFailedException e) {
            String exceptionEntity = e.getEntity(String.class);
            log.error("exceptionEntity = " + exceptionEntity + ";requestMsg = " + pRequestMsg, e);
            throw new Exception(exceptionEntity);
        } catch (BaseException e) {
            log.error(e.getMessage() + ";requestMsg = " + pRequestMsg, e);
            tStateCode = getStateCode(e.getStateEnum().getCode());
            tDescription.append(getDescriptionByLocale(tStateCode, locale));
            if (!StringUtil.isEmptyOrSpace(e.getMessage())) {
                tDescription.append(MdcSymbolConstant.COLON_CN).append(e.getMessage());
            }
        } catch (Exception e) {
            log.error(e.getMessage() + ";requestMsg = " + pRequestMsg, e);
            tStateCode = getStateCode(ValidateStateEnum.UNEXPECTED_ERROR.getCode());
            tDescription.append(getDescriptionByLocale(tStateCode, locale));
        }
        return convertErrorResult(tStateCode.getCode(), tDescription.toString());
    }

    private Map<String, Object> convertErrorResult(String stateCode, String description) {
        Map<String, Object> response = new HashMap<>();
        ErrorCodeEnum errorCode = ErrorCodeEnum.ERRORCODEENUM_CODE_MAP.get(stateCode);
        if (null == errorCode) {
            errorCode = ErrorCodeEnum.UNEXPECTED_ERROR;
        }
        response.put("errorCode", errorCode.getErrorCode());
        // 获取码值对应的异常信息
        response.put("errorMessage", description);
        String chainInfo = DWServiceChainContext.getContext().getTraceElement().getTraceInfo(true);
        chainInfo = StringUtils.substringAfter(chainInfo, MdcConstant.SUBSTRING);
        Map chainInfoNode = new HashMap();
        chainInfoNode.put("chainInfo", chainInfo);
        // 获取信息链路信息
        response.put("errorInstructors", chainInfoNode);
        return response;
    }

    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws Exception {
        String tDescription = getDescriptionByLocale(SUCCESS_STATE, locale);
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
                + ", 原始訊息: " + tRequestJsonNode);
        SeviceBaseInfoGetRequest request = JsonUtil.toJavaObj(tRequestJsonNode, SeviceBaseInfoGetRequest.class);
        SeviceBaseInfoGetResponse response = new SeviceBaseInfoGetResponse();
        String apiName = request.getApiName();
        String tenantId = request.getTenantId();

        SeviceBaseInfoGetResponse.Api apiInfo = getApiInfo(apiName, tenantId, request.isNeedIdempotency());
        response.setApi(apiInfo);

        String realApiName = StringUtils.isNotBlank(apiInfo.getParentApi()) ? apiInfo.getParentApi() : apiName;
        SeviceBaseInfoGetResponse.Product product = getProductInfo(request, realApiName);

        response.setProd(product);
        if (null == product || ApiTypeEnum.agile.toString().equals(product.getProductType())) {
            String prodName = null == product ? request.getServiceProd() : product.getName();
            Map<String, Object> eoc = createAgileEmptyEocRs(prodName, tenantId);
            response.setEoc(eoc);
            return createResponseMap(SUCCESS_STATE.getCode(), tDescription, response.toMap());
        }

        //获取当前租户的版本
        String version = IamVerifyServiceUtil.getCurrentTenantVersion(tenantId);
        response.setTenantVersion(version);

        Map<String, Object> eoc;
        //v2 版本逻辑处理
        if(MdcConstant.TENANT_VERSION_V2.equalsIgnoreCase(version)){
            eoc = tenantProductOperationListGetService.tenantProdOperationListGetV2(tenantId, product.getName(), product.getUid(), null);
        //v1版本处理
        }else {
            eoc = tenantProductOperationListGetService.processTenantProdOperationListGet(tenantId, product.getName(), product.getUid(), null);
        }
        eoc.remove(MdcConstant.EXECUTION);
        response.setEoc(eoc);
        if (!request.isNeedDispatch()) {
            return createResponseMap(SUCCESS_STATE.getCode(), tDescription, response.toMap());
        }
        if (!dispatchService.isCanDispatch(product.getName(), realApiName)) {
            return createResponseMap(SUCCESS_STATE.getCode(), tDescription, response.toMap());
        }
        List<Map<String, Object>> dispatchList;
        //v2 版本逻辑处理
        if(MdcConstant.TENANT_VERSION_V2.equalsIgnoreCase(version)){
            dispatchList = dispatchService.getDispatchV2(tenantId, product.getName());
        //v1版本处理
        }else {
            dispatchList = dispatchService.getDispatch(tenantId, product.getName());
        }
        response.setDispatch(dispatchList);
        return createResponseMap(SUCCESS_STATE.getCode(), tDescription, response.toMap());
    }

    private SeviceBaseInfoGetResponse.Product getProductInfo(SeviceBaseInfoGetRequest request, String realApiName) throws BaseException {
        List<Product> products = getProductFromCache(request, realApiName);
        if (CollectionUtils.isEmpty(products)) {
            products = productService.getProductFromDb(request.getTenantId(), realApiName, request.getServiceProd(), request.getServiceProdUid());
        }
        if (CollectionUtils.isEmpty(products)) {
            return null;
        }
        Product matchedProd = null;
        if (products.size() == 1) {
            matchedProd = products.get(0);
        } else {
            matchedProd = matchProductByConfig(request.getTenantId(), realApiName, products);
        }
        return new SeviceBaseInfoGetResponse.Product(matchedProd.getName(), matchedProd.getUid(), matchedProd.getProductType());
    }

    private SeviceBaseInfoGetResponse.Product getProductInfoV2(SeviceBaseInfoGetRequest request, String realApiName) throws BaseException {
        List<Product> products = productService.getProductFromDbNoCache(request.getTenantId(), realApiName, request.getServiceProd(), request.getServiceProdUid());

        if (CollectionUtils.isEmpty(products)) {
            return null;
        }
        Product matchedProd = null;
        if (products.size() == 1) {
            matchedProd = products.get(0);
        } else {
            matchedProd = matchProductByConfig(request.getTenantId(), realApiName, products);
        }
        return new SeviceBaseInfoGetResponse.Product(matchedProd.getName(), matchedProd.getUid(), matchedProd.getProductType());
    }

    private Product matchProductByConfig(String tenantId, String realApi, List<Product> products) throws BaseException {
        KgServiceProductConfigDto configDto = kgService.getServiceProductConfig(tenantId, realApi);
        if (null == configDto
                || (null == configDto.getTenantServiceConfig() && CollectionUtils.isEmpty(configDto.getSystemServiceConfig()))) {
            throw new MultipleProdException(getClass());
        }
        Product matchedProd = null;
        if (configDto.getTenantServiceConfig() != null) {
            matchedProd = products.stream().filter(prod -> StringUtils.equals(prod.getName(), configDto.getTenantServiceConfig().getProductName()))
                    .findFirst().orElse(null);
        }
        if (CollectionUtils.isEmpty(configDto.getSystemServiceConfig())) {
            return matchedProd;
        }
        return products.stream().filter(prod -> configDto.getSystemServiceConfig().contains(prod.getName()))
                .findFirst().orElse(null);
    }

    /**
     * 通过租户+api查询父类Api名称
     * @param apiName 子类Api名称
     * @param tenantId 租户Id
     * @return 父类Api名称
     */
    private SeviceBaseInfoGetResponse.Api getApiInfo(String apiName, String tenantId, boolean needIdempotency) {
        ApiRelationCacheVal relationCacheVal = localCacheService.getFromL2Cache(CacheMapTypeEnum.api_relation_idempotency_cachemap, apiName, tenantId);
        SeviceBaseInfoGetResponse.Api api = new SeviceBaseInfoGetResponse.Api();
        api.setName(apiName);
        if (relationCacheVal != null) {
            api.setParentApi(relationCacheVal.getParent());
            if (!needIdempotency) {
                return api;
            }
            if (relationCacheVal.getIdempotency() != null) {
                api.setIdempotency(relationCacheVal.getIdempotency());
                return api;
            }
        } else {
            String parentName = getParentName(apiName, tenantId);
            api.setParentApi(parentName);
        }
        if (needIdempotency) {
            api.setIdempotency(getIdempotency(apiName, tenantId, api.getParentApi()));
        }
        //缓存api信息
        saveApiRelationCache(apiName, tenantId, new ApiRelationCacheVal(api.getParentApi(), api.getIdempotency()));
        return api;
    }

    /**
     * 通过租户+api查询父类Api名称
     * @param apiName 子类Api名称
     * @param tenantId 租户Id
     * @return 父类Api名称
     */
    private SeviceBaseInfoGetResponse.Api getApiInfoV2(String apiName, String tenantId, boolean needIdempotency) {
        SeviceBaseInfoGetResponse.Api api = new SeviceBaseInfoGetResponse.Api();
        api.setName(apiName);
        String parentName = getParentName(apiName, tenantId);
        api.setParentApi(parentName);

        if (needIdempotency) {
            api.setIdempotency(getIdempotency(apiName, tenantId, api.getParentApi()));
        }
        return api;
    }

    private boolean getIdempotency(String apiName, String tenantId, String parentApi) {
        String realApiName = null == parentApi ? apiName : parentApi;
        ApiVersion version = getBaseInfo(realApiName, tenantId);
        boolean idempotency = null == version ? false : version.isIdempotency();
        return idempotency;
    }

    private void saveApiRelationCache(String apiName, String tenantId, ApiRelationCacheVal val) {
        localCacheService.putToL2Cache(CacheMapTypeEnum.api_relation_idempotency_cachemap, apiName, tenantId, val);
    }

    private List<Product> getProductFromCache(SeviceBaseInfoGetRequest request, String realApiName) {
        String tenantId = request.getTenantId();
        String tStandardKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + tenantId;
        Map<String, List<String>> standardCache = (Map<String, List<String>>) cacheService.get(tStandardKey);
        List<Product> products = productService.getStandardProductCache(tenantId, standardCache, realApiName);
        Predicate<Product> filter = getProductFilter(request);
        if (CollectionUtils.isNotEmpty(products)) {
            log.info("命中快取: " + tStandardKey);
            products = products.stream().filter(filter).collect(Collectors.toList());
        }
        if (CollectionUtils.isEmpty(products)) {
            products = productService.getAgileProductCache(tenantId, realApiName);
            if (CollectionUtils.isNotEmpty(products)) {
                products = products.stream().filter(filter).collect(Collectors.toList());
            }
        }
        return products;
    }

    private Predicate<Product> getProductFilter(SeviceBaseInfoGetRequest request) {
        return product -> {
            if (StringUtils.isBlank(request.getServiceProd()) && StringUtils.isBlank(request.getServiceProdUid())) {
                return true;
            }
            boolean nameMatch = StringUtils.isBlank(request.getServiceProd()) || product.getName().equals(request.getServiceProd());
            boolean uidMatch = StringUtils.isBlank(request.getServiceProdUid()) || product.getUid().equals(request.getServiceProdUid());
            return nameMatch && uidMatch;
        };
    }

    private Map<String, Object> createAgileEmptyEocRs(String prod, String tenantId) {
        Map<String, Object> tResponseJsonNode = new HashMap<>();
        tResponseJsonNode.put(MdcConstant.ORG_TYPE_ENTERPRISE, new HashMap<>());
        tResponseJsonNode.put(MdcConstant.ORG_TYPE_COMPANY, new ArrayList<>());
        tResponseJsonNode.put(MdcConstant.ORG_TYPE_REGION, new ArrayList<>());
        tResponseJsonNode.put(MdcConstant.TENANT_ID, tenantId);
        tResponseJsonNode.put(MdcConstant.PROD_NAME, prod);
        return tResponseJsonNode;
    }

}
