package com.digiwin.metadatacache.apiservice;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.MdcApplicationParameter;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.digiwin.metadatacache.dao.EaiTenantMappingDao;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.enums.ValidateStateEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.EaiTenantMapping;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.Service;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 查詢租戶所有可調用的產品
 * Path:[/restful/standard/mdc/TenantBindProductList/Get]
 */
@org.springframework.stereotype.Service
public class TenantBindProductListGetService extends AbstractApiService {

	@Autowired
	private ProductDao productDao;

	@Autowired
	private EaiTenantMappingDao eaiTenantMappingDao;

	public TenantBindProductListGetService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.TENANT_BIND_PRD_LIST_GET_SCHEMA;
	}

	/**
	 * 入参解析
	 * @param validatorResult      * @param validatorResult
	 * @return map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
				+ ", 原始訊息: " + tRequestJsonNode);
		// 取出入參tenant, prod_type
		String tTenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
		return processBindProductGet(tTenantId);
	}

	/**
	 * 根據租戶ID取得有綁定的產品主機清單
	 * 回傳清單=[該租戶有綁定的產品主機清單]
	 */
	private Map<String, Object> processBindProductGet(String pTenantId) {
		// 組成回傳訊息必要節點
		ObjectNode tDataNode = mapper.createObjectNode();
		tDataNode.put(MdcConstant.TENANT_ID, pTenantId);
		ArrayNode tGatewayArrayNode = mapper.createArrayNode();
		tDataNode.set(MdcConstant.GATEWAY, tGatewayArrayNode);
		// 先取得該租戶所屬地端中台UID
		Map<String, String> tEaiTenantMappingCondition = new HashMap<>();
		tEaiTenantMappingCondition.put(MdcConstant.TENANT, pTenantId);
		List<EaiTenantMapping> tEaiTenantMappings = eaiTenantMappingDao.fetch(tEaiTenantMappingCondition);
		// 有對應中台UID
		if (!tEaiTenantMappings.isEmpty()) {
			// 找出該地中台所有的配置租戶ID，用eai uid找出其他的租戶
			String tEaiUid = tEaiTenantMappings.get(0).getEaUid();
			Map<String, String> tEaiTenantMappingConditionForAllTenant = new HashMap<>();
			tEaiTenantMappingConditionForAllTenant.put(MdcConstant.EAI_UID, tEaiUid);
			List<EaiTenantMapping> tEaiTenantMappingsForAllTenant = eaiTenantMappingDao
					.fetch(tEaiTenantMappingConditionForAllTenant);
			List<String> tTenantList = new ArrayList<>();
			for (EaiTenantMapping tTenantMapping : tEaiTenantMappingsForAllTenant) {
				tTenantList.add(tTenantMapping.getTenantId());
			}
			// 取得該中台UID上的所有產品(tProductList)
			Map<String, String> pProductCondition = new HashMap<>();
			pProductCondition.put(MdcConstant.EAI_UID, tEaiTenantMappings.get(0).getEaUid());
			List<Product> tEaiAllProductList = productDao.fetch(pProductCondition);
			// 將產品主機依照產品名稱分群放入各List
			Map<String, List<Product>> tProductMapGroupByProdName = new HashMap<>();
			for (Product tProduct : tEaiAllProductList) {
				List<Product> tProducts = tProductMapGroupByProdName.get(tProduct.getName());
				if (CollectionUtils.isEmpty(tProducts)) {
					tProducts = new ArrayList<>();
					tProductMapGroupByProdName.put(tProduct.getName(), tProducts);
				}
				tProducts.add(tProduct);
			}
			List<Product> tReturnProductList = new ArrayList<>();
			List<String> tExistUidList = new ArrayList<>();
			// 根據各產品名稱找出有對應到租戶ID的產品主機
			for (String tProductType : tProductMapGroupByProdName.keySet()) {
				for (Product tProduct : tProductMapGroupByProdName.get(tProductType)) {
					// 不要顯示重複產品UID
					if (tProduct.getTenantId() != null && !tExistUidList.contains(tProduct.getUid())
							&& tTenantList.contains(tProduct.getTenantId())) {
						tReturnProductList.add(tProduct);
					}
					if (!StringUtil.isEmptyOrSpace(tProduct.getUid())) {
						tExistUidList.add(tProduct.getUid());
					}
				}
			}
			if (!tReturnProductList.isEmpty()) {
				// 將產品清單轉成分群Map
				Map<String, Map<String, Map<String, List<Product>>>> tProductsMap = transformListToGroupMap(
						tReturnProductList, pTenantId);
				// Gateway
				for (String tGatewayId : tProductsMap.get(pTenantId).keySet()) {
					ObjectNode tGatewayItemNode = mapper.createObjectNode();
					tGatewayItemNode.put(MdcConstant.GATEWAY_ID, tGatewayId);
					tGatewayArrayNode.add(tGatewayItemNode);
					ArrayNode tProductArrayNode = mapper.createArrayNode();
					// Product Name
					for (String tProductName : tProductsMap.get(pTenantId).get(tGatewayId).keySet()) {
						// Product
						for (Product tProduct : tProductsMap.get(pTenantId).get(tGatewayId).get(tProductName)) {
							ObjectNode tProductItemNode = mapper.createObjectNode();
							tProductItemNode.put(MdcConstant.NAME, tProduct.getName());
							tProductItemNode.put(MdcConstant.VERSION, tProduct.getVersion());
							tProductItemNode.put(MdcConstant.IP, tProduct.getIp());
							tProductItemNode.put(MdcConstant.APID, tProduct.getApid());
							tProductItemNode.put(MdcConstant.UID, tProduct.getUid());
							if (MdcApplicationParameter._ERP_LIST.contains(tProduct.getName())) {
								tProductItemNode.put(MdcConstant.TYPE, MdcConstant.ERP);
							}
							tProductArrayNode.add(tProductItemNode);
						}
					}
					tGatewayItemNode.set(MdcConstant.PRODUCT, sortProductArray(tProductArrayNode));
				}
			}
		}
		StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 將List轉成Map，並依照租戶ID、GatewayID及產品名稱分群 <br>
	 * 第一層key: 租戶ID <br>
	 * 第二層key: GatewayID <br>
	 * 第三層key: 產品名稱
	 * 
	 * @param pList pList
	 * @param pTenant pTenant
	 * @return map
	 */
	private <T> Map<String, Map<String, Map<String, List<T>>>> transformListToGroupMap(List<T> pList, String pTenant) {
		Map<String, Map<String, Map<String, List<T>>>> tTenantMap = new HashMap<>();
		for (T tObject : pList) {
			Product tProduct = null;
			if (tObject instanceof Product) {
				tProduct = ((Product) tObject);
			} else if (tObject instanceof Service) {
				tProduct = ((Service) tObject).getProduct();
			}
			if (tProduct != null) {
				// 第一層key: 租戶ID
				Map<String, Map<String, List<T>>> tGatewayMap = tTenantMap.get(pTenant);
				if (tGatewayMap == null) {
					// 建立一個此租戶的gateway Map
					tGatewayMap = new HashMap<>();
					tTenantMap.put(pTenant, tGatewayMap);
				}
				// 第二層key: GatewayID
				Map<String, List<T>> tProductNameMap = tGatewayMap.get(tProduct.getGatewayId());
				if (tProductNameMap == null) {
					// 建立一個此gateway的產品名稱 Map
					tProductNameMap = new HashMap<>();
					tGatewayMap.put(tProduct.getGatewayId(), tProductNameMap);
				}
				// 第三層key: 產品名稱
				List<T> tServiceList = tProductNameMap.get(tProduct.getName());
				if (tServiceList == null) {
					// 建立產品名稱的產品主機list
					tServiceList = new ArrayList<>();
					tProductNameMap.put(tProduct.getName(), tServiceList);
				}
				tServiceList.add(tObject);
			}
		}
		return tTenantMap;
	}

	public ArrayNode sortProductArray(ArrayNode pArrayNode) {
		List<JsonNode> prdList = new ArrayList<>();
		for (JsonNode node : pArrayNode) {
			prdList.add(node);
		}
		Collections.sort(prdList, new Comparator() {
			@Override
			public int compare(Object o1, Object o2) {
				String name1;
				String name2;
				name1 = ((ObjectNode) o1).get(MdcConstant.NAME).asText();
				name2 = ((ObjectNode) o2).get(MdcConstant.NAME).asText();
				return name1.compareTo(name2);
			}
		});
		// 排序後的json array
		ArrayNode sortedJsonArray = mapper.createArrayNode();
		for (JsonNode jsonNode : prdList) {
			sortedJsonArray.add(jsonNode);
		}
		return sortedJsonArray;
	}
}
