package com.digiwin.metadatacache.apiservice;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.util.IamVerifyServiceUtil;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

import com.digiwin.metadatacache.dao.EocIntgMappingDao;
import com.digiwin.metadatacache.dao.PrdDatakeyMappingDao;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.enums.EocLevelEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.EocIntgMapping;
import com.digiwin.metadatacache.model.PrdDatakeyMapping;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 取得租戶服務分派資訊 - A<br>
 * Path:[/restful/standard/mdc/TenantProductInfo/Get]
 */
@org.springframework.stereotype.Service
public class TenantProductInfoGetService extends AbstractApiService {

	@Autowired
	private ProductDao productDao;

	@Autowired
	private EocIntgMappingDao eocIntgMappingDao;

	@Autowired
	private PrdDatakeyMappingDao prdDatakeyMappingDao;

	public TenantProductInfoGetService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.TENANT_PRD_INFO_GET_SCHEMA;
	}

	/**
	 *  查询租户对应的服务分派资讯
	 * @param validatorResult      * @param validatorResult
	 * @return Map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
				+ ", 原始訊息: " + tRequestJsonNode);
		
		// 取得租戶ID、產品、運營單元
		String tTenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
		String tProdName = tRequestJsonNode.get(MdcConstant.PROD_NAME).asText();

		//获取当前租户的版本
		String version = IamVerifyServiceUtil.getCurrentTenantVersion(tTenantId);
		//老版本处理
		if(MdcConstant.TENANT_VERSION_V1.equalsIgnoreCase(version)){
			return getTenantProductInfo(validatorResult, tTenantId, tProdName);
		}else {
			return getTenantProductInfoV2(validatorResult, tTenantId, tProdName);
		}
	}

	private Map<String, Object> getTenantProductInfo(ValidatorResult validatorResult, String tTenantId, String tProdName) throws IOException {
		// 建立data節點
		ObjectNode tDataNode = mapper.createObjectNode();

		// 組CacheKey+'/'+tenant_id
		// 细化粒度 在原有的基础上加上租户作为主key 产品名称为副key
		String tCacheMapKey = CacheMapTypeEnum.tenant_product_dispatch_cachemap.getCode() + MdcSymbolConstant.COLON + tTenantId;
		// 進元數據快取中找出對應資料
		String tDataFromCache = cacheService.getCacheMapValue(tCacheMapKey, tProdName);
		// 有對應資料直接回傳
		if (null != tDataFromCache) {
			log.info("命中快取: " + tCacheMapKey + "// " + tProdName);
			return converJsonStringToMap(tDataFromCache);
		}
		// 根據入參取得對應的服務分派資訊
		Map<String, String> tProductCondition = new HashMap<>();
		tProductCondition.put(MdcConstant.NAME, tProdName);
		List<Product> tProducts = productDao.fetch(tProductCondition);
		if (!tProducts.isEmpty()) {
			// 新 只要產品是敏態一律回固定訊息
			if (tProducts.get(0).getProductType().equals(ApiTypeEnum.agile.toString())) {
				tDataNode = (ObjectNode) mapper.readTree("{ \"tenant_id\": \"" + tTenantId
						+ "\", \"is_enterprise\": true, \"dispatch\": [ { \"service_product\": { \"name\": \""
						+ tProducts.get(0).getName() + "\", \"uid\": \"\" } } ] }");
			} else {
				// 目標資料表：eoc_intg_mapping
				Map<String, String> pEocIntgMapCondition = new HashMap<>();
				// 規則:找出符合tenant_id, product name的eoc_intg_mapping資料列
				pEocIntgMapCondition.put(MdcConstant.TENANT_ID, tTenantId);
				pEocIntgMapCondition.put(MdcConstant.PRODUCT_NAME, tProdName);
				List<EocIntgMapping> eocIntgMappings = eocIntgMappingDao.fetch(pEocIntgMapCondition);
				// 先將eoc intg mappings轉換成map 租户Id|产品名称|产品uid
				Map<String, Map<String, Map<String, List<EocIntgMapping>>>> tTotalMap
						= groupEocMappingByTenantAndProdNameAndUid(eocIntgMappings, tTenantId);
				// 再把eoc intg mappings的map轉成json node
				// 填上tenant_id
				tDataNode.put(MdcConstant.TENANT_ID, tTenantId);
				// is_enterprise預設為false
				boolean isEnterprise = false;
				// 建立dispatch陣列
				ArrayNode tDispatchAryNode = mapper.createArrayNode();
				tDataNode.set(MdcConstant.DISPATCH, tDispatchAryNode);
				// 決定dispatch深度 决定返回的层级关系
				int dispatchLevel = 0;
				Map<String, String> tCondition = new HashMap<>();
				tCondition.put(MdcConstant.PROD_NAME, tProdName);
				List<PrdDatakeyMapping> tDatakeyMappings = prdDatakeyMappingDao.fetch(tCondition);
				if (CollectionUtils.isNotEmpty(tDatakeyMappings)) {
					for (PrdDatakeyMapping tDatakeyMapping : tDatakeyMappings) {
						if (tDatakeyMapping.getEocLevel().equals(EocLevelEnum.site_id.toString())) {
							dispatchLevel = 1;
						} else if (tDatakeyMapping.getEocLevel().equals(EocLevelEnum.region_id.toString())) {
							dispatchLevel = 2;
						}
					}
				}
				// 如果沒有找到任何對應關係,則"dispatch":[]
				if (!eocIntgMappings.isEmpty()) {
					if (tTotalMap.get(tTenantId).get(tProdName) != null) {
						for (String key : tTotalMap.get(tTenantId).get(tProdName).keySet()) {
							// 每台主機(UID)
							boolean tMappingsNoDatakeyValue = true;
							// 每台主機(UID)上的每個運營單元對應
							for (EocIntgMapping mapping : tTotalMap.get(tTenantId).get(tProdName).get(key)) {
								// 該租戶有eoc_level為enterprise_id的mapping且其EOC代號有值，則is_enterprise=true
								if (mapping.getEocLevel().equals(MdcConstant.ENTERPRISE_ID)
										&& !StringUtil.isEmptyOrSpace(mapping.getEocId())) {
									isEnterprise = true;
								}
								ObjectNode tDispatchItemNode = mapper.createObjectNode();
								if (!StringUtil.isEmptyOrSpace(mapping.getDatakey())) {
									// 填上service_product物件
									ObjectNode tServiceProductNode = mapper.createObjectNode();
									tDispatchItemNode.set(MdcConstant.SERVICE_PRODUCT, tServiceProductNode);
									tServiceProductNode.put(MdcConstant.NAME, mapping.getProducName());
									tServiceProductNode.put(MdcConstant.UID, mapping.getProductUid());
									ObjectNode tOperationManagerNode = mapper.createObjectNode();
									if (mapping.getEocLevel().equals(EocLevelEnum.company_id.toString())
											&& dispatchLevel == 0) {
										// 同一個uid下，有datakey的eoc_intg_mapping才能dispatch
										tDispatchAryNode.add(tDispatchItemNode);
										tOperationManagerNode.put(MdcConstant.EOC_COMPANY_ID, mapping.getEocId());
									} else if (mapping.getEocLevel().equals(EocLevelEnum.site_id.toString())
											&& dispatchLevel == 1) {
										// 同一個uid下，有datakey的eoc_intg_mapping才能dispatch
										tDispatchAryNode.add(tDispatchItemNode);
										if (mapping.getEocIntgMapping() != null) {
											tOperationManagerNode.put(MdcConstant.EOC_COMPANY_ID,
													mapping.getEocIntgMapping().getEocId());
										}
										tOperationManagerNode.put(MdcConstant.EOC_SITE_ID, mapping.getEocId());
									} else if (mapping.getEocLevel().equals(EocLevelEnum.region_id.toString())
											&& dispatchLevel == 2) {
										// 同一個uid下，有datakey的eoc_intg_mapping才能dispatch
										tDispatchAryNode.add(tDispatchItemNode);

										if (mapping.getEocIntgMapping() != null) {
											if (mapping.getEocIntgMapping().getEocIntgMapping() != null) {
												tOperationManagerNode.put(MdcConstant.EOC_COMPANY_ID,
														mapping.getEocIntgMapping().getEocIntgMapping().getEocId());
											}
											tOperationManagerNode.put(MdcConstant.EOC_SITE_ID,
													mapping.getEocIntgMapping().getEocId());
										}
										tOperationManagerNode.put(MdcConstant.EOC_REGION_ID, mapping.getEocId());
									}
									tDispatchItemNode.set(MdcConstant.OPERATION_MANAGER, tOperationManagerNode);
									tMappingsNoDatakeyValue = false;
								}
							}
							// 如果都沒有datakey則dispatch此prod uid一次即可
							if (tMappingsNoDatakeyValue) {
								// 填上service_product物件
								ObjectNode tDispatchItemNode = mapper.createObjectNode();
								ObjectNode tServiceProductNode = mapper.createObjectNode();
								tDispatchItemNode.set(MdcConstant.SERVICE_PRODUCT, tServiceProductNode);
								tServiceProductNode.put(MdcConstant.NAME,
										tTotalMap.get(tTenantId).get(tProdName).get(key).get(0).getProducName());
								tServiceProductNode.put(MdcConstant.UID,
										tTotalMap.get(tTenantId).get(tProdName).get(key).get(0).getProductUid());
								tDispatchAryNode.add(tDispatchItemNode);
							}
						}
					}
				}
				// 填上is_enterprise
				tDataNode.put(MdcConstant.IS_ENTERPRISE, isEnterprise);
			}
		}
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);
		if (tStateCode.getCode().equals("000") && null != tResponseJsonNode.get(MdcConstant.DATA)
				&& null != tResponseJsonNode.get(MdcConstant.DATA).get(MdcConstant.DISPATCH)
				&& tResponseJsonNode.get(MdcConstant.DATA).get(MdcConstant.DISPATCH).size() >0) {
			cacheService.setCacheMapValue(tCacheMapKey, tProdName, tResponseJsonNode.toString());
		}
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * v2 版本
	 */
	private Map<String, Object> getTenantProductInfoV2(ValidatorResult validatorResult, String tTenantId, String tProdName) throws IOException {
		// 建立data節點
		ObjectNode tDataNode = mapper.createObjectNode();

		// 根據入參取得對應的服務分派資訊
		Map<String, String> tProductCondition = new HashMap<>();
		tProductCondition.put(MdcConstant.NAME, tProdName);
		List<Product> tProducts = productDao.fetch(tProductCondition);
		if (!tProducts.isEmpty()) {
			// 新 只要產品是敏態一律回固定訊息
			if (tProducts.get(0).getProductType().equals(ApiTypeEnum.agile.toString())) {
				tDataNode = (ObjectNode) mapper.readTree("{ \"tenant_id\": \"" + tTenantId
						+ "\", \"is_enterprise\": true, \"dispatch\": [ { \"service_product\": { \"name\": \""
						+ tProducts.get(0).getName() + "\", \"uid\": \"\" } } ] }");
			} else {
				// 目標資料表：eoc_intg_mapping
				Map<String, String> pEocIntgMapCondition = new HashMap<>();
				// 規則:找出符合tenant_id, product name的eoc_intg_mapping v2資料列
				pEocIntgMapCondition.put(MdcConstant.TENANT_ID, tTenantId);
				pEocIntgMapCondition.put(MdcConstant.PRODUCT_NAME, tProdName);
				pEocIntgMapCondition.put(MdcConstant.EOC_LEVEL, MdcConstant.CLOUD_MAPPING_ID);

				List<EocIntgMapping> eocIntgMappings = eocIntgMappingDao.fetchGroupByEocId(pEocIntgMapCondition);
				// 先將eoc intg mappings轉換成map 租户Id|产品名称|产品uid
				Map<String, Map<String, Map<String, List<EocIntgMapping>>>> tTotalMap
						= groupEocMappingByTenantAndProdNameAndUid(eocIntgMappings, tTenantId);
				// 再把eoc intg mappings的map轉成json node
				// 填上tenant_id
				tDataNode.put(MdcConstant.TENANT_ID, tTenantId);
				// is_enterprise預設為false
				boolean isEnterprise = false;
				// 建立dispatch陣列
				ArrayNode tDispatchAryNode = mapper.createArrayNode();
				tDataNode.set(MdcConstant.DISPATCH, tDispatchAryNode);
				// 決定dispatch深度 决定返回的层级关系
				int dispatchLevel = 0;
				Map<String, String> tCondition = new HashMap<>();
				tCondition.put(MdcConstant.PROD_NAME, tProdName);
				List<PrdDatakeyMapping> tDatakeyMappings = prdDatakeyMappingDao.fetch(tCondition);
				if (CollectionUtils.isNotEmpty(tDatakeyMappings)) {
					for (PrdDatakeyMapping tDatakeyMapping : tDatakeyMappings) {
						if (tDatakeyMapping.getEocLevel().equals(EocLevelEnum.site_id.toString())) {
							dispatchLevel = 1;
						} else if (tDatakeyMapping.getEocLevel().equals(EocLevelEnum.region_id.toString())) {
							dispatchLevel = 2;
						}
					}
				}
				// 如果沒有找到任何對應關係,則"dispatch":[]
				if (!eocIntgMappings.isEmpty()) {
					if (tTotalMap.get(tTenantId).get(tProdName) != null) {
						for (String key : tTotalMap.get(tTenantId).get(tProdName).keySet()) {
							// 每台主機(UID)
							boolean tMappingsNoDatakeyValue = true;
							// 每台主機(UID)上的每個運營單元對應
							for (EocIntgMapping mapping : tTotalMap.get(tTenantId).get(tProdName).get(key)) {
								// 該租戶有eoc_level為enterprise_id的mapping且其EOC代號有值，則is_enterprise=true
								if (mapping.getCloudMappingType().equals(MdcConstant.ENTERPRISE_ID)
										&& !StringUtil.isEmptyOrSpace(mapping.getEocId())) {
									isEnterprise = true;
								}
								ObjectNode tDispatchItemNode = mapper.createObjectNode();
								if (!StringUtil.isEmptyOrSpace(mapping.getDatakey())) {
									// 填上service_product物件
									ObjectNode tServiceProductNode = mapper.createObjectNode();
									tDispatchItemNode.set(MdcConstant.SERVICE_PRODUCT, tServiceProductNode);
									tServiceProductNode.put(MdcConstant.NAME, mapping.getProducName());
									tServiceProductNode.put(MdcConstant.UID, mapping.getProductUid());
									ObjectNode tOperationManagerNode = mapper.createObjectNode();
									if (mapping.getCloudMappingType().equals(EocLevelEnum.company_id.toString())
											&& dispatchLevel == 0) {
										// 同一個uid下，有datakey的eoc_intg_mapping才能dispatch
										tDispatchAryNode.add(tDispatchItemNode);
										tOperationManagerNode.put(MdcConstant.EOC_MAPPING_ID, mapping.getEocId());
									} else if (mapping.getCloudMappingType().equals(EocLevelEnum.site_id.toString())
											&& dispatchLevel == 1) {
										// 同一個uid下，有datakey的eoc_intg_mapping才能dispatch
										tDispatchAryNode.add(tDispatchItemNode);
										tOperationManagerNode.put(MdcConstant.EOC_MAPPING_ID, mapping.getEocId());
									} else if (mapping.getCloudMappingType().equals(EocLevelEnum.region_id.toString())
											&& dispatchLevel == 2) {
										// 同一個uid下，有datakey的eoc_intg_mapping才能dispatch
										tDispatchAryNode.add(tDispatchItemNode);
										tOperationManagerNode.put(MdcConstant.EOC_MAPPING_ID, mapping.getEocId());
									}
									tDispatchItemNode.set(MdcConstant.OPERATION_MANAGER, tOperationManagerNode);
									tMappingsNoDatakeyValue = false;
								}
							}
							// 如果都沒有datakey則dispatch此prod uid一次即可
							if (tMappingsNoDatakeyValue) {
								// 填上service_product物件
								ObjectNode tDispatchItemNode = mapper.createObjectNode();
								ObjectNode tServiceProductNode = mapper.createObjectNode();
								tDispatchItemNode.set(MdcConstant.SERVICE_PRODUCT, tServiceProductNode);
								tServiceProductNode.put(MdcConstant.NAME,
										tTotalMap.get(tTenantId).get(tProdName).get(key).get(0).getProducName());
								tServiceProductNode.put(MdcConstant.UID,
										tTotalMap.get(tTenantId).get(tProdName).get(key).get(0).getProductUid());
								tDispatchAryNode.add(tDispatchItemNode);
							}
						}
					}
				}
				// 填上is_enterprise
				tDataNode.put(MdcConstant.IS_ENTERPRISE, isEnterprise);
			}
		}
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tDataNode);

		return converJsonNodeToMap(tResponseJsonNode);
	}
}