package com.digiwin.metadatacache.apiservice;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.annotate.AppTokenVerify;
import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import com.digiwin.metadatacache.util.StringUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import com.digiwin.metadatacache.dao.EaiTenantMappingDao;
import com.digiwin.metadatacache.dao.EocIntgMappingDao;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.EaiTenantMapping;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.services.MdcCacheService;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * 雲中台整合設定同步產品清單
 * Path:[/restful/standard/mdc/TenantProductList/Update]
 */
@org.springframework.stereotype.Service
@AppTokenVerify
public class TenantProductListUpdateService extends AbstractApiService {

	@Autowired
	private ProductDao productDao;

	@Autowired
	private EaiTenantMappingDao eaiTenantMappingDao;

	@Autowired
	private EocIntgMappingDao eocIntgMappingDao;

	@Autowired
	private MdcCacheService cacheService;

	@Autowired
    private TenantProductUpdateProcessService tenantProductUpdateProcessService;

	public TenantProductListUpdateService() {
		super();
		jsonSchemaFileName = JsonSchemaFileConstant.TENANT_PRD_LIST_UPDATE_SCHEMA;
	}

	/**
	 * 雲中台整合設定同步產品清單 逻辑处理
	 * 更新1：修改对应的快取结构
	 * @param validatorResult      * @param validatorResult
	 * @return Map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		// 雲端saas應用更新
		String tKey = CacheMapTypeEnum.tenant_product_list_update_cachemap.getCode() + MdcSymbolConstant.COLON + MdcConstant.ESP;
		try {
			String tEaiUid = null, tAction = null;
			if (tRequestJsonNode.get(MdcConstant.EAI_UID) != null
					&& tRequestJsonNode.get(MdcConstant.EAI_UID).asText().length() != 0) {
				tEaiUid = tRequestJsonNode.get(MdcConstant.EAI_UID).asText();
				// 地端產品更新
				tKey = CacheMapTypeEnum.tenant_product_list_update_cachemap.getCode() + MdcSymbolConstant.COLON + tEaiUid;
			}
			if (tRequestJsonNode.get(MdcConstant.ACTION) != null
					&& tRequestJsonNode.get(MdcConstant.ACTION).asText().length() != 0) {
				tAction = tRequestJsonNode.get(MdcConstant.ACTION).asText();
			}
			if (tAction == null || tAction.matches(MdcConstant.UPDATE)) {
				updateProductAndServiceList(tRequestJsonNode, requestMsg, tKey, tEaiUid);
			} else if (tAction.matches(MdcConstant.REMOVE)) {
				removeEaiInfo(tEaiUid, tKey);
			}
		} finally {
			cacheService.unLock(tKey + MdcConstant.CACHE);
		}
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = getDescriptionByLocale(tStateCode, locale);
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 刪除地中台所有相關資訊
	 * @param pEaiUid 地中台ID
	 * @param pKey 快取主键
	 */
	private void removeEaiInfo(String pEaiUid, String pKey) {
		// 1.刪除eai_tenant_mapping
		eaiTenantMappingDao.removeByEaiUid(pEaiUid);
		// 2.刪除product(會同時刪除其服務、運營配置、以及快取),取得該eai_uid的產品清單(tProductList)
		Map<String, String> tProductCondition = new HashMap<>();
		tProductCondition.put(MdcConstant.EAI_UID, pEaiUid);
		// MDC中該地中台的所有產品 (tProductList)
		List<Product> tProducts = productDao.fetch(tProductCondition);
        tenantProductUpdateProcessService.removeProductList(tProducts);
		// 清掉redis裡面的最後更新紀錄,取得最後一次的更新訊息
//		cacheService.remove(pKey);
		log.info("刪掉快取中的最新同步產品服務訊息紀錄 key: " + pKey);
	}

	/**
	 * 更新產品清單product資料表、更新產品服務清單service資料表
	 * @param pNode 节点
	 * @param pMessage 消息
	 * @param pKey  主键
	 * @param pEaiUid  地中台Id
	 * @throws InterruptedException  InterruptedException
	 */
	private void updateProductAndServiceList(JsonNode pNode, String pMessage, String pKey, String pEaiUid)
			throws InterruptedException {
		// 需要更新的产品
		List<Product> tListToUpdate = new ArrayList<>();
		// 需要删除的产品
		List<Product> tListToRemove = new ArrayList<>();
		String tGatewayId = null;
		if (pNode.get(MdcConstant.GATEWAY_ID) != null && pNode.get(MdcConstant.GATEWAY_ID).asText().length() != 0) {
			tGatewayId = pNode.get(MdcConstant.GATEWAY_ID).asText();
		}
		if (!cacheService.lock(pKey + MdcConstant.CACHE)) {
			return;
		}
		// 取得最後一次的更新訊息
//		String tLastUpdateMessage = (String) cacheService.get(pKey);
//		if (StringUtils.isNotBlank(tLastUpdateMessage)) {
//			log.info("命中快取: " + pKey);
//			// 如果更新訊息同最後一次紀錄則直接回傳成功，不用進行更新
//			if (tLastUpdateMessage.equals(pMessage)) {
//				log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "產品服務同步訊息與快取最後一次的紀錄相同，不執行更新。");
//				return;
//			}
//		}
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "產品服務同步訊息與快取最後一次的紀錄不同，執行更新。");
		// 租户列表
		ArrayNode tTenantList = (ArrayNode) pNode.get(MdcConstant.TENANT);
		// 訊息的產品清單(tProductAryNode)
		ArrayNode tProductAryNode = (ArrayNode) pNode.get(MdcConstant.PRODUCT);
		if (tProductAryNode.size() == 0) {
			log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "同步產品清單為空，不執行更新(EAI UID = " + pEaiUid + ")：");
			return;
		}
		Map<String, String> tProductCondition = new HashMap<>();
		tProductCondition.put(MdcConstant.EAI_UID, pEaiUid);
		// 获取敏态产品或者是对应地中态产品清单
		List<Product> tProducts = productDao.fetch(tProductCondition);
		for (Product product : tProducts) {
			log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "eai_uid = " + product.getEaiUid()
					+ ", tenant_id = " + product.getTenantId() + ", name = " + product.getName() + ", uid = "
					+ product.getUid());
		}
		// 把所有產品灌到要刪除的產品清單中tListToRemove
		boolean b = tListToRemove.addAll(tProducts);
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "開始執行產品更新(EAI UID = " + pEaiUid + ")：" + b);
		// 從訊息中的第一個產品開始，與MDC該地中台所有產品比對
		// 为什么产品可以采用更新的方式，而服务不采用此方式？ 是由于产品的个数远小于服务个数
		for (JsonNode tNode : tProductAryNode) {
			boolean productExist = false;
			if (tProductAryNode.isArray()) {
				for (Product tProduct : tProducts) {
					// 采用name, uid做判定
					if (tProduct.getName().equals(tNode.get(MdcConstant.NAME).asText())
							&& tProduct.getUid().equals(tNode.get(MdcConstant.UID).asText())) {
						tProduct.setVersion(tNode.get(MdcConstant.VERSION).asText());
						tProduct.setIp(tNode.get(MdcConstant.IP).asText());
						tProduct.setApid(tNode.get(MdcConstant.ID).asText());
						if (tNode.get(MdcConstant.URL) != null) {
							tProduct.setUrl(tNode.get(MdcConstant.URL).asText());
						}
						if (!StringUtil.isEmptyOrSpace(tNode.get(MdcConstant.SETTING_EOC_FLAG))) {
							tProduct.setSettingEocFlag(tNode.get(MdcConstant.SETTING_EOC_FLAG).asInt());
						}
						if (!StringUtil.isEmptyOrSpace(tNode.get(MdcConstant.APP_ID))) {
							tProduct.setAppId(tNode.get(MdcConstant.APP_ID).asText());
						}
						tProduct.setGatewayId(tGatewayId);
						tProduct.setLastUpdateTime(Calendar.getInstance());
						productExist = true;
						tListToUpdate.add(tProduct);
						tListToRemove.remove(tProduct);
					}
				}
				// MDC該地中台所有產品沒有與node相同者，則新增一筆產品
				if (!productExist) {
					Product tProduct = new Product();
					tProduct.setIp(tNode.get(MdcConstant.IP).asText());
					tProduct.setApid(tNode.get(MdcConstant.APID).asText());
					tProduct.setUid(tNode.get(MdcConstant.UID).asText());
					tProduct.setGatewayId(tGatewayId);
					tProduct.setName(tNode.get(MdcConstant.NAME).asText());
					tProduct.setVersion(tNode.get(MdcConstant.VERSION).asText());
					if (tNode.get(MdcConstant.URL) != null) {
						tProduct.setUrl(tNode.get(MdcConstant.URL).asText());
					}
					if (!StringUtil.isEmptyOrSpace(tNode.get(MdcConstant.SETTING_EOC_FLAG))) {
						tProduct.setSettingEocFlag(tNode.get(MdcConstant.SETTING_EOC_FLAG).asInt());
					}
					if (!StringUtil.isEmptyOrSpace(tNode.get(MdcConstant.APP_ID))) {
						tProduct.setAppId(tNode.get(MdcConstant.APP_ID).asText());
					}
					tProduct.setBuildTime(Calendar.getInstance());
					tProduct.setEaiUid(pEaiUid);
					if (pEaiUid == null) {
						tProduct.setProductType(ApiTypeEnum.agile.toString());
					} else {
						tProduct.setProductType(ApiTypeEnum.standard.toString());
					}
					tListToUpdate.add(tProduct);
				}
			}
		}
        tenantProductUpdateProcessService.removeProductList(tListToRemove);
		// 获取地中台对应的租户
		List<String> tCurrentTenantIdList = new ArrayList<>();
		if (StringUtils.isNotBlank(pEaiUid)) {
			Map<String, String> tConditionMap = new HashMap<>();
			tConditionMap.put(MdcConstant.EAI_UID, pEaiUid);
			List<EaiTenantMapping> eaiTenantMappings = eaiTenantMappingDao.fetch(tConditionMap);
			for (EaiTenantMapping tenantMapping : eaiTenantMappings) {
				tCurrentTenantIdList.add(tenantMapping.getTenantId());
			}
		}
		// 更新产品信息
		productDao.updateList(tCurrentTenantIdList,tListToUpdate);
		// 更新服务信息
		updateServiceList(tProductAryNode, tListToUpdate);
		// 更新eai_tenant_mapping
		if (pEaiUid != null && pEaiUid.length() != 0) {
			List<String> tUpdateTenantIdList = new ArrayList<>();
			// 將此eai_uid的eai_tenant_mapping刪除
			eaiTenantMappingDao.removeByEaiUid(pEaiUid);
			// 新增eai_tenant_mapping
			for (JsonNode tNode : tTenantList) {
				tUpdateTenantIdList.add(tNode.textValue());
				EaiTenantMapping tEaiTenantMapping = new EaiTenantMapping();
				tEaiTenantMapping.setTenantId(tNode.textValue());
				tEaiTenantMapping.setEaUid(pEaiUid);
				tEaiTenantMapping.setBuildTime(Calendar.getInstance());
				eaiTenantMappingDao.save(tEaiTenantMapping);
			}
			List<String> tTenantIdDifferenceList
					= (List<String>) CollectionUtils.disjunction(tCurrentTenantIdList,tUpdateTenantIdList);
			for (String tenantId : tTenantIdDifferenceList) {
				String tKey = CacheMapTypeEnum.tenant_product_mapping_list.getCode() + MdcSymbolConstant.COLON + tenantId;
				cacheService.remove(tKey);
				log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "清除快取(租戶的整個產品清單)：" + "key = " + tKey);
				log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "清除快取(租戶的整個產品清單)：" + "tenant = "
						+ tenantId);
				// 被刪除掉的租戶，要把其對應的運營配置給刪除
				if (tCurrentTenantIdList.contains(tenantId)) {
					eocIntgMappingDao.deleteMappingByTenantId(tenantId);
					// 用租戶ID找出綁定的產品主機
					Map<String, String> tCondition = new HashMap<>();
					tCondition.put(MdcConstant.TENANT, tenantId);
					List<Product> tProductList = productDao.fetch(tCondition);
					// 則將該產品主機的tenant_id設為空字串
					if (tProductList != null && tProductList.size() != 0) {
						for (Product tProduct : tProductList) {
							tProduct.setTenantId("");
							productDao.save(tProduct);
						}
					}
				}
			}
		}
		// 更新成功，update該eai_uid最後的更新訊息紀錄
//		cacheService.set(pKey, pMessage);
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "完成產品服務更新(EAI UID = " + pEaiUid + ")");
	}

	/**
	 * 新增方法 用于调用下面服务 增加事务处理机制
	 * 解决原有删除服务与新增服务之间，原有存在的服务查询不到的问题处理
	 * 保证删除与新增在一个事务中,大大减少数据不存在风险
	 * @param pProductAryNode pProductAryNode
	 * @param pListToUpdate pListToUpdate
	 */
	private void updateServiceList(ArrayNode pProductAryNode, List<Product> pListToUpdate) {
		// pListToUpdate：要新增的產品or資料庫中要更新服務的產品
		if (pProductAryNode.isArray()) {
			log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "開始執行服務更新：");
			for (JsonNode tProductNode : pProductAryNode) {
				tenantProductUpdateProcessService.updateProductService(tProductNode, pListToUpdate);
			}
		}
	}
}
