package com.digiwin.metadatacache.apiservice;

import com.digiwin.metadatacache.constant.JsonSchemaFileConstant;
import com.digiwin.metadatacache.constant.MdcConstant;
import com.digiwin.metadatacache.dao.EocIntgMappingDao;
import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.enums.ApiTypeEnum;
import com.digiwin.metadatacache.enums.EocLevelEnum;
import com.digiwin.metadatacache.enums.ValidateStateEnum;
import com.digiwin.metadatacache.exception.BaseException;
import com.digiwin.metadatacache.model.EocIntgMapping;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.model.StateCode;
import com.digiwin.metadatacache.util.EocInfoServiceUtil;
import com.digiwin.metadatacache.util.IntgInfoServiceUtil;
import com.digiwin.metadatacache.util.StringUtil;
import com.digiwin.metadatacache.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 取得租戶產品運營清單(含運營階層關係)(ESP-SDK未透传token) v2逻辑,返回云端和地端数据
 * Path:[/restful/standard/mdc/TenantProductOperationListV2/Get]
 */
@org.springframework.stereotype.Service
public class TenantProductOperationListV2GetService extends AbstractApiService {

    @Autowired
    private EocIntgMappingDao eocIntgMappingDao;

    @Autowired
    private ProductDao productDao;

    public TenantProductOperationListV2GetService() {
        super();
        jsonSchemaFileName = JsonSchemaFileConstant.TENANT_PRD_OPERATION_LIST_V2_GET_SCHEMA;
    }

    /**
     * 查询产品运营单元清单
     *
     * @param validatorResult * @param validatorResult
     * @return Map
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        // 取出入參tenant, api_name
        String tTenantId = tRequestJsonNode.get(MdcConstant.TENANT_ID).asText();
        String tUid = null;
        String tId = null;
        if (tRequestJsonNode.get(MdcConstant.UID) != null
                && !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.UID).asText())) {
            tUid = tRequestJsonNode.get(MdcConstant.UID).asText();
        }
        if (tRequestJsonNode.get(MdcConstant.ID) != null
                && !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.ID).asText())) {
            tId = tRequestJsonNode.get(MdcConstant.ID).asText();
        }
        if (tRequestJsonNode.get(MdcConstant.PROD_NAME) != null
                && !StringUtil.isEmptyOrSpace(tRequestJsonNode.get(MdcConstant.PROD_NAME).asText())) {
            String tProdName = tRequestJsonNode.get(MdcConstant.PROD_NAME).asText();

            return processTenantProdOperationListGetV2(tTenantId, tProdName, tUid, tId);
        } else {
            // 如果有註明需要"敘述" "name_include": true 則先取EOC運營單元碼清單
            boolean tNameInclude = false;
            if (tRequestJsonNode.get(MdcConstant.NAME_INCLUDE) != null) {
                tNameInclude = tRequestJsonNode.get(MdcConstant.NAME_INCLUDE).asBoolean();
            }

            return processTenantOperationListGetV2(tTenantId, tNameInclude);

        }
    }



    /**
     * 取得租戶所有產品的運營清單(蔣穎EOC配置畫面呼叫) v2版本
     */
    private Map<String, Object> processTenantOperationListGetV2(String pTenantId, Boolean pNameInclude) {
        //根據入參tenant_id、eocLevel取得eoc_intg_mapping v2 資料
        Map<String, String> pCondition = new HashMap<>();
        pCondition.put(MdcConstant.TENANT_ID, pTenantId);
        pCondition.put(MdcConstant.EOC_LEVEL, MdcConstant.CLOUD_MAPPING_ID);
        List<EocIntgMapping> tMappings = eocIntgMappingDao.fetch(pCondition);

        Map<String, String> tEocNode = new HashMap<>();
        // NameInclude=true，代表回傳訊息中要包含運營單元的描述(name)，須向EOC取得
        // 若該租戶還未配置運營，則不用取得token呼叫EOC
        if (pNameInclude && !tMappings.isEmpty()) {
            try {
                // 根據租戶ID取得EOC運營單元碼 v2版本接口
                tEocNode = EocInfoServiceUtil.getEocInfoV2MapByTenantId(pTenantId);
                // 取得eoc清單時發生任何錯誤都繼續執行，因為取得name值(敘述)是額外的資訊
                // 不要因為取得name值(敘述)失敗，而影響取得運營大表
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }

        // 將tMappings依照產品名稱分出來
        Map<String, List<EocIntgMapping>> tMappingsGroupByProductName = new HashMap<>();
        for (EocIntgMapping mapping : tMappings) {
            if (tMappingsGroupByProductName.get(mapping.getProducName()) == null) {
                List<EocIntgMapping> tProductNameMappingList = new ArrayList<>();
                tProductNameMappingList.add(mapping);
                tMappingsGroupByProductName.put(mapping.getProducName(), tProductNameMappingList);
            } else {
                tMappingsGroupByProductName.get(mapping.getProducName()).add(mapping);
            }
        }
        ArrayNode tProdEocMappingAryNode = mapper.createArrayNode();
        for (String tProductName : tMappingsGroupByProductName.keySet()) {
            JsonNode tIntgInfoNode = null;
            // 如果有註明需要"敘述" "name_include": true 則根據產品UID取得ESP單元對應碼
            try {
                if (pNameInclude) {
                     //取得ESP單元對應碼
                    // 根據tProdName, tUid呼叫ESP->ERP取得集成代碼
                    tIntgInfoNode = IntgInfoServiceUtil.getIntgInfo(null, pTenantId, tProductName,
                            tMappingsGroupByProductName.get(tProductName).get(0).getProductUid());
                }
                // 取得地端om時發生任何錯誤也略過 因為取得name_include是額外的功能
                // 不要因為取得name_include失敗而影響取得運營大表
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }

            // 個別產品名稱的運營物件
            ObjectNode tProdEocMappingObjectNode = mapper.createObjectNode();
            if (pNameInclude) {
                Map<String, String> tPrdCondition = new HashMap<>();
                tPrdCondition.put(MdcConstant.NAME, tProductName);
                tPrdCondition.put(MdcConstant.UID, tMappingsGroupByProductName.get(tProductName).get(0).getProductUid());
                List<Product> tProducts = productDao.fetch(tPrdCondition);
                String tEocSetting = "";
                String tEocSettingRegion = "";
                if (tProducts != null && !tProducts.isEmpty() && tProducts.get(0).getEocSetting() != null) {
                    tEocSetting = tProducts.get(0).getEocSetting();
                    tEocSettingRegion = tProducts.get(0).getEocSettingRegion();
                }
                tProdEocMappingObjectNode.put(MdcConstant.EOC_SETTING, tEocSetting);
                tProdEocMappingObjectNode.put(MdcConstant.EOC_SETTING_REGION, tEocSettingRegion);
            }
            tProdEocMappingObjectNode.put(MdcConstant.PROD_NAME, tProductName);
            tProdEocMappingObjectNode.put(MdcConstant.UID, tMappingsGroupByProductName.get(tProductName).get(0).getProductUid());

            //組成回傳json
            ArrayNode orgTypeMapping = mapper.createArrayNode();
            tProdEocMappingObjectNode.set(MdcConstant.ORG_TYPE_MAPPING, orgTypeMapping);

            for (EocIntgMapping mapping : tMappingsGroupByProductName.get(tProductName)) {
                ObjectNode orgTypeMappingNode = mapper.createObjectNode();
                orgTypeMappingNode.put(MdcConstant.EOC_MAPPING_ID, mapping.getEocId());
                orgTypeMappingNode.put(MdcConstant.OM_MAPPING_ID, mapping.getIntgId());

                //赋值eoc companyName
                if (pNameInclude && tEocNode.get(mapping.getEocId()) !=  null) {
                    orgTypeMappingNode.put(MdcConstant.EOC_MAPPING_NAME, tEocNode.get(mapping.getEocId()));
                }

                //type
                if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.enterprise_id.toString())) {
                    orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.ENTERPRISE_ID);
                }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.company_id.toString())){
                    orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.COMPANY_ID);
                    if (tIntgInfoNode != null) {
                        ArrayNode tIntgCompanyAryNode = (ArrayNode) tIntgInfoNode.get(MdcConstant.ORG_TYPE_COMPANY);
                        for (JsonNode tIntgCompanyObject : tIntgCompanyAryNode) {
                            if (mapping.getIntgId().equals(tIntgCompanyObject.get(MdcConstant.COMPANY_NO).asText())) {
                                orgTypeMappingNode.set(MdcConstant.OM_MAPPING_NAME, tIntgCompanyObject.get(MdcConstant.COMPANY_NAME));
                            }
                        }
                    }
                }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.site_id.toString())){
                    orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.SITE_ID);
                    if (tIntgInfoNode != null) {
                        ArrayNode tIntgCompanyAryNode = (ArrayNode) tIntgInfoNode.get(MdcConstant.ORG_TYPE_COMPANY);
                        for (JsonNode tIntgCompanyObject : tIntgCompanyAryNode) {
                                if (tIntgCompanyObject.get(MdcConstant.ORG_TYPE_SITE) != null) {
                                    for (JsonNode tIntgSiteObject : tIntgCompanyObject.get(MdcConstant.ORG_TYPE_SITE)) {
                                        if (mapping.getIntgId().equals(tIntgSiteObject.get(MdcConstant.SITE_NO).asText())) {
                                            orgTypeMappingNode.set(MdcConstant.OM_MAPPING_NAME, tIntgSiteObject.get(MdcConstant.SITE_NAME));
                                        }
                                    }
                                }
                            }
                        }
                }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.region_id.toString())){
                    orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.REGION_ID);
                    if (tIntgInfoNode != null) {
                        ArrayNode tIntgRegionAryNode = (ArrayNode) tIntgInfoNode.get(MdcConstant.ORG_TYPE_REGION);

                        if (tIntgRegionAryNode != null) {
                            for (JsonNode tIntgRegionObject : tIntgRegionAryNode) {
                                if (mapping.getIntgId().equals(tIntgRegionObject.get(MdcConstant.REGION_NO).asText())) {
                                    orgTypeMappingNode.set(MdcConstant.OM_MAPPING_NAME, tIntgRegionObject.get(MdcConstant.REGION_NAME));
                                }
                            }
                        }
                    }
                }

                orgTypeMapping.add(orgTypeMappingNode);
            }

            tProdEocMappingAryNode.add(tProdEocMappingObjectNode);
        }
        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);
        tResponseJsonNode.set("prod_eoc_mapping", tProdEocMappingAryNode);
        tResponseJsonNode.put(MdcConstant.TENANT_ID, pTenantId);
        return converJsonNodeToMap(tResponseJsonNode);
    }


    /**
     * 取得租戶特定產品的運營清單 v2 数据去重返回云端id
     */
    public Map<String, Object> processTenantProdOperationListGetV2(String pTenantId, String pProdName, String pUid,
                                                                 String pId) {

        //根據入參tenant_id, prod_name、云地映射标识取得eoc_intg_mapping資料
        Map<String, String> pCondition = new HashMap<>();
        pCondition.put(MdcConstant.TENANT_ID, pTenantId);
        pCondition.put(MdcConstant.PROD_NAME, pProdName);
        pCondition.put(MdcConstant.EOC_LEVEL, MdcConstant.CLOUD_MAPPING_ID);
        if (pUid != null) {
            pCondition.put(MdcConstant.UID, pUid);
        }
        List<EocIntgMapping> tMappings = eocIntgMappingDao.fetch(pCondition);

        //組成回傳json
        ArrayNode orgTypeMapping = mapper.createArrayNode();

        for (EocIntgMapping mapping : tMappings) {

            ObjectNode orgTypeMappingNode = mapper.createObjectNode();
            orgTypeMappingNode.put(MdcConstant.EOC_MAPPING_ID, mapping.getEocId());
            orgTypeMappingNode.put(MdcConstant.OM_MAPPING_ID, mapping.getIntgId());

            //type
            if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.enterprise_id.toString())) {
                orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.ENTERPRISE_ID);
            }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.company_id.toString())){
                orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.COMPANY_ID);
            }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.site_id.toString())){
                orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.SITE_ID);
            }else if (mapping.getCloudMappingType().equalsIgnoreCase(EocLevelEnum.region_id.toString())){
                orgTypeMappingNode.put(MdcConstant.MAPPING_TYPE,MdcConstant.REGION_ID);
            }

            orgTypeMapping.add(orgTypeMappingNode);
        }

        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = getDescriptionByLocale(tStateCode, locale);
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, null);

        tResponseJsonNode.set(MdcConstant.ORG_TYPE_MAPPING, orgTypeMapping);
        tResponseJsonNode.put(MdcConstant.TENANT_ID, pTenantId);
        tResponseJsonNode.put(MdcConstant.PROD_NAME, pProdName);
        // 穩態產品取eoc_intg_mapping表中product_uid
        if (!tMappings.isEmpty()) {
            tResponseJsonNode.put(MdcConstant.UID, tMappings.get(0).getProductUid());
            tResponseJsonNode.put(MdcConstant.PROD_TYPE, ApiTypeEnum.standard.toString());
            tResponseJsonNode.put(MdcConstant.URL, "");
        } else {
            // 敏態產品取product表中的產品資訊(新規則)
            Map<String, String> tAgileAppCondition = new HashMap<>();
            tAgileAppCondition.put(MdcConstant.PRODUCT_TYPE, ApiTypeEnum.agile.toString());
            tAgileAppCondition.put(MdcConstant.NAME, pProdName);
            if (pId != null) {
                tAgileAppCondition.put(MdcConstant.ID, pId);
            } else if (pUid != null) {
                tAgileAppCondition.put(MdcConstant.UID, pUid);
            }
            List<Product> tAgileAppList = productDao.fetch(tAgileAppCondition);
            // 有找到對應的敏態產品
            if (tAgileAppList != null && !tAgileAppList.isEmpty()) {
                tResponseJsonNode.put(MdcConstant.UID, tAgileAppList.get(0).getUid());
                tResponseJsonNode.put(MdcConstant.PROD_TYPE, ApiTypeEnum.agile.toString());
                tResponseJsonNode.put(MdcConstant.URL, tAgileAppList.get(0).getUrl());
            }
        }
        return converJsonNodeToMap(tResponseJsonNode);
    }
}