package com.digiwin.metadatacache.dao.impl;

import com.digiwin.metadatacache.dao.ApiRelationDao;
import com.digiwin.metadatacache.model.ApiRelation;
import org.apache.commons.collections.CollectionUtils;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Restrictions;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;


@Service
@Transactional("mdcTransactionManager")
public class ApiRelationDaoImpl extends GenericDaoImpl<ApiRelation, Long> implements ApiRelationDao {

	public ApiRelationDaoImpl() {
		super(ApiRelation.class);
	}

	/**
	 * 根据子类关系查询关联关系
	 * @param apiName api名称
	 * @param tenantId 租户Id
	 * @return ApiRelation
	 */
	@Override
	public List<ApiRelation> getRelationBySub(String apiName, String tenantId) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(ApiRelation.class);
		tDetachedCriteria.add(Restrictions.eq("subApiName", apiName));
		tDetachedCriteria.add(Restrictions.eq("deleteFlag", 0));
		Disjunction disjunction = Restrictions.disjunction();
		disjunction.add(Restrictions.eq("subTenantId", tenantId));
		disjunction.add(Restrictions.isNull("subTenantId"));
		tDetachedCriteria.add(disjunction);
		return (List<ApiRelation>)getHibernateTemplate().findByCriteria(tDetachedCriteria);
	}

	@Override
	public List<ApiRelation> getRelationBySub(List<String> apiNames, String tenantId) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(ApiRelation.class);
		tDetachedCriteria.add(Restrictions.in("subApiName", apiNames));
		tDetachedCriteria.add(Restrictions.eq("deleteFlag", 0));
		Disjunction disjunction = Restrictions.disjunction();
		disjunction.add(Restrictions.eq("subTenantId", tenantId));
		disjunction.add(Restrictions.isNull("subTenantId"));
		tDetachedCriteria.add(disjunction);
		return (List<ApiRelation>)getHibernateTemplate().findByCriteria(tDetachedCriteria);
	}

	/**
	 * 查询关联关系
	 * @param subApiName 子类名称
	 * @param subTenantId 子类租户
	 * @return List
	 */
	@Override
	public ApiRelation getRelationBySubInfo(String subApiName, String subTenantId) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(ApiRelation.class);
		tDetachedCriteria.add(Restrictions.eq("subApiName", subApiName));
		tDetachedCriteria.add(Restrictions.eq("deleteFlag", 0));
		if (null != subTenantId) {
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.eq("subTenantId", subTenantId));
			disjunction.add(Restrictions.isNull("subTenantId"));
			tDetachedCriteria.add(disjunction);
		} else {
			tDetachedCriteria.add(Restrictions.isNull("subTenantId"));
		}
		List<?> apiRelations = getHibernateTemplate().findByCriteria(tDetachedCriteria);
		if (CollectionUtils.isNotEmpty(apiRelations)) {
			 return (ApiRelation) apiRelations.get(0);
		}
		return null;
	}

	/**
	 * 父类API判定是否为其他关联关系的子类API
	 * @param apiName 父类名称
	 * @param tenantId 租户
	 * @return ApiRelation
	 */
	@Override
	public ApiRelation judgeIsSub(String apiName, String tenantId) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(ApiRelation.class);
		tDetachedCriteria.add(Restrictions.eq("subApiName", apiName));
		tDetachedCriteria.add(Restrictions.eq("deleteFlag", 0));
		if (null != tenantId) {
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.eq("subTenantId", tenantId));
			disjunction.add(Restrictions.isNull("subTenantId"));
			tDetachedCriteria.add(disjunction);
		} else {
			tDetachedCriteria.add(Restrictions.isNull("subTenantId"));
		}
		List<?> apiRelations = getHibernateTemplate().findByCriteria(tDetachedCriteria);
		if (CollectionUtils.isNotEmpty(apiRelations)) {
			return (ApiRelation) apiRelations.get(0);
		}
		return null;
	}

	/**
	 * 子类API判定是否为其他关联关系的父类API
	 * @param apiName 子类名称
	 * @param tenantId 租户
	 * @return ApiRelation
	 */
	@Override
	public ApiRelation judgeIsParent(String apiName, String tenantId) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(ApiRelation.class);
		tDetachedCriteria.add(Restrictions.eq("parentApiName", apiName));
		tDetachedCriteria.add(Restrictions.eq("deleteFlag", 0));
		if (null != tenantId) {
			tDetachedCriteria.add(Restrictions.eq("subTenantId", tenantId));
		}
		List<?> apiRelations = getHibernateTemplate().findByCriteria(tDetachedCriteria);
		if (CollectionUtils.isNotEmpty(apiRelations)) {
			return (ApiRelation) apiRelations.get(0);
		}
		return null;
	}

}
