package com.digiwin.metadatacache.dao.impl;

import java.util.List;
import java.util.Map;

import com.digiwin.metadatacache.constant.MdcSymbolConstant;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.criterion.CriteriaSpecification;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.orm.hibernate5.HibernateCallback;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.digiwin.metadatacache.dao.ProductDao;
import com.digiwin.metadatacache.enums.CacheMapTypeEnum;
import com.digiwin.metadatacache.model.Product;
import com.digiwin.metadatacache.services.MdcCacheService;

@Service
@Transactional("mdcTransactionManager")
public class ProductDaoImpl extends GenericDaoImpl<Product, Long> implements ProductDao {

	@Autowired
	private MdcCacheService cacheService;

	public ProductDaoImpl() {
		super(Product.class);
	}

	// 查詢資料
	@SuppressWarnings("unchecked")
	@Override
	public List<Product> fetch(Map<String, String> pConditions) {
		return getHibernateTemplate().execute(new HibernateCallback<List<Product>>() {
			@Override
			public List<Product> doInHibernate(Session pSession) throws HibernateException {
				// Hibernate - Criteria Query 標準查詢
				Criteria tCriteria = pSession.createCriteria(Product.class);
				tCriteria.setResultTransformer(CriteriaSpecification.DISTINCT_ROOT_ENTITY);
				// 設查詢條件
				// containsKey - boolean containsKey(Object key) 判斷map是否包含指定的key
				// MatchMode.ANYWHERE 等同於 like="%name%"
				if (pConditions.containsKey("tenant")) {
					tCriteria.add(Restrictions.eq("tenantId", pConditions.get("tenant")));
				}
				if (pConditions.containsKey("name")) {
					tCriteria.add(Restrictions.eq("name", pConditions.get("name")));
				}
				if (pConditions.containsKey("version")) {
					tCriteria.add(Restrictions.eq("version", pConditions.get("version")));
				}
				if (pConditions.containsKey("eai_uid")) {
					if (pConditions.get("eai_uid") != null) {
						tCriteria.add(Restrictions.eq("eaiUid", pConditions.get("eai_uid")));
					} else {
						tCriteria.add(Restrictions.isNull("eaiUid"));
					}
				}
				if (pConditions.containsKey("uid")) {
					tCriteria.add(Restrictions.eq("uid", pConditions.get("uid")));
				}
				if (pConditions.containsKey("id")) {
					tCriteria.add(Restrictions.eq("apid", pConditions.get("id")));
				}
				if (pConditions.containsKey("product_type")) {
					tCriteria.add(Restrictions.eq("productType", pConditions.get("product_type")));
				}
				return tCriteria.list();
			}
		});
	}

	/**
	 * MDC更新產品清單(觸發自地中台產品異動->同步到雲中台->雲中台同步租戶產品清單到MDC[雲中台只會同步該地端中台的所有產品清單(specific gatewayid and eai_uid)])
	 * @param tCurrentTenantIdList 租户IDs
	 * @param pList 产品信息s
	 */
	@Override
	public void updateList(final List<String> tCurrentTenantIdList, final List<Product> pList) {
		for (Product tProduct : pList) {
			  saveOrupdate(tProduct);
			  for (String tenantId : tCurrentTenantIdList) {
			      cacheService.removeCacheMapValueContainsWord(
			  		CacheMapTypeEnum.tenant_product_operation_cachemap.getCode() + MdcSymbolConstant.COLON + tenantId, tProduct.getName());
		     }
	    }
	}

	@Override
	public List<Product> queryByAppId(String appId) {
		DetachedCriteria tDetachedCriteria = DetachedCriteria.forClass(Product.class);
		tDetachedCriteria.add(Restrictions.eq("appId", appId));
		return (List<Product>)getHibernateTemplate().findByCriteria(tDetachedCriteria);
	}
}
