package com.digiwin.athena.schedulemanager.core.config;

import com.digiwin.app.autoconfigure.DWHttpClientAutoConfiguration;
import com.digiwin.loadbalance.util.HttpRouteUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.ServiceUnavailableRetryStrategy;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.routing.HttpRoutePlanner;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.retry.annotation.EnableRetry;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.web.client.RestTemplate;

/**
 * @author zhangww
 * @description: restTemplate配置类
 * @date 2021/4/12 14:02
 */
@EnableRetry
@Configuration("scheduleRestTemplateConfig")
@EnableAsync(proxyTargetClass = true)
public class RestTemplateConfig {

    @Value("${schedule.httpClientMaxTotal}")
    private int maxTotal;

    @Value("${schedule.httpClientMaxPerRoute}")
    private int maxPerRoute;

    @Value("${schedule.httpClientRequestTimeOutMilliSecond}")
    private int requestTimeOutMilliSecond;

    @Value("${schedule.httpClientConnectTimeOutMilliSecond}")
    private int connectTimeOutMilliSecond;

    @Value("${schedule.httpClientSocketTimeOutMilliSecond}")
    private int socketTimeOutMilliSecond;

    /**
     * 根据需要使用 HttpRouteUtils.DEFAULT_HTTPCLIENT_ROUTE_PLAN
     * 或者   HttpRouteUtils.ATTEMPT_HTTPCLIENT_ROUTE_PLAN （设置此值时 需设置 dwhttpClient.loadbalance.mode=auto    ）
     */
    @Autowired(required = false)
    @Qualifier(HttpRouteUtils.ATTEMPT_HTTPCLIENT_ROUTE_PLAN)
    HttpRoutePlanner httpRoutePlanner;

    @Bean("schedule-sm-resttemplate")
    public RestTemplate restTemplate(
            @Qualifier("schedule-sm-resttemplate-requestfactory")
                    ClientHttpRequestFactory httpRequestFactory) {
        RestTemplate restTemplate = new RestTemplate(httpRequestFactory);
        return restTemplate;
    }

    @Bean("schedule-httpClientConnectionManager")
    public PoolingHttpClientConnectionManager getConnectionManager() {
        return new PoolingHttpClientConnectionManager();
    }

    @Bean("schedule-sm-resttemplate-requestfactory")
    public ClientHttpRequestFactory httpRequestFactory(
            @Qualifier(DWHttpClientAutoConfiguration.BEAN_NAME_DW_HTTPCLIENT_REQUEST_RETRY_HANDLER)
                    HttpRequestRetryHandler requestRetryHandler,
            @Qualifier(DWHttpClientAutoConfiguration.BEAN_NAME_DW_HTTPCLIENT_RESPONSE_RETRY_STRATEGY)
                    ServiceUnavailableRetryStrategy serviceUnavailableRetryStrategy) {

        HttpClientBuilder builder = HttpClientBuilder.create()
                .setRetryHandler(requestRetryHandler)
                .setServiceUnavailableRetryStrategy(serviceUnavailableRetryStrategy);

        PoolingHttpClientConnectionManager connectionManager = getConnectionManager();
        /** 设置整个连接池最大连接数 */
        connectionManager.setMaxTotal(maxTotal);

        /** 路由是对maxTotal的细分 */
        connectionManager.setDefaultMaxPerRoute(maxPerRoute);
        RequestConfig requestConfig = RequestConfig.custom()
                /** 返回数据的超时时间 */
                .setSocketTimeout(socketTimeOutMilliSecond)
                /** 连接上服务器的超时时间 */
                .setConnectTimeout(connectTimeOutMilliSecond)
                /** 从连接池中获取连接的超时时间 */
                .setConnectionRequestTimeout(requestTimeOutMilliSecond)
                .build();
        builder.setDefaultRequestConfig(requestConfig).setConnectionManager(connectionManager)
                .setRoutePlanner(httpRoutePlanner).build();

        HttpClient httpClient = builder.build();
        HttpComponentsClientHttpRequestFactory clientHttpRequestFactory = new HttpComponentsClientHttpRequestFactory(httpClient);

        return clientHttpRequestFactory;
    }
}

