package com.digiwin.athena.schedulemanager.service.facade.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWRuntimeException;
import com.digiwin.athena.schedulemanager.core.constant.FieldNameConstant;
import com.digiwin.athena.schedulemanager.core.constant.ScheduleConstant;
import com.digiwin.athena.schedulemanager.core.dict.TypeDict;
import com.digiwin.athena.schedulemanager.core.message.ErrorCode;
import com.digiwin.athena.schedulemanager.core.util.CollectionUtil;
import com.digiwin.athena.schedulemanager.core.util.StringUtil;
import com.digiwin.athena.schedulemanager.dto.ScheduleCompareDto;
import com.digiwin.athena.schedulemanager.pojo.request.*;
import com.digiwin.athena.schedulemanager.repository.model.ScheduleDetailModel;
import com.digiwin.athena.schedulemanager.service.client.IDwScheduleClient;
import com.digiwin.athena.schedulemanager.service.facade.IScheduleFacadeService;
import com.digiwin.athena.schedulemanager.service.srp.IScheduleDefineService;
import com.digiwin.athena.schedulemanager.service.srp.IScheduleLogService;
import com.digiwin.athena.schedulemanager.util.JsonUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author zhangww
 * @description: 排程引擎业务处理核心类
 * @date 2022/4/25 17:44
 */
@Service("scheduleFacadeService")
public class ScheduleFacadeService implements IScheduleFacadeService {

    private static final Logger LOGGER = LoggerFactory.getLogger(ScheduleFacadeService.class);

    private IDwScheduleClient dapClient;

    private IScheduleLogService logService;

    private IScheduleDefineService defineService;

    public ScheduleFacadeService(IDwScheduleClient dapClient, IScheduleLogService logService, IScheduleDefineService defineService) {
        this.dapClient = dapClient;
        this.logService = logService;
        this.defineService = defineService;
    }

    @Override
    @Transactional(value = "schedule-dw-transactionManager")
    public void createSchedule(BatchScheduleReq rule) {
        rule.getRuleList().forEach(request -> createQuartzSchedule(rule.getTenantId(), request.getRuleId(), request.getActionId(), StringUtil.isEmpty(request.getEnableStatus()) ? ScheduleConstant.ENABLE : request.getEnableStatus(), request.getTriggers(), request.getCaller(), request.getScheduleParam()));
    }

    private void createQuartzSchedule(String tenantId, String ruleId, String actionId, String enableStatus, List<TriggerReq> triggerList, String caller, JSONObject param) {
        List<Map<String, Object>> scheduleParamList = new ArrayList();

        triggerList.forEach(trigger -> {
            String instanceId = UUID.randomUUID().toString();
            //排程参数组装
            Map<String, Object> scheduleParam = initScheduleParam(instanceId, tenantId, ruleId, enableStatus, caller, param);
            //根据规则获取排程类型
            Map<String, Object> scheduleDetail = getScheduleDetail(trigger);
            scheduleParam.putAll(scheduleDetail);
            scheduleParamList.add(scheduleParam);

            saveScheduleDefine(tenantId, ruleId, actionId, instanceId, JsonUtil.getJsonString(trigger), StringUtil.isEmpty(caller) ? ScheduleConstant.MONITORENGINE : caller, ScheduleConstant.VALID);
        });

        String result = dapClient.addQuartz(scheduleParamList, ScheduleConstant.SCHEDULE_ADD);
        JSONObject resultObj = JsonUtil.getObject(result);
        /*FIXME 后续dap的接口如果调用失败直接返回异常后，此处的逻辑判断需要调整 无需判断返回状态 */
        boolean isSuccess = ScheduleConstant.HTTP_RESPONSE_SUCCESS_STATUS == resultObj.getIntValue(ScheduleConstant.STATUS);
        //记录操作日志
        scheduleParamList.forEach(scheduleParam -> logService.saveScheduleLog(String.valueOf(scheduleParam.get(ScheduleConstant.SCHEDULE_ID)), ScheduleConstant.CREATE, ScheduleConstant.MONITORENGINE, isSuccess ? 1 : 0));

        if (!isSuccess) {
            LOGGER.error("调用api创建排程失败！{}", result);
            throw new DWRuntimeException(ErrorCode.ERROR_SCHEDULE_CREATE_CODE.getCode(), ErrorCode.ERROR_SCHEDULE_CREATE_CODE.getMessage());
        }
    }

    /**
     * 组装远程调用服务所需参数
     * @param scheduleId
     * @param tenantId
     * @param ruleId
     * @param enableStatus
     * @return
     */
    private Map<String, Object> initScheduleParam(String scheduleId, String tenantId, String ruleId, String enableStatus, String caller, JSONObject param) {
        Map<String, Object> scheduleParam = new HashMap<>(ScheduleConstant.CAPACITY_4);
        scheduleParam.put(ScheduleConstant.TENANTID, tenantId);
        scheduleParam.put(ScheduleConstant.RULEID, ruleId);
        scheduleParam.put(ScheduleConstant.INSTANCE_ID, scheduleId);
        scheduleParam.put(ScheduleConstant.CALLER, caller);
        //排程接收的自定义参数
        scheduleParam.put(ScheduleConstant.SCHEDULE_CUSTOM_PARAM, param);

        Map<String, Object> paramMap = new HashMap<>(ScheduleConstant.CAPACITY_8);
        paramMap.put(ScheduleConstant.SCHEDULE_PARAM, scheduleParam);
        paramMap.put(ScheduleConstant.SCHEDULE_ID, scheduleId);
        paramMap.put(ScheduleConstant.JOB_NAME, ScheduleConstant.ISCHEDULEJOBSERVICE);
        paramMap.put(ScheduleConstant.ENABLE_STATUS, enableStatus);
        paramMap.put(ScheduleConstant.MODULE_NAME, ScheduleConstant.SCHEDULEMANAGER);
        return paramMap;
    }

    /**
     * 修改排程
     * @param rule
     * @param enableStatus
     * @return
     * @throws Exception
     */
    @Override
    @Transactional(value = "schedule-dw-transactionManager")
    public void updateSchedule(ScheduleUpdateReq rule, String enableStatus) {
        List<ScheduleDetailModel> updateList = defineService.getScheduleInfoByRuleId(rule.getRuleId());
        updateTriggers(rule, enableStatus, updateList);
    }

    private void updateTriggers(ScheduleUpdateReq rule, String enableStatus, List<ScheduleDetailModel> sourceSchedule) {
        //数据准备
        ScheduleCompareDto scheduleCompareDto = handleScheduleDifference(rule, sourceSchedule);

        //删除
        if (CollectionUtil.isNotEmpty(scheduleCompareDto.getSourceScheduleMap())) {
            executeSchedule(scheduleCompareDto.getSourceScheduleMap().values(), ScheduleConstant.DELETE);
        }

        //新增
        if (CollectionUtil.isNotEmpty(scheduleCompareDto.getAddSchedules())) {
            if (StringUtil.isBlank(enableStatus)) {
                enableStatus = ScheduleConstant.ENABLE;
            }
            createQuartzSchedule(rule.getTenantId(), rule.getRuleId(), rule.getActionId(), enableStatus, scheduleCompareDto.getAddSchedules(), rule.getCaller(), rule.getScheduleParam());
        }

        //修改
        if (CollectionUtil.isNotEmpty(scheduleCompareDto.getExistsSchedules())) {
            int callType = ScheduleConstant.DISABLE.equals(enableStatus) ? ScheduleConstant.STOP : ScheduleConstant.START;
            executeSchedule(scheduleCompareDto.getExistsSchedules(), callType);
        }
    }

    /**
     * 处理排程更新时的数据差异，分别计算出需要删除，新增和更新的数据
     * @param sourceSchedule
     * @return
     */
    private ScheduleCompareDto handleScheduleDifference(ScheduleUpdateReq requestDetail,
                                                        List<ScheduleDetailModel> sourceSchedule) {
        /** 数据差异整理开始 */
        Map<JSONObject, ScheduleDetailModel> sourceScheduleMap = new HashMap<>(ScheduleConstant.CAPACITY_4);
        //将需要更新的排程规则转为map
        sourceSchedule.forEach(schedule -> sourceScheduleMap.put(JsonUtil.getObject(schedule.getTriggers()), schedule));
        //对比后需要新增的排程规则
        List<TriggerReq> addSchedules = new ArrayList<>();
        //对比后需要更新的排程规则
        List<ScheduleDetailModel> existsSchedules = new ArrayList<>();
        //原排程规则和需要更新的排程规则作对比
        requestDetail.getTriggers().forEach(trigger -> {
            JSONObject key = JsonUtil.getObject(JsonUtil.getJsonString(trigger));
            ScheduleDetailModel existsScheduleDetail = sourceScheduleMap.remove(key);
            //需要更新的排程规则如果在原排程规则里则做更新，否则做添加，剩下还存在在原排程规则里的排程则作删除
            if (null != existsScheduleDetail) {
                existsSchedules.add(existsScheduleDetail);
            } else {
                addSchedules.add(trigger);
            }
        });
        /** 数据差异整理结束 */
        LOGGER.info("ScheduleFacadeService#handleScheduleDifference sourceScheduleMap  :{},addSchedules  :{}, existsSchedules  :{} ",
                sourceScheduleMap.size(), addSchedules.size(), existsSchedules.size());
        ScheduleCompareDto scheduleCompareDto = new ScheduleCompareDto();
        scheduleCompareDto.setSourceScheduleMap(sourceScheduleMap);
        scheduleCompareDto.setAddSchedules(addSchedules);
        scheduleCompareDto.setExistsSchedules(existsSchedules);
        return scheduleCompareDto;
    }

    @Override
    @Transactional(value = "schedule-dw-transactionManager")
    public void deleteSchedule(ScheduleReq rule) {
        handleSchedule(rule, ScheduleConstant.DELETE);
    }

    @Override
    public void stopSchedule(ScheduleReq rule) {
        handleSchedule(rule, ScheduleConstant.STOP);
    }

    @Override
    public void startSchedule(ScheduleReq rule) {
        handleSchedule(rule, ScheduleConstant.START);
    }

    /**
     * 跨租户删除排程
     *
     * @param cross@return
     */
    @Override
    public void deleteScheduleCross(CrossReq cross) {
        dapClient.deleteQuartzCross(cross);
    }

    @Override
    public JSONObject getSchedule(ScheduleReq rule) {
        List<ScheduleDetailModel> list = defineService.getScheduleInfoByRuleId(rule.getRuleId());
        if (CollectionUtil.isEmpty(list)) {
            return new JSONObject();
        }

        return getDapScheduleList(list);
    }

    private JSONObject getDapScheduleList(List<ScheduleDetailModel> list) {
        JSONObject result = new JSONObject();
        List<String> scheduleIdList = list.stream().map(ScheduleDetailModel::getInstanceId).collect(Collectors.toList());

        // 调用api查询排程
        String response = dapClient.getQuartz(scheduleIdList);
        JSONObject resultObj = JsonUtil.getObject(response);
        JSONArray schedules = new JSONArray();
        if (ScheduleConstant.HTTP_RESPONSE_SUCCESS_STATUS == resultObj.getIntValue(FieldNameConstant.DW_SERVICE_STATUS)) {
            schedules = resultObj.getJSONArray(FieldNameConstant.DW_SERVICE_RESPONSE);
            result.put(ScheduleConstant.DATA_LIST, schedules);
        }
        result.put(ScheduleConstant.ENABLE_STATUS, schedules.isEmpty() ? "" : schedules.getJSONObject(ScheduleConstant.START_POS).getJSONObject(ScheduleConstant.DATA).getString(ScheduleConstant.ENABLE_STATUS));
        return result;
    }

    private void handleSchedule(ScheduleReq rule, int callType) {
        List<ScheduleDetailModel> list = defineService.getScheduleInfoByRuleId(rule.getRuleId());
        if (CollectionUtil.isEmpty(list)) {
            LOGGER.warn("通过条件{}查询，暂无匹配排程记录", rule);
            return;
        }
        boolean result = executeSchedule(list, callType);
        if (!result) {
            throw new DWRuntimeException(ErrorCode.ERROR_SCHEDULE_HANDLE_CODE.getCode(), ErrorCode.ERROR_SCHEDULE_HANDLE_CODE.getMessage());
        }
    }

    /**
     * 根据操作类型处理排程数据
     * @param list
     * @param callType
     * @return
     */
    private boolean executeSchedule(Collection<ScheduleDetailModel> list, int callType) {
        List<String> scheduleIdList = list.stream().map(ScheduleDetailModel::getInstanceId).collect(Collectors.toList());
        String result = dapClient.executeQuartz(scheduleIdList, TypeDict.getMethodName(callType));
        JSONObject resultObj = JsonUtil.getObject(result);

        boolean isSuccess = ScheduleConstant.HTTP_RESPONSE_SUCCESS_STATUS == resultObj.getIntValue(ScheduleConstant.STATUS);

        if (ScheduleConstant.DELETE == callType && isSuccess) {
            scheduleIdList.forEach(instanceId -> defineService.deleteScheduleDefine(instanceId));
        }

        int flag = isSuccess ? ScheduleConstant.SUCCESS : ScheduleConstant.FAIL;
        scheduleIdList.forEach(instanceId -> logService.saveScheduleLog(instanceId, callType, ScheduleConstant.MONITORENGINE, flag));
        return isSuccess;
    }

    /**
     * 根据规则获取排程类型
     * @param trigger
     * @return
     */
    private Map<String, Object> getScheduleDetail(TriggerReq trigger) {
        Map<String, Object> paramMap = new HashMap<>(ScheduleConstant.CAPACITY_16);
        if (StringUtil.isNotBlank(trigger.getAssignTime())) {
            paramMap.put(ScheduleConstant.SCHEDULE_TYPE, ScheduleConstant.ASSIGN_TIME);
            paramMap.put(ScheduleConstant.ASSIGN_TIME_FIELD, trigger.getAssignTime());
        } else if (StringUtil.isNotBlank(trigger.getMonth()) || StringUtil.isNotBlank(trigger.getWeekOfMonth()) || StringUtil.isNotBlank(trigger.getDayOfWeek()) || StringUtil.isNotBlank(trigger.getDayOfMonth()) || StringUtil.isNotBlank(trigger.getTime()) || StringUtil.isNotBlank(trigger.getTime2()) || StringUtil.isNotBlank(trigger.getTime3()) || StringUtil.isNotBlank(trigger.getTime1())) {
            paramMap.put(ScheduleConstant.SCHEDULE_TYPE, ScheduleConstant.PERIOD);
            paramMap.put(ScheduleConstant.MONTH, trigger.getMonth());
            paramMap.put(ScheduleConstant.WEEK_OF_MONTH, trigger.getMonth());
            paramMap.put(ScheduleConstant.DAY_OF_WEEK, trigger.getDayOfWeek());
            paramMap.put(ScheduleConstant.DAY_OF_MONTH, trigger.getDayOfMonth());
            paramMap.put(ScheduleConstant.TIME, trigger.getTime());
            paramMap.put(ScheduleConstant.TIME1, trigger.getTime1());
            paramMap.put(ScheduleConstant.TIME2, trigger.getTime2());
            paramMap.put(ScheduleConstant.TIME3, trigger.getTime3());
        } else if (StringUtil.isNotBlank(trigger.getRepeatType())) {
            paramMap.put(ScheduleConstant.SCHEDULE_TYPE, ScheduleConstant.COMPLEX_PERIOD);
            paramMap.put(ScheduleConstant.START_TIME, trigger.getStartTime());
            paramMap.put(ScheduleConstant.END_TIME, trigger.getEndTime());
            paramMap.put(ScheduleConstant.REPEAT_TYPE, trigger.getRepeatType());
            paramMap.put(ScheduleConstant.REPEAT_COUNT, trigger.getRepeatCount());
            paramMap.put(ScheduleConstant.FREQUENCY, trigger.getFrequency());
        } else {
            paramMap.put(ScheduleConstant.SCHEDULE_TYPE, ScheduleConstant.REPEAT);
            paramMap.put(ScheduleConstant.START_TIME, trigger.getStartTime());
            paramMap.put(ScheduleConstant.END_TIME, trigger.getEndTime());
            paramMap.put(ScheduleConstant.REPEAT_COUNT, trigger.getRepeatCount());
            paramMap.put(ScheduleConstant.MINUTELY, trigger.getMinutely());
            paramMap.put(ScheduleConstant.HOURLY, trigger.getHourly());
            paramMap.put(ScheduleConstant.DAILY, trigger.getDaily());
            paramMap.put(ScheduleConstant.WEEKLY, trigger.getWeekly());
            paramMap.put(ScheduleConstant.BYDAY, trigger.getByday());
            paramMap.put(ScheduleConstant.MONTHLY, trigger.getMonthly());
            paramMap.put(ScheduleConstant.BYMONTHDAY, trigger.getBymonthday());
            paramMap.put(ScheduleConstant.BYWEEKLYDAY, trigger.getByweeklyday());
            paramMap.put(ScheduleConstant.R_RULE, trigger.getRRule());
        }
        return paramMap;
    }

    /**
     * 保存排程和规则定义关系
     * @param tenantId
     * @param ruleId
     * @param actionId
     * @param scheduleId
     * @param trigger
     * @param caller
     * @param valid
     */
    private void saveScheduleDefine(String tenantId, String ruleId, String actionId, String scheduleId, String trigger, String caller, int valid) {
        ScheduleDetailModel model = new ScheduleDetailModel(scheduleId, ruleId, tenantId, actionId, trigger, caller, valid);
        defineService.saveScheduleDefine(model);
    }
}
